// =====================================================================================================================================================================================================
// (c) 2021 Lynn Hansen, KU7Q															                                                                                                               |
// This Source Code Form is subject to the terms of the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007. A copy of this license can be found here: https://choosealicense.com/licenses/gpl-3.0/|
// =====================================================================================================================================================================================================


// THIS MODULE INCLUDES THE DECODERS THAT DECODE MODES and THE TxInfo BLOCKS SENT FROM EACH ACTIVE PAGE


//===================================================================================================================
// Mode Decoders - return with text for mode
//===================================================================================================================

String Dec_gRMode()
{
	//returns with radio mode for gRMode
	switch (gRMode)
	{
	case RADIO_CW:
		return "CW";
		break;
	case RADIO_CW_R:
		return "CWr";
		break;
	case RADIO_LSB:
		return "LSB";
		break;
	case RADIO_USB:
		return "USB";
		break;
	case RADIO_FM_N:
		return "FMn";
		break;
	case RADIO_FM_W:
		return "FMw";
		break;
	case RADIO_AM:
		return "AM";
		break;
	case RADIO_DIG_L:
		return "DigL";
		break;
	case RADIO_DIG_H:
		return "DigH";
		break;
	}
	return " "; //must return with something or audio locks up
}

uint8_t Dec_RMode_Txt(String txt)
{	
	//return with value of radio mode from Edit page
	if (txt == "CWr")
	{
		return RADIO_CW_R;
	}
	else if (txt == "LSB")
	{
		return RADIO_LSB;
	}
	else if (txt == "USB")
	{
		return RADIO_USB;
	}
	else if (txt == "FMn")
	{
		return RADIO_FM_N;
	}
	else if (txt == "FMw")
	{
		return RADIO_FM_W;
	}
	else if (txt == "AM")
	{
		return RADIO_AM;
	}
	else if (txt == "DgL")
	{
		return RADIO_DIG_L;
	}
	else if (txt == "DgH")
	{
		return RADIO_DIG_H;
	}
	return RADIO_CW; //default to cw
}

String Dec_gOMode()
{
	//return with text for gOMode
	switch (gOMode)
	{
	case MODE_CW:
		return "CW";
		break;
	case MODE_VOICE:
		return "Voice";
		break;
	case MODE_RTTY:
		return "RTTY";
		break;
	case MODE_DIG:
		return "Digital";
		break;
	case MODE_BEACON:
		return "BEACON";
		break;
	}
	return " "; //must return with something or audio locks up
}

uint8_t Dec_OMode_Txt(String txt)
{
	//return with mode value from edit page
	if (txt == "Voc")
	{
		return MODE_VOICE;
	}
	else if (txt == "RTY")
	{
		return MODE_RTTY;
	}
	else if (txt == "Dig")
	{
		return MODE_DIG;
	}
	else if (txt == "Bcn")
	{
		return MODE_BEACON;
	}
	return MODE_CW;
}

void Dec_TxInfo()
{
	/*
	  decode the txInfo data stream in the hmiRx[] array
	  [1] holds page #, [2] holds bit counter
	  [3] is the first byte of data

	  RULES FOR UPDATING HMI OR PROC VALUES
		1. If HMI value == current value, set previous value = to HMI value
		   a. This will reset previous value if it's causing proc to send updates to HMI
		2. If HMI value == previous value but not current value, ignore HMI value
		   a. We're waiting for proc to get update to HMI
		3. If current value == previous value but HMI value is different, set current and previous values to HMI value and update the radio values if needed
		   a. Change came from HMI while proc values were stable, update proc & radio state

	*/

	uint8_t ptr = 3; //data starts on bit 3 (hmiRx[] is one's based)	

	//change these values if the txinfo packet size changes on the Home Page
	uint8_t HomePkt1 = 0x07; //display transmits this packet while waiting for initialization sequence from processor
	uint8_t HomePkt2 = 0x32; //basic home page info
	uint8_t HomePkt3 = 0x39; //info from cw, voice, rtty, dig, ant, and levels pages

	if (!hmiRadioLoaded)
	{
		//gSize and gRadioSel are sent from S page before hmi is initialized
		if (hmiRx[1] == 0 && hmiRx[2] == 8)
		{
			Dec_S_Data(ptr); //display size		
			SerialOut("Display size = " + String(float(gSize) / 10), true);
		}
		else
		{
			//SerialOut("Waiting for gSize and gRadioSel message", true);
		}
		return; //wait for first radio file to load
	}

	//always update hmiPage, even if we're blocked - not doing this causes delays when moving between pages on display
	hmiPage = hmiRx[1];
	if (hmiPage != hmiPagePrev)
	{
		Chk4PageChange();
		blockHMI = 0;
	}

	//block changes if loading new radio file
	if (millis() < blockHMI) //if blocking or we've been alive less than 20 seconds, ignore status updates
	{
		//SerialOut("***************", true);
		//SerialOut("*************** >>> blockHMI for " + String(blockHMI - millis()) + " mSec more", true);
		//SerialOut("***************", true);
		return;
	}

	//format here must follow the data formatting in the Tx{Page Name} code in each page
	//switch on page# - 1st byte after preamble
	
	switch (hmiPage)
	{
	case HMI_HOME:
		//Home sends three separate messages
		//Message 3 sends a combination of data from several pages - the format in the HMI must match the data on those pages exactly
		//	
		//SerialOut("HomePkt= " + String(hmiRx[2], HEX), true);
		if (hmiRx[2] == HomePkt1)
		{
			//gRadioSel changed or radio not loaded yet, save current port and load the new one
			gRadioSel = CalcInfoMsgVal(ptr, 1);
			if (gRadioSel != gRadioSelPrev)
			{
				//new selection save existing
				if (hmiRadioLoaded)
				{
					SerialOut("Saving old radio " + String(gRadioSelPrev + 1), true);
					SaveRadio(gRadioSelPrev + 1);
					SaveBandList(gRadioSelPrev + 1);
				}
				SerialOut("Loading new radio " + String(gRadioSel + 1), true);
				LoadRadio();
				return;
			}
		}
		else if (hmiRx[2] == HomePkt2)
		{
			//split Home txInfo into two blocks - had intermittant rx failures with longer blocks
			//decode the first block of 48 bytes
			Dec_Home_Data(ptr);
		}
		else if (hmiRx[2] == HomePkt3)
		{
			//decode the second block of 54 bytes			
			Dec_CW_Data(ptr);
			Dec_Voice_Data(ptr);
			Dec_RTTY_Data(ptr);
			Dec_Dig_Data(ptr);
			Dec_Ant_Data(ptr);
			Dec_Levels_Data(ptr);
		}
		break;
	case HMI_BAND:
		Dec_Band_Data(ptr);
		break;
	case HMI_MODE:
		Dec_Mode_Data(ptr);
		break;
	case HMI_CW:
		Dec_CW_Data(ptr);
		break;
	case HMI_VOICE:
		Dec_Voice_Data(ptr);
		break;
	case HMI_RTTY:
		Dec_RTTY_Data(ptr);
		break;
	case HMI_DIG:
		Dec_Dig_Data(ptr);
		break;
	case HMI_TXMEM:
		Dec_TxMem_Data(ptr);
		break;
	case HMI_FREQMEM:
		Dec_FreqMem_Data(ptr);
		break;
	case HMI_LOG:
		Dec_Log_Data(ptr);
		break;
	case HMI_HELP:
		Dec_Help_Data(ptr);
		break;
	case HMI_CONFIG:
		Dec_Config_Data(ptr);
		break;
	case HMI_KEYPAD:
		Dec_KeyPad_Data(ptr);
		break;
	case HMI_RXMSG:
		Dec_RxMsg_Data(ptr);
		break;
	case HMI_INPUT:
		Dec_Input_Data(ptr);
		break;
	case HMI_EDIT:
		Dec_Edit_Data(ptr);
		break;
	case HMI_SAVELOG:
		Dec_SaveLog_Data(ptr);
		break;
	case HMI_AUTOLOG:
		Dec_AutoLog_Data(ptr);
		break;
	case HMI_FLEX:
		Dec_Flex_Data(ptr);
		break;
	case HMI_RADIO:
		Dec_Radio_Data(ptr);
		break;
	case HMI_PCR:
		Dec_PCR_Data(ptr);
		break;
	case HMI_VFO:
		Dec_VFO_Data(ptr);
		break;
	case HMI_AUDIO:
		Dec_Audio_Data(ptr);
		break;
	case HMI_ANT:
		Dec_Ant_Data(ptr);
		break;
	case HMI_LEVELS:
		Dec_Levels_Data(ptr);
		break;
	}

	//we always end with these
	gEncoder = CalcInfoMsgVal(ptr, 1);
	//SerialOut("ptr= " + String(ptr) + ", gEncoder = " + String(gEncoder), true);

	ptr++;
	gAction = CalcInfoMsgVal(ptr, 1);
	//SerialOut("ptr= " + String(ptr) + ", gAction = " + String(gAction), true);

	ptr++;
	gSvRadio = CalcInfoMsgVal(ptr, 1);


	if (gEncoder != gEncoderPrev)
	{
		hmiEncoderTimeout = millis(); //restart timeout
		gEncoderPrev = gEncoder;
	}
}

//==============================================================================================
//The routines below decode the information for each page in the txInfo packet
// * There is one routine for each page - even if no decoding is required now
//==============================================================================================

void Dec_S_Data(uint8_t & ptr)
{
	//decode display size sent from S page on boot
	gSize = CalcInfoMsgVal(ptr, 1);

	ptr++; //inc this because we didn't call a subroutine that did it for us
	gRadioSel = CalcInfoMsgVal(ptr, 1); //so we know what radio port to load at startup

	ptr++;
}

void Dec_Home_Data(uint8_t & ptr)
{
	uint32_t temp32 = 0; //to chk for changes
	uint8_t temp8 = 0;

	//Deocde data for Home Page =================================================================
	temp32 = CalcInfoMsgVal(ptr, 4);
	if (temp32 == gUnit)
	{
		//if hmi and proc match, update previous in case we're trying to updater HMI
		gUnitPrev = temp32;
		//SerialOut("gUnitPrev=" + String(gUnit), true);
	}
	else if (gUnit == gUnitPrev) // && temp32 != gUnit)
	{
		//update came from hmi, update proc
		gUnit = temp32;
		gUnitPrev = gUnit;
		SerialOut("gUnit=" + String(gUnit), true);
	}

	ptr += 4;
	temp32 = CalcInfoMsgVal(ptr, 4);
	if (temp32 >= 100000)
	{
		if (temp32 == gFreqRadio)
		{
			gFreqRadioPrev = temp32;
		}
		else if (gFreqRadio == gFreqRadioPrev)
		{
			gFreqRadio = temp32;
		}
	}

	ptr += 4;
	gVFOA = CalcInfoMsgVal(ptr, 4); //not using previous value on gVFOA

	ptr += 4;
	gVFOB = CalcInfoMsgVal(ptr, 4); //not using previous value on gVFOB

	ptr += 4;
	temp8 = CalcInfoMsgVal(ptr, 1);
	//verify gOMode is within bounds
	if (temp8 <= 4 || temp8 == 10)
	{
		if (temp8 == gOMode)
		{
			gOModePrev = temp8;
		}
		else if (gOMode == gOModePrev)
		{
			gOMode = temp8;
		}
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	//verify gRMode is within bounds
	if (temp8 < 9)
	{
		if (temp8 == gRMode)
		{
			gRModePrev = temp8;
		}
		else if (gRMode == gRModePrev)
		{
			gRMode = temp8;
		}
	}

	ptr++;
	gRadio = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gRadioSel = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);	
	if (temp8 == (gFFTFreq / 25))
	{
		gFFTFreqPrev = temp8 * 25;
	}
	else if (gFFTFreq == gFFTFreqPrev && temp8 >= 8)
	{
		gFFTFreq = temp8 * 25;
		hmiEncoderTimeout = millis(); //restart encoder timeout		
	}	

	ptr++;
	gMSel = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	//will have a value if we're scanning memory - override gMSel
	if (temp8 > 0 && temp8 <= 100)
	{
		gMSel = temp8;
	}

	ptr++;
	gFFTAvg = (CalcInfoMsgVal(ptr, 1)) & 0x01; //prev not used here

	ptr++;
	gWiFi = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gShareDB = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gMon = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gVocal = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gLock = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gSplit = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTxLev)
	{
		gTxLevPrev = temp8;
	}
	else if (gTxLev == gTxLevPrev)
	{
		gTxLev = temp8;
	}
	
	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gLPFltr)
	{
		gLPFltrPrev = temp8;
	}
	else if (gLPFltr == gLPFltrPrev)
	{
		gLPFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gNFltr)
	{
		gNFltrPrev = temp8;
	}
	else if (gNFltr == gNFltrPrev)
	{
		gNFltr = temp8;
	}

	ptr++;
	gCtstCtr = CalcInfoMsgVal(ptr, 2);

	ptr += 2; //two bytes for gCtrsCtr above
	gIcmAdr = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gTxModeSet2 = CalcInfoMsgVal(ptr, 1); //gTxModeSet2 will cause gTxMode to change		

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gFFTGain)
	{
		gFFTGainPrev = temp8;
	}
	else if (gFFTGain == gFFTGainPrev)
	{
		gFFTGain = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gVolGain)
	{
		gVolGainPrev = temp8;
	}
	else if (gVolGain == gVolGainPrev)
	{
		gVolGain = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gSqGain)
	{
		gSqGainPrev = temp8;
	}
	else if (gSqGain == gSqGainPrev)
	{
		gSqGain = temp8;
	}

	ptr++;
	gAudioVMsg = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gAudioRecRx = CalcInfoMsgVal(ptr, 1);	

	ptr++;
	hmiRTCMin = CalcInfoMsgVal(ptr, 1);

	ptr++;
	hmiRTCSec = CalcInfoMsgVal(ptr, 1);

	ptr++;


	//process changes *****************************************************************************
	
	if (gFreqRadio != gFreqRadioPrev)
	{
		ChkFreqChange();
	}

	if (gOMode != gOModePrev)
	{
		SetOMode();
	}

	if (gRMode != gRModePrev)
	{
		SetRMode();
	}

	if (gFFTFreq > 2975)
	{
		gFFTFreq = 2975;
	}
	else if (gFFTFreq < 75)
	{
		gFFTFreq = 75; //limit to 100 hz - below that the audio volume drops too low
	}
	if (gFFTFreq != gFFTFreqPrev)
	{		
		SetFilterFreq(18);
		gFFTFreqPrev = gFFTFreq;
		Sidetone.frequency(gFFTFreq);
		Sidetone.update();
		hmiFFTMarkerUpdate = 0; //immediate update
	}

	if (gMon != gMonPrev)
	{
		SetOutputSource(0xff, true); //change output if this changes		
		gMonPrev = gMon;
	}

	if (gTxLev != gTxLevPrev)
	{
		//set common values - tm0 on Home moves these to individual pages			
		bandMemTxLev[gBand] = gTxLev;				
		SetOutputSource(0xff, true); //change level here					
	}

	if (gLPFltr != gLPFltrPrev)
	{
		//update LP filter			
		SetFilterFreq(19);
		bandMemLPFltr[gBand] = gLPFltr;
	}

	if (gNFltr != gNFltrPrev)
	{
		SetFilterFreq(20);
		bandMemNFltr[gBand] = gNFltr;
	}

	if (gFFTGain > 100)
	{
		//bad value, revert to previous
		gFFTGain = gFFTGainPrev;
	}
	if (gFFTGain != gFFTGainPrev)
	{
		//if - set to 0. - gain just inverts phase
		float g = gFFTGain + (gFFTLv - 50.0);
		if (g < 0)
		{
			g = LEV_OFF;
		}
		Amp_FFT.gain(g);
		Amp_FFT.update();
		gFFTGainPrev = gFFTGain;
	}

	//SerialOut("gVolGain= " + String(gVolGain), true);
	ChkVolGain();

	if (gSqGain > 100)
	{
		//revert
		gSqGain = gSqGainPrev;
	}
}

void Dec_Band_Data(uint8_t & ptr)
{
	uint8_t temp8 = CalcInfoMsgVal(ptr, 1);
	//make sure gBand is in range
	if (temp8 < 16)
	{
		if (temp8 == gBand)
		{
			gBandPrev = temp8;
		}
		else if (gBand == gBandPrev)
		{
			gBand = temp8;
			SerialOut(">>>>> New band= " + String(gBand), true);
		}
		Chk4BandChange();
	}

	ptr++;
}

void Dec_Mode_Data(uint8_t & ptr)
{
	uint8_t temp8 = CalcInfoMsgVal(ptr, 1);
	uint8_t prev = gRMode;
	if (temp8 < 9)
	{
		gRMode = temp8;
		if (gRMode != prev)
		{
			SetRMode();
		}
	}

	ptr++;
	prev = gOMode;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 < 5 || temp8 == 10)
	{
		gOMode = temp8;
		if (gOMode != prev)
		{
			SetOMode();
		}
	}

	ptr++;

}

void Dec_CW_Data(uint8_t & ptr)
{
	//Decode CW Page data =================================================================
	uint8_t temp8 = 0;

	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cLPFltr)
	{
		cLPFltrPrev = temp8;
	}
	else if (cLPFltr == cLPFltrPrev)
	{
		cLPFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cNFltr)
	{
		cNFltrPrev = temp8;
	}
	else if (cNFltr == cNFltrPrev)
	{
		cNFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cTxSpeed)
	{
		cTxSpeedPrev = temp8;
	}
	else if (cTxSpeed == cTxSpeedPrev)
	{
		cTxSpeed = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cFarnSp)
	{
		cFarnSpPrev = temp8;
	}
	else if (cFarnSp == cFarnSpPrev)
	{
		cFarnSp = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cKeyrMod)
	{
		cKeyrModPrev = temp8;
	}
	else if (cKeyrMod == cKeyrModPrev)
	{
		cKeyrMod = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == cSideTn)
	{
		cSideTnPrev = temp8;
	}
	else if (cSideTn == cSideTnPrev)
	{
		cSideTn = temp8;
	}

	ptr++;
	btPaddle = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gPlay)
	{
		gPlayPrev = temp8;
	}
	else if (gPlay == gPlayPrev)
	{
		gPlay = temp8;
	}

	ptr++;
	gDisLeft = CalcInfoMsgVal(ptr, 1);

	ptr++;
	btTuneTx = CalcInfoMsgVal(ptr, 1);

	ptr++;

	//process changes *******************************************************************************

	if (cTxSpeed != cTxSpeedPrev)
	{
		//speed changed - reset morse decoder time to match
		decodeWPM = cTxSpeed; //set it twice in case isr fires
		decodeWPM = cTxSpeed;
	}
	keyerDitDur = (1200 / (ISR_Interval / 1000)) / cTxSpeed; //adjust speed based on IRS_Timer setting - 1200 constant is based on 1 mSec timing	
	keyerDitFarnsDur = (1200 / (ISR_Interval / 1000)) / cFarnSp; //dit duration for farnsworth spacing

	ChkFilterChng(0); //chk for filter change from HMI

	if (gPlay != gPlayPrev)
	{
		SetTxMemState(gPlay);
	}

}

void Dec_Voice_Data(uint8_t & ptr)
{
	//Decode Voice Page data =================================================================
	uint8_t temp8 = 0;

	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == vTxLev)
	{
		vTxLevPrev = temp8;
	}
	else if (vTxLev == vTxLevPrev)
	{
		vTxLev = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == vLPFltr)
	{
		vLPFltrPrev = temp8;
	}
	else if (vLPFltr == vLPFltrPrev)
	{
		vLPFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == vNFltr)
	{
		vNFltrPrev = vNFltr;
	}
	else if (vNFltr == vNFltrPrev)
	{
		vNFltr = temp8;
	}

	//these were in gStat8
	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTx0)
	{
		gTx0Prev = temp8;
	}
	else if (gTx0 == gTx0Prev)
	{
		gTx0 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTx1)
	{
		gTx1Prev = temp8;
	}
	else if (gTx1 == gTx1Prev)
	{
		gTx1 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTx2)
	{
		gTx2Prev = temp8;
	}
	else if (gTx2 == gTx2Prev)
	{
		gTx2 = temp8;
	}


	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTx3)
	{
		gTx3Prev = temp8;
	}
	else if (gTx3 == gTx3Prev)
	{
		gTx3 = temp8;
	}


	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gTx4)
	{
		gTx4Prev = temp8;
	}
	else if (gTx4 == gTx4Prev)
	{
		gTx4 = temp8;
	}

	ptr++;
	gTxComp = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gEQSel = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gTxEQ = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRx0)
	{
		gRx0Prev = temp8;
	}
	else if (gRx0 == gRx0Prev)
	{
		gRx0 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRx1)
	{
		gRx1Prev = temp8;
	}
	else if (gRx1 == gRx1Prev)
	{
		gRx1 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRx2)
	{
		gRx2Prev = temp8;
	}
	else if (gRx2 == gRx2Prev)
	{
		gRx2 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRx3)
	{
		gRx3Prev = temp8;
	}
	else if (gRx3 == gRx3Prev)
	{
		gRx3 = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRx4)
	{
		gRx4Prev = temp8;
	}
	else if (gRx4 == gRx4Prev)
	{
		gRx4 = temp8;
	}

	ptr++;
	gRxEQ = CalcInfoMsgVal(ptr, 1);

	ptr++;

	//process changes ***************************************************************************

	if (hmiPage == HMI_VOICE)
	{
		if (vTxLev != vTxLevPrev)
		{
			if (gOMode == MODE_VOICE)
			{
				gTxLev = vTxLev;
				gTxLevPrev = gTxLev;
				vTxLevPrev = vTxLev;
			}
		}
		ChkFilterChng(0); //chk for filter change from HMI
	}

	SetEqualizer(false); //chk if there's any changes

}

void Dec_RTTY_Data(uint8_t & ptr)
{
	//Decode RTTY Page data =================================================================
	uint8_t temp8 = 0;

	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == rTxLev)
	{
		rTxLevPrev = temp8;
	}
	else if (rTxLev == rTxLevPrev)
	{
		rTxLev = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == rLPFltr)
	{
		rLPFltrPrev = temp8;
	}
	else if (rLPFltr == rLPFltrPrev)
	{
		rLPFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == rNFltr)
	{
		rNFltrPrev = temp8;
	}
	else if (rNFltr == rNFltrPrev)
	{
		rNFltr = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRtyFrq)
	{
		gRtyFrqPrev = temp8;
	}
	else if (gRtyFrq == gRtyFrqPrev)
	{
		gRtyFrq = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRtySft)
	{
		gRtySftPrev = gRtySft;
	}
	else if (gRtySft == gRtySftPrev)
	{
		gRtySft = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRtyStp)
	{
		gRtyStpPrev = temp8;
	}
	else if (gRtyStp == gRtyStpPrev)
	{
		gRtyStp = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gRtyBaud)
	{
		gRtyBaudPrev = temp8;
	}
	else if (gRtyBaud == gRtyBaudPrev)
	{
		gRtyBaud = temp8;
	}


	ptr++;

	//process changes *******************************************

	if (hmiPage == HMI_RTTY)
	{		
		if (rTxLev != rTxLevPrev)
		{
			if (gOMode == MODE_RTTY)
			{
				gTxLev = rTxLev;
				gTxLevPrev = gTxLev;
				rTxLevPrev = rTxLev;
			}
		}
		
		ChkFilterChng(0); //chk for filter change from HMI
		
		if (gRtyFrq != gRtyFrqPrev)
		{
			SetFilterFreq(11);
		}
		if (gRtySft != gRtySftPrev)
		{
			SetFilterFreq(12);
		}
	}


}

void Dec_Dig_Data(uint8_t & ptr)
{
	//Decode Dig Page data =================================================================
	uint8_t temp8 = 0;

	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == dTxLev)
	{
		dTxLevPrev = temp8;
	}
	else if (dTxLev == dTxLevPrev)
	{
		dTxLev = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1) & 0x01;
	if (temp8 == gDigRxPath)
	{
		gDigRxPathPrev = temp8;
	}
	else if (gDigRxPath == gDigRxPathPrev)
	{
		gDigRxPath = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1) & 0x01;
	if (temp8 == gDigTxPath)
	{
		gDigTxPathPrev = temp8;
	}
	else if (gDigTxPath == gDigTxPathPrev)
	{
		gDigTxPath = temp8;
	}
	
	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gDigVoxLev)
	{
		gDigVoxLevPrev = temp8;
	}
	else if (gDigVoxLev == gDigVoxLevPrev)
	{
		gDigVoxLev = temp8;
	}

	ptr++;
	gDigTxActiv = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gMon)
	{
		gMonPrev = temp8;
	}
	else if (gMon == gMonPrev)
	{
		gMon = temp8;
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gDigProc)
	{
		gDigProcPrev = temp8;
	}
	else if (gDigProc == gDigProcPrev)
	{
		gDigProc = temp8;
	}


	ptr++;

	//process changes ***********************************************************************************

	if (gOMode == MODE_DIG && hmiPage == HMI_DIG)
	{
		if (dTxLev != dTxLevPrev)
		{
			if (gTxLev == gTxLevPrev)
			{
				gTxLev = dTxLev;
				gTxLevPrev = gTxLev;
				dTxLevPrev = dTxLev;
			}
		}		
		//SerialOut("Dec Dig gTxLev=" + String(gTxLev), true);
	}

	if (gDigTxPath != gDigTxPathPrev || gDigRxPath != gDigRxPathPrev)
	{
		SetTxPath();
		SetOutputSource(0xff, true); //use the last state we set
	}

	if (gMon != gMonPrev)
	{
		SetOutputSource(0xff, true); //change output if this changes
		gMonPrev = gMon;
	}

	if (gDigProc != gDigProcPrev)
	{		
		SetEqualizer(true);
		SetOutputSource(0xff, true); //change output if this changes
		gDigProcPrev = gDigProc;
	}

	switch (gDigVoxLev)
	{
	case 0:
		gDigTxActiv = DIG_ACT_NOTHING; //turn off if disabled
		break;
	case 1:
		gDigVoxThresh = 4; //-40 dbm
		break;
	case 2:
		gDigVoxThresh = 6; //-37 dbm
		break;
	case 3:
		gDigVoxThresh = 8; //-34 dbm
		break;
	case 4:
		gDigVoxThresh = 14; //-30 dbm
		break;
	case 5:
		gDigVoxThresh = 24; //-25 dbm
		break;
	case 6:
		gDigVoxThresh = 42; //-20 dbm
		break;
	case 7:
		gDigVoxThresh = 75; //-15 dbm
		break;
	case 8:
		gDigVoxThresh = 141; //-10 dbm
		break;
	case 9:
		gDigVoxThresh = 3183; //-7.5 dbm
		break;
	case 10:
		gDigVoxThresh = 342; //-2.5 dbm
		break;
	}
}

void Dec_TxMem_Data(uint8_t & ptr)
{
	//Decode TxMem page data ===============================================================
	gAudioVMsg = CalcInfoMsgVal(ptr, 1);
	
	ptr++;
	gAudioRecSrc = CalcInfoMsgVal(ptr, 1);

	ptr++;
	//mode
	hmiTxMemMode = CalcInfoMsgVal(ptr, 1);
	
	ptr++;
	hmiTxMemCtr = CalcInfoMsgVal(ptr, 1);	

	ptr++;

}

void Dec_FreqMem_Data(uint8_t & ptr)
{
	uint8_t temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == hmiFreqMemStartAt && hmiFreqMemStartAtPrev != 0xff)
	{
		//ignore if we're waiting to update the freqmem page
		hmiFreqMemStartAtPrev = temp8;
	}
	else if (hmiFreqMemStartAt == hmiFreqMemStartAtPrev)
	{
		hmiFreqMemStartAt = temp8;
	}
	
		
	ptr++; //inc this because we didn't call a subroutine that did it for us


	//process changes *******************************************

	//normal freq list startAt selection				
	//if startAt=0 FreqMem page was just opened, reset prev so we'll update this page
	if (hmiFreqMemStartAt == 0)
	{
		if (hmiFreqMemStartAtPrev != 0xff)
		{
			if (hmiFreqMemStartAtPrev > 0)
			{
				hmiFreqMemStartAt = hmiFreqMemStartAtPrev;
			}
			else
			{
				hmiFreqMemStartAt = 1;
			}			
		}
	}
	else
	{		
		//round it to nearest 10 block					
		hmiFreqMemStartAt = (round(hmiFreqMemStartAt / 10) * 10) + 1;		
	}
}

void Dec_Log_Data(uint8_t & ptr)
{
	hmiLogOffset = CalcInfoMsgVal(ptr, 1); //value of h1 slider	
	
	ptr++;

}

void Dec_Help_Data(uint8_t & ptr)
{
	uint8_t temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == HELP_DISCLAIMERPG)
	{
		hmiReadDisclaimer = true; //if user has viewed the 2nd page, create file so we don't keep showing it		
		Chk4DisclaimerFile(); //create disclaimer file if it hasn't been created yet
	}
	ptr++;
}

void Dec_Config_Data(uint8_t & ptr)
{	
	gFFTAvg = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gMon = CalcInfoMsgVal(ptr, 1);
	if (gMon != gMonPrev)
	{
		SetOutputSource(0xff, true); //change output if this changes
		gMonPrev = gMon;
	}

	ptr++;
	bool temp = gVocal; //chk for cos	
	gVocal = CalcInfoMsgVal(ptr, 1);
	if (temp == false && gVocal == true)
	{
		hmiAnnounce = VOICE_SAY_Freq + VOICE_SAY_gRMode + VOICE_SAY_gOMode; //triggers saying freq & modes
	}

	ptr++;

}

void Dec_KeyPad_Data(uint8_t & ptr)
{
	//future

}

void Dec_RxMsg_Data(uint8_t & ptr)
{
	//future
}

void Dec_Input_Data(uint8_t & ptr)
{
	// future
}

void Dec_Edit_Data(uint8_t & ptr)
{
	editPgSplit = CalcInfoMsgVal(ptr, 1); 

	ptr++;
}

void Dec_SaveLog_Data(uint8_t & ptr)
{
	gCtstCtr = CalcInfoMsgVal(ptr, 2); //use 0xffff range

	ptr+=2; //inc twice for double word above
	gLogMode = CalcInfoMsgVal(ptr, 1); //0=normal, 1=contest	

	ptr++;	
}

void Dec_AutoLog_Data(uint8_t & ptr)
{
	//future
}

void Dec_Flex_Data(uint8_t & ptr)
{
	//Decode Flex Page data ==========================================================
	gFlexWNB = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexNB = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexNR = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexAF = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexPO = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexTO = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexLO = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gFlexAnt = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gWiFi = CalcInfoMsgVal(ptr, 1);

	ptr++;

}

void Dec_Radio_Data(uint8_t & ptr)
{
	//Decode Radio Page data ==========================================================
	gShareDB = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gRadioSel = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gRadioSelNum = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gRadioPopup = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gRadioChange = CalcInfoMsgVal(ptr, 1);

	ptr++;

}

void Dec_PCR_Data(uint8_t & ptr)
{
	uint32_t temp = gPCRStat;
	gPCRStat = CalcInfoMsgVal(ptr, 4);
	if (gPCRStat != temp)
	{
		Dec_gPCRStat();
	}

	ptr += 4;
	temp = gPCRCtrl;
	gPCRCtrl = CalcInfoMsgVal(ptr, 4);
	if (gPCRCtrl != temp)
	{
		Dec_gPCRCtrl();
	}

	ptr += 4;

}

void Dec_VFO_Data(uint8_t & ptr)
{
	//future
}

void Dec_Audio_Data(uint8_t & ptr)
{
	uint8_t temp8 = 0;
	gAudioRecBtn = CalcInfoMsgVal(ptr, 1);
	
	ptr++;
	gAudioPlayBtn = CalcInfoMsgVal(ptr, 1);
	
	ptr++;
	gAudioStartBtn = CalcInfoMsgVal(ptr, 1);
		
	ptr++;
	gAudioRecSrc = CalcInfoMsgVal(ptr, 1);
	
	ptr++;
	gAudioRecGain = CalcInfoMsgVal(ptr, 1);

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == gVolGain)
	{
		gVolGainPrev = temp8;
	}
	else if (gVolGain == gVolGainPrev)
	{
		gVolGain = temp8;
	}
	
	ptr++;
	gAudioVMsg = CalcInfoMsgVal(ptr, 1);
	
	ptr++;

	//================================================================
	ChkVolGain();
}

void Dec_Ant_Data(uint8_t & ptr)
{

	//Decode Ant Page data ==================================================================	
	gAnt = CalcInfoMsgVal(ptr, 1);
	//SerialOut("ptr= " + String(ptr) + ", gAnt = " + String(gAnt), true);

	ptr++;
	if (hmiPage == HMI_ANT)
	{
		antSaveTag = CalcInfoMsgVal(ptr, 1) & 0x07; //mask off all but lsb - 0xf1 is dummied in on home page since this isn't available from Ant page
		//SerialOut("antSaveTag = " + String(antSaveTag), true);
	}

	ptr++;
	gAntMode = CalcInfoMsgVal(ptr, 1);

	ptr++;
	gAntBand = CalcInfoMsgVal(ptr, 1); //this is dummied in at 0xff in the Home page block

	ptr++;

	//process changes *******************************************

	if (gAntBand != gAntBandPrev && gAnt != 0)
	{
		//check on Home page too to update antenna switch
		if (bandMemAnt[gAntBand] != 0)
		{
			//only allow band based switching if gAnt is valid for this band
			gAntPrev = gAnt; //save current ant
			gAnt = bandMemAnt[gAntBand];
			if (SetAntSwitch(false))
			{
				gAntBandPrev = gAntBand;
				gAntPrev = gAnt;
			}
			else
			{
				gAnt = gAntPrev; //revert if failed to switch
			}
		}
	}

	if (hmiPage == HMI_ANT)
	{
		//only time this is valid
		if (gAnt != gAntPrev)
		{
			if (gAntBand == 0xff)
			{
				//reset for all bands to same ant				
				for (int i = 0; i < bandMemMax; i++)
				{
					bandMemAnt[i] = gAnt;
				}
			}
			else
			{
				//update bandMemAnt[gAntBand] with new gAnt value				
				bandMemAnt[gAntBand] = gAnt;
				//chk for empty slots - fill them with this one - this is the first change
				for (int i = 0; i < bandMemMax; i++)
				{
					if (bandMemAnt[i] == 0)
					{
						bandMemAnt[i] = gAnt;
					}
				}
			}
			if (SetAntSwitch(false))
			{
				gAntPrev = gAnt; //if switched ok
			}
		}
	}

}

void Dec_Levels_Data(uint8_t& ptr)
{
	// using raw values from sliders - subtract 50 when you use them
	//slider value must be above 0 or it's ignored
	uint8_t temp8 = 0;
	
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gLnInLv))
	{
		gLnInLvPrev = temp8;
	}
	else if (uint8_t(gLnInLv) == gLnInLvPrev)
	{
		gLnInLv = float(temp8);
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gMicInLv))
	{
		gMicInLvPrev = temp8;
	}
	else if (uint8_t(gMicInLv) == gMicInLvPrev)
	{
		gMicInLv = float(temp8);
	}
	
	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gLnOutLv))
	{
		gLnOutLvPrev = temp8;
	}
	else if (uint8_t(gLnOutLv) == gLnOutLvPrev)
	{
		gLnOutLv = float(temp8);
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gUSBInLv))
	{
		gUSBInLvPrev = temp8;
	}
	else if (uint8_t(gUSBInLv) == gUSBInLvPrev)
	{
		gUSBInLv = float(temp8);
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gUSBOutLv))
	{
		gUSBOutLvPrev = temp8;
	}
	else if (uint8_t(gUSBOutLv) == gUSBOutLvPrev)
	{
		gUSBOutLv = float(temp8);
	}

	ptr++;
	temp8 = CalcInfoMsgVal(ptr, 1);
	if (temp8 == uint8_t(gFFTLv))
	{
		gFFTLvPrev = temp8;
	}
	else if (uint8_t(gFFTLv) == gFFTLvPrev)
	{
		gFFTLv = float(temp8);
	}

	ptr++;

	//process changes *******************************************

	//extend encoder time if changing
	if ((uint8_t(gLnInLv) != gLnInLvPrev) || (uint8_t(gMicInLv) != gMicInLvPrev) || (uint8_t(gUSBInLv) != gUSBInLvPrev) || (uint8_t(gUSBOutLv) != gUSBOutLvPrev) || (uint8_t(gLnOutLv) != gLnOutLvPrev) || (uint8_t(gFFTLv) != gFFTLvPrev))
	{
		hmiEncoderTimeout = millis(); //restart timeout
		hmiUpdateCtr = 0; //immediate display update
	}	
}

void Dec_StringInfo()
{
	/*
	  decode string in info block
	  string blocks use '0xbf' in preamble instead of '0xaf'
	  [0] is not used in hmiRx[]
	  page # is [1]
	  index of string is [2]
	  strings starts at [3] to '0xfb' terminator
	*/

	//get string
	String temp = " ";
	String txt = "";
	for (int i = 3; i < 150; i++)
	{
		if (hmiRx[i] == 0xfb)
		{
			//terminator - exit
			break;
		}
		temp += char(hmiRx[i]);
	}

	temp = temp.trim();

	byte idx = hmiRx[2]; //index points to which variable is here
	//SerialOut(String(idx) + ": " + temp, true);
		
	int offset = 0;

	switch (hmiRx[1])
	{
	case HMI_TXMEM:
		if (idx > 9)
		{
			return; //bad index
		}
		if (hmiTxMemMode == 0)
		{
			offset = 0; //cw
		}
		else if (hmiTxMemMode == 1)
		{
			offset = 10; //voice
		}
		else if (hmiTxMemMode == 2)
		{
			offset = 20; //rtty
		}
		else if (hmiTxMemMode == 3)
		{
			offset = 30; //Rx>SD
		}

		hmiTxMem[idx + offset] = temp;
		break;
	case HMI_EDIT:
		if (idx >= 0 && idx < 7)
		{
			editPgTxt[idx] = temp; //holds bx.txt values while Edit page is open - save them in SaveEditPgFromHMI() when bSave is pressed
		}
		break;
	case HMI_SAVELOG:
		switch (idx)
		{
		case 0:			
			if (temp.length() > 0)
			{
				//good data, format it and add it to the log string				
				logADI[idx] = "<CALL:" + String(temp.length()) + ":S>" + temp;				
			}
			else
			{
				logADI[idx] = "<CALL:0:S>"; //not set
				temp = "Call....";
			}			
			break;
		case 1:
			if (temp.length() == 10)
			{
				//format it to YYYYMMDD
				txt = temp.substring(7, 10) + temp.substring(1, 2) + temp.substring(4, 5);
				
				logADI[idx] = "<QSO_DATE:" + String(txt.length()) + ":D>" + txt;
				temp = txt;
			}
			else
			{
				logADI[idx] = "<QSO_DATE:0:D>"; //not set
				temp = "Date....";
			}
			break;
		case 2:
			if (temp.length() > 0)
			{				
				logADI[idx] = "<FREQ:" + String(temp.length()) + ":N>" + temp;				
			}
			else
			{
				logADI[idx] = "<FREQ:0:N>"; //not set
				temp = "Frequency";
			}
			break;
		case 3:
			if (temp.length() > 0)
			{				
				logADI[idx] = "<MODE:" + String(temp.length()) + ":S>" + temp;				
			}
			else
			{
				logADI[idx] = "<MODE:0:S>"; //not set
				temp = "Mode";
			}
			break;
		case 4:
			if (temp.length() > 0)
			{				
				logADI[idx] = "<RST_SENT:" + String(temp.length()) + ":S>" + temp;				
			}
			else
			{
				logADI[idx] = "<RST_SENT:0:S>"; //not set
				temp = "RSTs";
			}
			break;
		case 5:
			if (temp.length() > 0)
			{			
				logADI[idx] = "<COMMENT:" + String(temp.length()) + ":S>" + temp;			
			}
			else
			{
				logADI[idx] = "<COMMENT:0:S>"; //not set
				temp = "";
			}
			break;
		case 6:
			if (temp.length() == 8)
			{
				//format time to HHMM	
				txt = temp.substring(1, 2) + temp.substring(4,5);
				logADI[idx]= "<TIME_OFF:" + String(txt.length()) + ":T>" + txt;
				temp = txt;
			}
			else
			{
				logADI[idx] = "<TIME_OFF:0:T>"; //not set
				temp = "Time....";
			}
			break;
		case 7:
			if (temp.length() > 0)
			{				
				logADI[idx] = "<TX_PWR:" + String(temp.length()) + ":N>" + temp;				
			}
			else
			{
				logADI[idx] = "<TX_PWR:0:N>"; //not set
				temp = "Pwr";
			}
			break;
		case 8:
			if (temp.length() > 0)
			{				
				logADI[idx] = "<RST_RCVD:" + String(temp.length()) + ":S>" + temp;				
			}
			else
			{
				logADI[idx] = "<RST_RCVD:0:S>"; //not set
				temp = "RSTr";
			}
			break;
		case 9:
			if (temp.length() > 0)
			{
				int val = temp.toInt();
				if (val > 0 && val < 0xffff)
				{
					//contest serial #
					logADI[idx] = "<SRX:" + String(temp.length()) + ">" + temp; //integer indicator not required										
				}
				else
				{
					logADI[idx] = "<SRX:0>";
					temp = "";
				}
			}			
			break;
		case 10:
			if (temp.length() > 0)
			{
				logADI[idx] = "SRX_STRING:" + String(temp.length()) + ":S>" + temp;				
			}
			else
			{
				logADI[idx] = "<SRX_STRING:0:S>";
				temp = "";
			}
			break;
		}				
		logTxt[idx] = temp;
		break;
	case HMI_INPUT:
		if (temp.length())
		{
			hmi_tData = temp; //so we can use it if using keyboard keyer
		}		
		break;
	case HMI_FLEX:
		//IP info send every 5 seconds from Flex & Config pages - same format
IPInfo:
		switch (idx)
		{
		case 0:
			gAPName = temp;
			break;
		case 1:
			gAPKey = temp;
			break;
		case 2:
			gAPChan = uint8_t(temp.toInt());
			break;
		case 3:
			gSSID = temp;
			SerialOut("gSSID= " + gSSID, true);
			break;
		case 4:
			gStaKey = temp;
			break;
		case 5:
			gStaAdrs = temp;
			break;
		case 6:
			gSubnet = temp;
			break;
		case 7:
			gGW = temp;
			break;
		case 8:
			gIPMode = uint8_t(temp.toInt());
			break;
		case 9:
			gFlexIP = temp;	
			break;
		}
		break;
	case HMI_CONFIG:
		goto IPInfo;
		break;
	}
}



//==============================================================================================
// End of txInfo Decoding Routines
// Add more routines above when adding additional pages to the HMI
//==============================================================================================

/*
   This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of the MPL was not distributed with this  file, You can obtain one at https://mozilla.org/MPL/2.0/.
*/

uint32_t CalcInfoMsgVal(uint8_t ptr, uint8_t cntr)
{
	//calculate the value of the data in hmiRx[] starting at ptr for cntr bits
	//data is in little-endian format and is moved from the ptr location to hmi[2][3][4][5] so the normal decoders work
	//  NOTE:  hmi[2][3][4][5]  hold two of the start bytes, the page#, and # of total bytes, so they're not needed

	uint32_t val = hmiRx[ptr];
	hmiRx[2] = hmiRx[ptr];
	if (cntr > 1)
	{
		val += hmiRx[ptr + 1] << 8;
		hmiRx[3] = hmiRx[ptr + 1];
	}
	if (cntr > 2)
	{
		val += hmiRx[ptr + 2] << 16;
		hmiRx[4] = hmiRx[ptr + 2];
	}
	if (cntr > 3)
	{
		val += hmiRx[ptr + 3] << 24;
		hmiRx[5] = hmiRx[ptr + 3];
	}
	return val;

}

void ChkEncoderChng()
{
	//This routine checks for changes from encoder and sends them to HMI
	//The Dec_xxx routines above check for changes from HMI - they don't resend changes back to HMI
	//HOME PAGE CHANGES
	//chk for encoder parameter change & send to hmi
	blockHMI = millis() + 1500;
	if (gUnit != gUnitPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("gUnit=" + String(gUnit));
		Tx2HMI("gUnit=" + String(gUnit));
		gUnitPrev = gUnit;
	}
	if (gFFTFreq != gFFTFreqPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("gFFTFreq=" + String(int(gFFTFreq / 25))); //send freq in 25 hz steps so they fit in 1 byte
		Tx2HMI("gFFTFreq=" + String(int(gFFTFreq / 25)));
		SetFilterFreq(13);
		Sidetone.frequency(gFFTFreq);
		Sidetone.update();
		hmiFFTMarkerUpdate = 0; //immediate update		
	}
	if (gTxLev != gTxLevPrev)
	{
		//set common values - tm0 on Home moves these to individual pages			
		hmiWait4Reply = 0;
		Tx2HMI("gTxLev=" + String(gTxLev));
		Tx2HMI("gTxLev=" + String(gTxLev));
		bandMemTxLev[gBand] = gTxLev;
		SetOutputSource(0xff, true); //change level here							
	}
	if (gLPFltr != gLPFltrPrev)
	{
		//update LP filter			
		hmiWait4Reply = 0;
		Tx2HMI("gLPFltr=" + String(gLPFltr));
		Tx2HMI("gLPFltr=" + String(gLPFltr));
		SetFilterFreq(14);
		bandMemLPFltr[gBand] = gLPFltr;		
	}

	if (gNFltr != gNFltrPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("gNFltr=" + String(gNFltr));
		Tx2HMI("gNFltr=" + String(gNFltr));
		SetFilterFreq(15);
		bandMemNFltr[gBand] = gNFltr;		
	}

	if (gFFTGain > 100)
	{
		//bad value, revert to previous
		gFFTGain = gFFTGainPrev;
	}
	if (gFFTGain != gFFTGainPrev)
	{		
		hmiWait4Reply = 0;
		Tx2HMI("Home.fftGain.val=" + String(int(gFFTGain)));
		Tx2HMI("Home.fftGain.val=" + String(int(gFFTGain)));
		//if - set to 0. - gain just inverts phase
		float g = gFFTGain + (gFFTLv - 50.0);
		if (g < 0)
		{
			g = LEV_OFF;
		}
		Amp_FFT.gain(g);
		Amp_FFT.update();		
	}

	if (gVolGain != gVolGainPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("gVolGain=" + String(int(gVolGain)));
		Tx2HMI("gVolGain=" + String(int(gVolGain)));
	}
	ChkVolGain();	

	if (gSqGain > 100)
	{
		//revert
		gSqGain = gSqGainPrev;
	}
	if (gSqGain != gSqGainPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("Home.sqGain.val=" + String(int(gSqGain)));
		Tx2HMI("Home.sqGain.val=" + String(int(gSqGain)));
		gSqGainPrev = gSqGain;
	}

	//CW PAGE CHANGES
	if (cTxSpeed != cTxSpeedPrev)
	{
		//speed changed - reset morse decoder time to match
		decodeWPM = cTxSpeed; //set it twice in case isr fires
		decodeWPM = cTxSpeed;
		hmiWait4Reply = 0;
		Tx2HMI("CW.cTxSpeed.val=" + String(cTxSpeed));
		Tx2HMI("CW.cTxSpeed.val=" + String(cTxSpeed));
		cTxSpeedPrev = cTxSpeed;
	}
	keyerDitDur = (1200 / (ISR_Interval / 1000)) / cTxSpeed; //adjust speed based on IRS_Timer setting - 1200 constant is based on 1 mSec timing	
	keyerDitFarnsDur = (1200 / (ISR_Interval / 1000)) / cFarnSp; //dit duration for farnsworth spacing

	if (hmiPage == HMI_CW)
	{
		if (cFarnSp != cFarnSpPrev)
		{
			//farnsworth speed changed by encoder
			hmiWait4Reply = 0;
			Tx2HMI("CW.cFarnSp.val=" + String(cFarnSp));
			Tx2HMI("CW.cFarnSp.val=" + String(cFarnSp));
			cFarnSpPrev = cFarnSp;
		}

		if (cKeyrMod != cKeyrModPrev)
		{
			//change keyer mode
			hmiWait4Reply = 0;
			Tx2HMI("CW.cKeyrMod.val=" + String(cKeyrMod));
			Tx2HMI("CW.cKeyrMod.val=" + String(cKeyrMod));
			cKeyrModPrev = cKeyrMod;
		}

		if (cSideTn != cSideTnPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("CW.cSideTn.val=" + String(cSideTn));
			Tx2HMI("CW.cSideTn.val=" + String(cSideTn));
			cSideTnPrev = cSideTn;
		}

		ChkFilterChng(1); //chk for filter change from encoder
	}

	//VOICE PAGE CHANGES
	if (hmiPage == HMI_VOICE)
	{
		if (vTxLev != vTxLevPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("Voice.vTxLev.val=" + String(vTxLev));
			Tx2HMI("Voice.vTxLev.val=" + String(vTxLev));
			if (gOMode == MODE_VOICE)
			{
				gTxLev = vTxLev;
				gTxLevPrev = gTxLev;
			}				
		}
		
		ChkFilterChng(1); //chk for filter change from encoder

		//update eq levels
		if (gEQSel == false && gRxEQ == true)
		{
			//rx selected
			if (gRx0 != gRx0Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gRx0.val=" + String(gRx0));
				Tx2HMI("Voice.gRx0.val=" + String(gRx0));
				gRx0Prev = gRx0;
			}
			if (gRx1 != gRx1Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gRx1.val=" + String(gRx1));
				Tx2HMI("Voice.gRx1.val=" + String(gRx1));
				gRx1Prev = gRx1;
			}
			if (gRx2 != gRx2Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gRx2.val=" + String(gRx2));
				Tx2HMI("Voice.gRx2.val=" + String(gRx2));
				gRx2Prev = gRx2;
			}
			if (gRx3 != gRx3Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gRx3.val=" + String(gRx3));
				Tx2HMI("Voice.gRx3.val=" + String(gRx3));
				gRx3Prev = gRx3;
			}
			if (gRx4 != gRx4Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gRx4.val=" + String(gRx4));
				Tx2HMI("Voice.gRx4.val=" + String(gRx4));
				gRx4Prev = gRx4;
			}
		}
		else if (gEQSel == true && gTxEQ == true)
		{
			//rx selected
			if (gTx0 != gTx0Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gTx0.val=" + String(gTx0));
				Tx2HMI("Voice.gTx0.val=" + String(gTx0));
				gTx0Prev = gTx0;
			}
			if (gTx1 != gTx1Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gTx1.val=" + String(gTx1));
				Tx2HMI("Voice.gTx1.val=" + String(gTx1));
				gTx1Prev = gTx1;
			}
			if (gTx2 != gTx2Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gTx2.val=" + String(gTx2));
				Tx2HMI("Voice.gTx2.val=" + String(gTx2));
				gTx2Prev = gTx2;
			}
			if (gTx3 != gTx3Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gTx3.val=" + String(gTx3));
				Tx2HMI("Voice.gTx3.val=" + String(gTx3));
				gTx3Prev = gTx3;
			}
			if (gTx4 != gTx4Prev)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.gTx4.val=" + String(gTx4));
				Tx2HMI("Voice.gTx4.val=" + String(gTx4));
				gTx4Prev = gTx4;
			}
		}
		SetEqualizer(false); //chk if there's any changes
	}

	//CHECK FOR RTTY PAGE CHANGES
	if (hmiPage == HMI_RTTY)
	{
		if (rTxLev != rTxLevPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("RTTY.rTxLev.val=" + String(rTxLev));
			Tx2HMI("RTTY.rTxLev.val=" + String(rTxLev));
			if (gOMode == MODE_RTTY)
			{
				gTxLev = rTxLev;
				gTxLevPrev = gTxLev;
			}			
		}
		
		ChkFilterChng(1); //chk for filter change from encoder

		if (gRtyFrq != gRtyFrqPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("RTTY.gRtyFrq.val=" + String(gRtyFrq));
			Tx2HMI("RTTY.gRtyFrq.val=" + String(gRtyFrq));
			SetFilterFreq(16);
			gRtyFrqPrev = gRtyFrq;
		}
		if (gRtySft != gRtySftPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("RTTY.gRtySft.val=" + String(gRtySft));
			Tx2HMI("RTTY.gRtySft.val=" + String(gRtySft));
			SetFilterFreq(17);
			gRtySftPrev = gRtySft;
		}
		if (gRtyStp != gRtyStpPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("RTTY.gRtyStp.val=" + String(gRtyStp));
			Tx2HMI("RTTY.gRtyStp.val=" + String(gRtyStp));
			gRtyStpPrev = gRtyStp;
		}
		if (gRtyBaud != gRtyBaudPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("RTTY.gRtyBaud.val=" + String(gRtyBaud));
			Tx2HMI("RTTY.gRtyBaud.val=" + String(gRtyBaud));
			gRtyBaudPrev = gRtyBaud;
		}
	}

	//CHK FOR DIGITAL PAGE CHANGES
	if (hmiPage == HMI_DIG)
	{
		if (dTxLev != dTxLevPrev)
		{
			hmiWait4Reply = 0;
			Tx2HMI("Dig.dTxLev.val=" + String(dTxLev));
			Tx2HMI("Dig.dTxLev.val=" + String(dTxLev));
			if (gOMode == MODE_DIG && gTxLev == gTxLevPrev)
			{
				gTxLev = dTxLev;
				gTxLevPrev = gTxLev;
			}			
		}		
	}

	if (gDigTxPath != gDigTxPathPrev || gDigRxPath != gDigRxPathPrev)
	{
		SetTxPath();
		SetOutputSource(0xff, true); //use the last state we set
		gDigTxPathPrev = gDigTxPath;
		gDigRxPathPrev = gDigRxPath;
	}

	if (gMon != gMonPrev)
	{
		SetOutputSource(0xff, true); //change output if this changes
		gMonPrev = gMon;
	}

	if (gDigVoxLev != gDigVoxLevPrev)
	{
		hmiWait4Reply = 0;
		Tx2HMI("Dig.gVoxLev.val=" + String(gDigVoxLev));
		Tx2HMI("Dig.gVoxLev.val=" + String(gDigVoxLev));
		gDigVoxLevPrev = gDigVoxLev;
	}

	switch (gDigVoxLev)
	{
	case 0:
		gDigTxActiv = DIG_ACT_NOTHING; //turn off if disabled
		break;
	case 1:
		gDigVoxThresh = 4; //-40 dbm
		break;
	case 2:
		gDigVoxThresh = 6; //-37 dbm
		break;
	case 3:
		gDigVoxThresh = 8; //-34 dbm
		break;
	case 4:
		gDigVoxThresh = 14; //-30 dbm
		break;
	case 5:
		gDigVoxThresh = 24; //-25 dbm
		break;
	case 6:
		gDigVoxThresh = 42; //-20 dbm
		break;
	case 7:
		gDigVoxThresh = 75; //-15 dbm
		break;
	case 8:
		gDigVoxThresh = 141; //-10 dbm
		break;
	case 9:
		gDigVoxThresh = 3183; //-7.5 dbm
		break;
	case 10:
		gDigVoxThresh = 342; //-2.5 dbm
		break;
	}
	
	gEncoderChng = false; //reset flag
	blockHMI = millis() + 1000;
}

void ChkVolGain()
{
	//chk gVolGain for proper bounds - if changed, set volume gain
	if (gVolGain > 50)
	{
		//bad value, revert to previous
		gVolGain = gVolGainPrev;
	}
	if (gVolGain != gVolGainPrev)
	{
		if (gTxMode < RADIO_TX_PADDLE && gTxModeSet2 < RADIO_TX_PADDLE)
		{
			ChkRxSquelch(true);
		}
	}
}

void ChkFilterChng(uint8_t src)
{
	//chk for filter change
	//src=0 if called from HMI decoder
	//src=1 if called from encoder
	uint8_t fromHMI = 0;
	uint8_t fromEnc = 1;

	//SerialOut("pg= " + String(hmiPage) + ", src= " + String(src) + ", vLPFltr= " + String(vLPFltr) + ",prev= " + String (vLPFltrPrev), true);
	
	if (hmiPage == HMI_CW || src == fromEnc)
	{
		if (cLPFltr != cLPFltrPrev)
		{			
			if (src == fromEnc)
			{
				hmiWait4Reply = 0;
				Tx2HMI("CW.cLPFltr.val=" + String(cLPFltr));
				Tx2HMI("CW.cLPFltr.val=" + String(cLPFltr));
			}
			if (gOMode == MODE_CW)
			{
				gLPFltr = cLPFltr;
				SetFilterFreq(21);
				bandMemLPFltr[gBand] = gLPFltr;
				if (src == fromHMI)
				{
					cLPFltrPrev = cLPFltr;
					gLPFltrPrev = gLPFltr;
				}
			}
		}

		if (cNFltr != cNFltrPrev)
		{			
			if (src == fromEnc)
			{
				hmiWait4Reply = 0;
				Tx2HMI("CW.cNFltr.val=" + String(cNFltr));
				Tx2HMI("CW.cNFltr.val=" + String(cNFltr));
			}
			if (gOMode == MODE_CW)
			{
				gNFltr = cNFltr;
				SetFilterFreq(22);
				bandMemNFltr[gBand] = gNFltr;
				if (src == fromHMI)
				{
					cNFltrPrev = cNFltr;
					gNFltrPrev = gNFltr;
				}
			}			
		}
	}
	
	if (hmiPage == HMI_VOICE || src == fromEnc)
	{
		if (vLPFltr != vLPFltrPrev)
		{
			if (src == fromEnc)
			{				
				hmiWait4Reply = 0;
				Tx2HMI("Voice.vLPFltr.val=" + String(vLPFltr));
				Tx2HMI("Voice.vLPFltr.val=" + String(vLPFltr));				
			}
			if (gOMode == MODE_VOICE)
			{
				gLPFltr = vLPFltr;
				SetFilterFreq(23);
				bandMemLPFltr[gBand] = gLPFltr;
				if (src == fromHMI)
				{
					vLPFltrPrev = vLPFltr;
					gLPFltrPrev = gLPFltr;
				}
			}			
		}
		if (vNFltr != vNFltrPrev)
		{
			if (src == fromEnc)
			{
				hmiWait4Reply = 0;
				Tx2HMI("Voice.vNFltr.val=" + String(vNFltr));
				Tx2HMI("Voice.vNFltr.val=" + String(vNFltr));
			}
			if (gOMode == MODE_VOICE)
			{				
				gNFltr = vNFltr;
				SetFilterFreq(24);
				bandMemNFltr[gBand] = gNFltr;
				if (src == fromHMI)
				{
					vNFltrPrev = vNFltr;
					gNFltrPrev = gNFltr;
				}
			}			
		}
	}
	
	if (hmiPage == HMI_RTTY || src == fromEnc)
	{
		if (rLPFltr != rLPFltrPrev)
		{
			if (src == fromEnc)
			{
				hmiWait4Reply = 0;
				Tx2HMI("RTTY.rLPFltr.val=" + String(rLPFltr));
				Tx2HMI("RTTY.rLPFltr.val=" + String(rLPFltr));
			}
			if (gOMode == MODE_RTTY)
			{
				gLPFltr = rLPFltr;
				SetFilterFreq(25);
				bandMemLPFltr[gBand] = gLPFltr;
				if (src == fromHMI)
				{
					rLPFltrPrev = rLPFltr;
					gLPFltrPrev = gLPFltr;
				}
			}			
		}
		if (rNFltr != rNFltrPrev)
		{
			if (src == fromEnc)
			{
				hmiWait4Reply = 0;
				Tx2HMI("RTTY.rNFltr.val=" + String(rNFltr));
				Tx2HMI("RTTY.rNFltr.val=" + String(rNFltr));
			}
			if (gOMode == MODE_RTTY)
			{
				gNFltr = rNFltr;
				SetFilterFreq(26);
				bandMemNFltr[gBand] = gNFltr;
				if (src == fromHMI)
				{
					rNFltrPrev = rNFltr;
					gNFltrPrev = gNFltr;
				}
			}
		}
	}
}