#!/usr/bin/python3
#
#
"""
     This program does a scan of admittance versus frequency
     over a range specified by the call
     
     The call is:
     
     ./y_scan.py b u n G B p q
     
     where  c is the centre frequency in Hz,
			w is the frequency width, in Hz,
			n is th number of frequencies
            G is the conductance to subtract
            B us the susceptance to subtract
            p, if zero, don't plot, otherwise plot
            q, if zero, don't write strays file, otherwise write
            
    the program writes a file, adm.txt. which gives frequency, G , B and phase in each line
    it also write another file, strays.txt which contains one line:
    f     G     B
    where: f is the estimted frequency with the highest G
           G is the average conductance at the two ends of the spectrum
           B is the average susceptance at the two ends of the spectrum 
        ... it does the average of 10% of all the data points at each end
			
	Jim Koehler, Comox, April, 2019
"""
import numpy as np
import math
import sys
import serial
import time
import os

def rect_to_polar(x, y):
	mag = math.sqrt(x**2 + y**2)
	ph = math.tan(y/x)
	return mag, ph
	
def polar_to_rect(mag, ph):
	x = mag * math.cos(ph)
	y = mag * math.sin(ph)
	return x, y

def get_cal():
	r = np.loadtxt("calibration.txt")
	return r
	
def calibrate( inp, f, c):
	rh = inp[0]  # inp is the uncorrected list with rho magnitude and phase
	ph = inp[1]
	i = round(f / 1e6) - 1 # points to the correct calibration number
	rh = rh * c[i, 0] # correct magnitude
	ph = ph - c[i, 1] # correct phase
	a, b = polar_to_rect(rh, ph) #convert to rectangular
	return a + b * 1j
	

def do_scan(b, u, n, G, B, write_enable): # arguments are all strings
    R = complex(50.0, 0)
    TWO = complex(2.0, 0)
    r = [0, 0]
    comm = serial.Serial('/dev/ttyACM0', baudrate = 115200, timeout = 0.3)
    comm.readlines()    # flush the read buffer
    comm.write(s2b("b " + b + "\r")) 
    time.sleep(0.2)
    comm.write(s2b("u " + u + "\r")) 
    time.sleep(0.2)
    comm.write(s2b("n " + n + "\r")) 
    time.sleep(0.2)    
    comm.write(s2b("l\r"))        
    time.sleep(0.2)    
    comm.readlines()
    
   
    c = get_cal()
    
    comm.write(s2b("g\r"))
    
    fh=open("tmp.txt", 'wb')   
    for i in range(int(n)):
        v = comm.readline()
        fh.write(v)
    fh.close()
    
    fh=open("tmp.txt", 'rb')
    fh_out = open("adm.txt", 'wb')
    fh_out.write(s2b("#    Admittance vs frequency\n"))
    fh_out.write(s2b("# \n"))
    fh_out.write(s2b("# freq         G              B             Phase\n"))
    fh_out.write(s2b("# \n"))
   
   
    for i in range(int(n)):
        v = str.split(b2s(fh.readline()))
        r[0] = float(v[1]) / float(v[3])
        r[1] = float(v[2]) - float(v[4])
        rho = calibrate(r, float(v[0]), c)
        z = R * rho/(TWO - rho)
        y = 1/z
        g_0 = y.real - float(G)
        b_0 = y.imag - float(B)
        y = g_0 + (b_0 * 1j)
        fh_out.write(s2b(v[0] + ' ' + str(y.real) + ' ' + str(y.imag) + ' ' + str(math.atan2(y.imag, y.real)) + '\n'))
 
    fh.close()
    fh_out.close()
    
    fh=open("adm.txt", 'rb')    # now to find average G and B
    for i in range(4): # read the four comment lines
        v = fh.readline()
        
    num = round(float(n) / 10.0)
    g_sum = 0.0
    b_sum = 0.0
    count = 0
    g_max = 0.0
    f_max = 0
    for i in range(int(n)):
        v = str.split(b2s(fh.readline()))
        
        freq = float(v[0])
        g = float(v[1])
        b = float(v[2])
        
        if (g > g_max):
            g_max = g
            f_max = freq
        
        if (i < num):
            g_sum += g
            b_sum += b
            count += 1
        if ( i > (int(n) - num)):
            g_sum += g
            b_sum += b
            count += 1
    fh_out.close()

    g_0 = g_sum / count
    b_0 = b_sum / count
    
    if (int(write_enable) != 0):
        f = open("strays.txt", 'wb')    # now to write stray parameters file
        f.write(s2b(str(f_max) + ' ' + str(g_0) + ' ' + str(b_0) + '\n'))
        f.close()
   
	

def b2s(message):
    '''Byte to string'''
    return bytes.decode(message)

def s2b(message):
    '''string to bytes'''
    return bytearray(message, "ascii")

def main():
    c = int(sys.argv[1]) # the first argument of the command line
    w = int(sys.argv[2]) # the second argument of the command line
    n = sys.argv[3] # the third argument of the command line
    G = sys.argv[4]
    B = sys.argv[5]
    plot = sys.argv[6]
    write_enable = sys.argv[7]
    b = str(c - (w/2))
    u = str(c + (w/2))
    do_scan(b, u, n, G, B, write_enable)
    if (int(plot) != 0):
        string = "-c admitplot_xtal.gp " + b + " " + u
        os.system("gnuplot " + string)
        os.system("feh adm.png")

if __name__=="__main__":
    main()
    
