#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  make_rho_cal.py
#  
#  Copyright 2019 James A.R. Koehler <jim@pythagoras>
#  
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#  
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#  
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#  MA 02110-1301, USA.
#  
#  
"""
		make_rho_cal.py
		
	A program to make the calibration file, "calibration.txt", from the
file created by the make_default_rho.py program.  This file, named
"default_rho_load.txt" is the measured rho's with a good load
termination and includes measured values of the secondary port which is
normally just left open so they're meaningless.  The file
has 600 measurements, one each MHz, from 1 to 600 MHz.  Each line is of
format:

frequency	rho_magnitude	rho_phase rho_mag_secondary rho_phase_secondary

	The output file, "calibration.txt" has 600 lines of format
	
rho_corr	phas_corr

where rho_corr is a number which you multiply to the measured magnitude'
and phas_corr is a phase correction you make to the measured phase.

NOTE: only the main port values are used to create the calibrtion.txt file


Jim Koehler, VE5FP, Comox, April, 2019

"""

import numpy as np
import math

def rect_to_polar(x, y):
	mag = math.sqrt(x**2 + y**2)
	ph = math.tan(y/x)
	return mag, ph
	
def polar_to_rect(mag, ph):
	x = mag * math.cos(ph)
	y = mag * math.sin(ph)
	return x, y

def main(args):	
	lo = np.loadtxt("default_rho_load.txt", delimiter=' ')
	result = np.ndarray((600, 2), dtype = float)
	for i in range(600):
		result[i, 0] = 1.0 / lo[i, 1]
		result[i, 1] = lo[i, 2]
	np.savetxt("calibration.txt", result)
			 
	return 0

if __name__ == '__main__':
	import sys
	sys.exit(main(sys.argv))
	
