#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#  make_default_rho.py
#
#  Copyright 2019 James A.R. Koehler <jim@pythagoras>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#  MA 02110-1301, USA.
#
#  Usage: make_default_rho.py n name
#       where n is the number of repetitons of measurement at each
#       frequency
#   and 'name' is added to the output file name thus:
#                       "default_rho" + name + ".txt"
#
#
#
#
# Jim Koehler, April 3, 2019
#

import sys
import serial
import time
import math
commPort = 'COMx'

###########################
# measure function
#
#  measure(f, comm, m)
#
# returns the ratio and phase
# from 'm' measurements at 'f' Hz
# using 'comm' as the communication
# port
#
# the return value is a list with five elements:
#       the frequency in Hz
#       ratio and phase (radians) of the primary port
#       ratio and phase of the secondary port
###########################
def measure(f, comm, m):
    comm.write(s2b("f " + str(f) + "\r"))   # set the frequency
    d =[]
    sum = []
    for j in range(int(m)):
        comm.write(s2b("m\r"))
        d = str.split(b2s(comm.readline())) # the seven list elements are all strings
        f = int(d[0])

        ratio = float(d[1]) / float(d[3])
        phase = float(d[2]) - float(d[4])
        if (phase > 3.14159265):
            phase = phase - 6.2818531
        if (phase < -3.14159265):
            phase = phase + 6.2818531


        ratio1 = float(d[5]) / float(d[3])
        phase1 = float(d[6]) - float(d[4])
        if (phase1 > 3.14159265):
            phase1 = phase1 - 6.2818531
        if (phase1 < -3.14159265):
            phase1 = phase1 + 6.2818531
        if (j == 0):
            sum = [f, ratio, phase, ratio1, phase1]
        else:
            sum[1] = sum[1] + ratio
            sum[2] = sum[2] + phase
            sum[3] = sum[3] + ratio1
            sum[4] = sum[4] + phase1

    sum[1] = sum[1] / int(m)
    sum[2] = sum[2] / int(m)
    sum[3] = sum[3] / int(m)
    sum[4] = sum[4] / int(m)
    return sum      # this is a list of an integer and 4 floats

###########################
# do_scan function
#
#  do_scan(b, u, n, filename, m)
#
# the writes results to a file named 'filename'
# each line in the result file is formatted:
#       freq ratio phase ratio phase
# where the first ratio and phase are for the
# primary port and the second are for the auxiliary port
#
###########################
def do_scan(b, u, n, filename, m):
    comm = serial.Serial(commPort, baudrate = 115200, timeout = 0.3)
    comm.readlines()    # flush the read buffer
    comm.write(s2b("p\n")) # an arbitrary command to clear the USB port
    comm.readlines()    # flush the read buffer

    comm.write(s2b("b " + b + "\r")) # set z_meter start frequency
    time.sleep(0.2)
    comm.write(s2b("u " + u + "\r")) # set z_meter end frequency
    time.sleep(0.2)
    comm.write(s2b("n " + n + "\r")) # set z_meter number of data points in the scan
    time.sleep(0.2)
    comm.readlines()

    fh=open(filename, 'wb')

    for i in range(int(n)):
        f = (i + 1) * (int(u) - int(b)) / (int(n) - 1) # f will be in Hz
        print(f)
        data=measure(f, comm, m)
        fh.write(s2b(str(data[0]) + ' ' + str(data[1]) + ' ' + str(data[2]) + ' ' + str(data[3]) + ' ' + str(data[4])))
        fh.write(s2b("\n"))

    fh.close()

def b2s(message):
    '''Byte to string'''
    return bytes.decode(message)

def s2b(message):
    '''string to bytes'''
    return bytearray(message, "ascii")

def main():
    global commPort
    b = "1000000"
    u = "600000000"
    n = "600"
    m = sys.argv[1]     # number of measures at each frequency ... this is a string
    name = sys.argv[2]
    commPort = sys.argv[3]
    filename = "default_rho_" + name + ".txt"
    do_scan(b, u, n, filename, m)

if __name__ == '__main__':
    main()
