#!/home/jim/anaconda3/bin/python3
#
# program to read measure a batch of crystals
# it prompts for centre freuency, width and number of samples
# as well as an output file name.  Then, doing one crystal at
# a time, it writes the values of F0, Q, and R to an
# output file.  It uses the best fit to z-meter
# admittance outputs
#
# usage: ./fit_batch.py
#
# Jim Koehler
# Comox, BC, May, 2018
#
import sys # need this to read command line arugments
import os
import serial
import time
from scipy.optimize import curve_fit
from numpy import arctan

# expected inputs
ARG_length=1

def b2s(message):
    '''Byte to string'''
    return bytes.decode(message)

def s2b(message):
    '''string to bytes'''
    return bytearray(message, "ascii")


comm =serial.Serial('/dev/ttyUSB0', baudrate = 115200, timeout = 0.3)

def initialize(centre, width, number):
    comm.write(s2b("l \r")) # set mode to linear
    time.sleep(0.5)
    comm.write(s2b("f " + str(centre) + "\r")) # set Si5351A to start frequency first
    time.sleep(0.5)
    comm.write(s2b("w " + str(width)  + "\r"))
    time.sleep(0.5)
    comm.write(s2b("n " + str(number) + "\r"))
    time.sleep(0.5)
    comm.readlines()    # flush the read buffer
    

def doscan_adm(centre, number):
#
# directs the z-meter to output a scan of admittance data
# returns the approximate centre frequency and all the z-meter
# output lines are of form: f, G, B, Y(mag), y(phase
# the returned variable, is a list of strings
#
    data=[]
    lines = []
    
    comm.timeout = 4.0
    comm.write(s2b("v\r"))   # this starts the scan process for crystal which
                             # takes into account the stray capacitance

    for i in range(number):
        data.append(comm.readline())

    for i in range(number):
        lines.append(b2s(data[i]))

    centre = int(lines[int(number / 2)].split()[0])
    return centre, lines
    
def extract_column(lines, number): #lines is a list of strings
    x = []
    for line in lines:
        if (number == 0):
            x.append(int(line.split()[number]))
        else:
            x.append(float(line.split()[number]))            
    return x
       
def phase_func(x, a, b):
    return arctan(a*(x - b))

def G_func(x, a, b, c):
    return (c / (1.0 + (a*(x-b))**2))

def plot_result(title, c): # 'c' is the centre frequency and is an integer
    os.system("gnuplot -c xtalplot_adm.gp " + title) # creates the graph and saves
    # 'xtal_adm.png' and 'xtal_adm_ph.png'
    os.system("feh xtal_adm.png&") # displays the graph
    os.system("feh xtal_adm_ph.png&") # displays the graph

# main part of the program here

def main():
    if len(sys.argv) == ARG_length:
        os.system("./calibrate_z_meter.py") # calibrates f and stray capacity
        time.sleep(0.5)
        centre = int(1.0e6*float(input('Centre frequency (MHz): ')))
        width = int(input('Width (Hz): '))
        number = int(input('Number of samples: '))
        initialize(centre, width, number)

        out_file_name = input('Type file name for data: ') # this will be a comma-separated file
        out_f = open(out_file_name, 'w')
        
        while True:
            n = int(input('Xtal # (negative to quit): '))
            if (n < 0): break
            c, lines = doscan_adm(centre, number) #does the scan in the z_meter

            frequency = extract_column(lines, 0)
            f0 = frequency[10]
            frequency[:] = [x - f0 for x in frequency]
            phase = extract_column(lines, 4)
            init_vals = [0.02, 0]
            popt, pcov = curve_fit(phase_func, frequency, phase, p0=init_vals)
            a = popt[0]
            b = popt[1]
            Q = -f0 * popt[0] / 2
            Q = 100 * round(Q / 100.0)
            f0 = f0 + popt[1]
            g = extract_column(lines, 1)
            init_vals = [popt[0],  popt[1], 0.1]
            popt, pcov = curve_fit(G_func, frequency, g, p0=init_vals)
            R = 1.0 / popt[2]
            R = round(R * 10.0) / 10.0
            f0 = round(f0)
            print("F: " + str(f0) + "     Q: " + str(int(Q)) + "  R: " + str(R))
            out_f.write(str(n) + ',' + str(f0) + ',' + str(Q) +',' + str(R) + '\r')
        out_f.close()
        comm.close()
    else:
        print("Incorrect number of parameters")
        print("Usage: ./fit_batch.py")
        
if __name__=="__main__":
    main()
    
