/* handle the encoder input
 */

#include <avr/interrupt.h>

#include "GPSReference.h"

// encoder connections
#define	PIN_A	2
#define	PIN_B	3
#define	PIN_S	4	// switch

// 2-bit gray-code sequence
#define	GRAY_A	0
#define GRAY_B	1
#define	GRAY_C	3
#define	GRAY_D	2

/* one interrupt handler for both encoder pins.
 * used for both pins because each do the same thing.
 * can not use an object method directly because of implied 'this' argument.
 * allow simultaneous DDS interrupts but not this one reentrant.
 */
static void myOnEncPinChange()
{
    enc->onEncPinChange();
}

/* construct an Encoder instance
 */
Encoder::Encoder ()
{
    // init pins with internal pullups
    pinMode (PIN_A, INPUT_PULLUP);
    pinMode (PIN_B, INPUT_PULLUP);
    pinMode (PIN_S, INPUT_PULLUP);

    // connect interrupt to each pin change
    attachInterrupt(digitalPinToInterrupt(PIN_A), myOnEncPinChange, CHANGE);
    attachInterrupt(digitalPinToInterrupt(PIN_B), myOnEncPinChange, CHANGE);

    // init state
    prev_a = digitalRead(PIN_A);
    prev_b = digitalRead(PIN_B);
    enc_history = (prev_a << 1) | prev_b;
    enc_value = 0;
    new_value_ready = false;
}

/* call any time to return the current value, return whether different than previous call
 */
bool Encoder::getValue (int32_t &n_v)
{
    bool nvr = new_value_ready;
    n_v = enc_value;
    new_value_ready = false;
    return (nvr);
}

/* return whether encoder switch has been engaged
 */
bool Encoder::switchPushed()
{
    if (!digitalRead (PIN_S)) {
	delay (200);	// crude debounce
	return (true);
    }
    return (false);
}

/* force the encoder count to the given value.
 */
void Encoder::setValue (int32_t n_v)
{
    enc_value = n_v;
}

/* update enc_value given direction and ms since last update.
 * faster rotations change the value more.
 */
void Encoder::gearBox (uint8_t cw, uint32_t dt)
{
    uint8_t neg = enc_value < 0;
    int8_t sgn_chg = (cw ^ neg ? 1 : -1);
    uint32_t abs_ev = neg ? -enc_value : enc_value;

    if (dt < 5) {
	abs_ev = abs_ev/100 * 100 + sgn_chg * 100;	// change 100s place
    } else if (dt < 20) {
	abs_ev = abs_ev/10 * 10 + sgn_chg * 10;		// change 10s place
    } else
	abs_ev += sgn_chg;				// change 1s place

    if (neg)
	enc_value = -abs_ev;
    else
	enc_value = abs_ev;

}

/* called when either encoder pin changes state.
 * if value has _reliably_ changed, set new_value_ready and update enc_value so
 * subsequent calls to getValue() will return the current value and whether it is new.
 * N.B. we only count full detent changes, ie, falling edges of pin a. This wastes 3/4 of the
 *      available encoder precision but feels more natural.
 * N.B. we implement a "gear box" so faster rotations change more significant digits of enc_value, this
 *      works better than a separate explicit scale change IMHO.
 * N.B. by using a history of Gray-code pin changes, we eliminate the need for explicit debouncing.
 */
void Encoder::onEncPinChange()
{
    // time now
    uint32_t now_ms = millis();

    // read both lines
    uint8_t a = digitalRead (PIN_A);
    uint8_t b = digitalRead (PIN_B);

    // skip unless one changed
    if (a != prev_a || b != prev_b) {

	// shift into history as a,b pair, keeping most recent 3 pairs:
	//    a[-2] b[-2] a[-1] b[-1] a b
	enc_history <<= 2;				// shift to make way for new pair
	enc_history &= 0x3F;				// discard oldest pair
	enc_history |= (a<<1);				// current a is upper bit of new pair
	enc_history |= (b);				// current b is lower bit of new pair
	// Serial.println (enc_history, HEX);

	// check for valid sequential change at full stops.
	// (uncomment other 6 if want full precision)
	switch (enc_history) {
	case ((GRAY_A<<4)|(GRAY_B<<2)|GRAY_C):		// rotation cw
	// case ((GRAY_B<<4)|(GRAY_C<<2)|GRAY_D):
	// case ((GRAY_C<<4)|(GRAY_D<<2)|GRAY_A):
	// case ((GRAY_D<<4)|(GRAY_A<<2)|GRAY_B):
	    gearBox (1, now_ms - a_last_ms);
	    a_last_ms = now_ms;
	    new_value_ready = true;
	    break;
	case ((GRAY_B<<4)|(GRAY_A<<2)|GRAY_D):		// rotation ccw
	// case ((GRAY_C<<4)|(GRAY_B<<2)|GRAY_A):
	// case ((GRAY_A<<4)|(GRAY_D<<2)|GRAY_C):
	// case ((GRAY_D<<4)|(GRAY_C<<2)|GRAY_B):
	    gearBox (0, now_ms - a_last_ms);
	    a_last_ms = now_ms;
	    new_value_ready = true;
	    break;
	default:
	    // ignore all other cases
	    break;
	}

	// persist
	prev_a = a;
	prev_b = b;
    }
}
