/*
	This code implements simple I/O via the MSP430F1232 USART
	The output is unbuffered, the input is buffered
*/

#if defined(__GNUC__)  &&  defined(__MSP430__)
	#include <legacymsp430.h>
#else
	#include "io430.h"
#endif
#include <string.h>
#include "uuart.h"

#define INBUFSIZE	30
struct {
	int	in;					// buffer index for character input
	int	out;
	char u0rctl;			// UART errors
	char overflow;
	char buf[INBUFSIZE];
} rxData;


void initUsart(int baudrate)
{
int br;
	memset(&rxData, 0, sizeof(rxData));
	P3DIR &= ~BIT5;				// Rx
	P3DIR |= BIT4;				// Tx
	P3SEL |= BIT4|BIT5;
	U0CTL |= SWRST;
	// No parity, 1 stop bit, 8-bit data, Listen disabled, UART mode, idle-line protocol, sw reset disabled
	U0CTL = CHAR;
	// SMCLK, start-edge feature disabled, next char is data
	U0TCTL = SSEL1|TXEPT;
	br = TARGET_FREQ/baudrate;
	U0BR0 = br & 0xff;
	U0BR1 = br >> 8;
	U0MCTL = 0;
	U0ME |=  UTXE0|URXE0;	// Enable USART0 transmit and receive (actually ME2 @ 0x05)
	U0CTL &= ~SWRST;
	U0IE |= URXIE0;			// Enable USART0 receive interrupt (actually IE2 @ 0x01)
}

#if defined(__GNUC__)  &&  defined(__MSP430__)
interrupt (UART0RX_VECTOR) usart0RcvIsr(void)
#else
#pragma vector=UART0RX_VECTOR
__interrupt void usart0RcvIsr(void)
#endif
{
int t;
char ch;
	t = (rxData.in + 1) % INBUFSIZE;
	if (t != rxData.out)
	{
		ch = U0RXBUF;
		rxData.buf[rxData.in] = ch;
		rxData.in = t;
		rxData.u0rctl |= U0RCTL;
	}
	else
		rxData.overflow = 1;
}

int rxBufEmpty(void)
{
	return rxData.in == rxData.out;
}

char getCh(void)
{
char ch;
	U0IE &= ~ URXIE0;
	if (rxData.out != rxData.in)
	{
		ch = rxData.buf[rxData.out];
		rxData.out = (rxData.out + 1) % INBUFSIZE;
	}
	else
		ch = 0;
	U0IE |= URXIE0;
	return ch;
}

int readLine(char *buf)
{
char ch;
int n=0;
	while (1)
	{
		ch = getCh();
		if (ch == 0)
			continue;
		*buf++ = ch;
		n++;
		if ((ch == '\r') | (ch == '\n'))
		{
			*buf = 0;	// Add string terminator
			break;
		}
	}
	return n;
}

int readLineTo(char *buf, int *currentTime, int timeOut)
{
/*
	*currentTime	-	current relative time (incremented periodically)
	*timeOut		-	time limit for read operation
*/
char ch;
int n=0;
	while (1)
	{
		if (*currentTime > timeOut)
			return -1;
		ch = getCh();
		if (ch == 0)
			continue;
		*buf++ = ch;
		n++;
		if ((ch == '\r') | (ch == '\n'))
		{
			*buf = 0;	// Add string terminator
			break;
		}
	}
	return n;
}

void sendMsg(char *msg, int length)
{
	// if length is not zero binary data is being transmitted.
	rxData.u0rctl = rxData.overflow = 0;
	if (length == 0)
		length = strlen(msg);
	if (length == 0)
		return;
	while (1)
	{
		if (U0TCTL & TXEPT)
		{
			U0TXBUF = *msg++;
			if (--length == 0)
				break;
		}
	}
	while (1)
		if (U0TCTL & TXEPT)	// Wait until last character has been transmitted
			break;
}

