/*
P1:
	0
	1
	2	testADC
	3

	4			}
	5			}	jtag
	6			}
	7			}
P2:
	0
	1
	2
	3
	4
	5
P3:
	0	ADCIN
	1
	2	OEPC
	3	OESIG
	4	TXOUT
	5	RXIN
	6
	7

Timer A
	CCTL0
	CCTL1
	CCTL2	DCO
*/

#include <msp430.h>
#include <legacymsp430.h>
#include <string.h>
#include "..\utils\uuart.h"
#include "..\utils\bin2hexS.h"
#include "..\utils\b2d.h"
#include "..\utils\str2bin.h"
#include "..\uart\delay.h"

#define OEPC	BIT2
#define OESIG	BIT3

#define DCO
#ifdef DCO
	#define DCODEBUG 0
	#include "..\uart\dco.h"
	#define DCOUPDATECOUNT	5	// Update every 5 minutes
	/*
	Update DCO once a minute for 5 minutes.
	Then update once every <DCOUPDATECOUNT> minutes.
	*/
	static struct {
		int	count;
		char startup;
		char updateEnable;
	} dco;
#endif

char msg[30];
interrupt (TIMERA1_VECTOR) Timer_A1 (void)
{
int taiv = TAIV;

	if (taiv == 4)			// Capture / compare 2
		processDCOUpdate();
	else if (taiv == 10)	// overflow
		;
}

static int sampleADC(void)
{
int adcv;
/*
	The ADC clock is SMCLK, 2 MHz, Tc= .5 usec, sample & hold time is 64 clks or 32 usec.
	The ADC channel source resistance is zero so Ts must be greater than (0+2e3)*7.625*20e-12 +2.5 usec = 2.8 usec.
*/

	IE1 &= ~WDTIE;					// Disable WDT interrupt
	ADC10CTL0 |= ENC|ADC10SC;		// Start
	while (ADC10CTL1 & ADC10BUSY)
		;
	adcv = ADC10MEM;
	ADC10CTL0 &= ~ENC;
	IE1 |= WDTIE;
	return adcv;
}

static unsigned int measureADC()
{
// returns the average of 16 samples
int v, k;
	v = 0;
	for (k=0; k<16; k++)
		v += sampleADC();
	v += 8;			// round up
	v >>= 4;
	return v;
}

void testADC(int n)
{
int k, v;
	for (k=0; k < n; k++)
	{
		P1OUT |= BIT2;
		v = measureADC();
		P1OUT &= ~BIT2;
		bin2dec((char*)&v, 2, UNSIGNED, msg, APPENDCRLF);
		sendMsg(msg, 0);
	}
}

#define WDTCOUNT	8
static int oneSecondCounter = WDTCOUNT;
static int oneMinuteCounter = 60;
static int rtimeSeconds;
interrupt (WDT_VECTOR) watchdog_timer(void)
{	// 8 Hz

	oneSecondCounter--;
	if (oneSecondCounter == 0)
	{	// Here once a second
		rtimeSeconds += 1;
		oneSecondCounter = WDTCOUNT;
		oneMinuteCounter--;
		if (oneMinuteCounter == 0)
		{
			oneMinuteCounter = 60;
			//------------------------------------------------------------------------
			if (dco.startup)
			{
				dco.updateEnable = 1;
				dco.count--;
				if (dco.count == 0)
				{
					dco.startup = 0;
					dco.count = DCOUPDATECOUNT;
				}
			}
			else
			{
				dco.count--;
				if (dco.count == 0)
				{
					dco.count = DCOUPDATECOUNT;
					dco.updateEnable = 1;
				}
			}
		} 	// if (oneMinuteCounter == 0)
	}
}

void backgroundProcessing()
{
	if (dco.updateEnable)
	{
		startDCOUpdate();
		dco.updateEnable = 0;
	}
}
/*
HP E4400 commands (length, text)
5 *RST
24 SYST:COMM:SER:BAUD 9600
13 OUTP:STAT ON
16 FREQ 7100000 Hz
16 FREQ 7200000 Hz
16 FREQ 7300000 Hz
*/

void sendToHp(char *cmd)
{
	P3OUT |= OEPC;		// Disconnect PC port
	P3OUT &= ~OESIG;	// Connect signal generator port
	sendMsg(cmd, 0);
	P3OUT |= OESIG;		// Disconnect signal generator
	P3OUT &= ~OEPC;		// Connect to PC
}

extern int __bss_start, __bss_size, __stack;
int main (void)
{
#define CR 13
int i;
char ch;
char *p;
static const char *startMsg = "Starting\n\r";
static const char *hpMsg = "*RST\n";

	WDTCTL = WDTPW + WDTHOLD;	// Stop watchdog timer
	p = (char*)(&__bss_start + __bss_size);
	while (p < (char*)&__stack)
		*p++ = 0;				// clear the stack
	P1DIR = P2DIR = P3DIR = 0xff;		// All outputs
	P1OUT = P2OUT = P3OUT = 0;

	P3OUT |= OESIG;		// Disconnect signal generator serial output port

	initUsart(9600);	// initialize MSP430F1232 USART

	// Initialization for DCO
	TACTL = TASSEL_2|MC_2|TAIE;		// SMCLK, continuous mode, enable overflow interrupt
	CCTL2 = CM_1 | CCIS_1 | CAP; 	// Capture on rising edge, CCI2B input, Capture mode, Interrupt enabled
	BCSCTL1 = XT2OFF | DIVA_3;		// ACLK divider is 8 ==> ACLK = 4096
	//DCOCTL = 0xa8;		// }
	//BCSCTL1 |= 0xb7;		// } 4 MHz values
	DCOCTL = 0x55;		// }
	BCSCTL1 |= 0xb6;	// } 2 MHz values

	// WDT initialization
	// ACLK is 4096 Hz. WD is triggred by ACLK.
	// WDTIS is 2 which causes an interrupt every 512 counts.
	// Thus WDT interrupts occur every 512 (counts) / 4096 (c/sec) = 125 ms (8 Hz)
	//                    pw       Interval mode      ACLK          div by 512
	WDTCTL = WDTPW | WDTTMSEL | WDTSSEL | WDTIS1;
	IE1 |= WDTIE;	// Enable WDT interrupts
	//TACTL |= TAIE;	// Enable Timer A overflow interrupt.

	// Initialize ADC
	// VR+=VCC, VR-=VSS,  ~50 ksps, no interrupt, sample & hold time is 64 adc clocks
	ADC10CTL0 = ADC10SHT_3|ADC10SR|ADC10ON;
	ADC10CTL0 |= REFON|REF2_5V|SREF_1;	// Enable internal 2.5 volt reference
	ADC10CTL1 = INCH_5|ADC10SSEL_3;		// ADC clock is SMCLK,
	ADC10AE = 1<<5;						// Input on A5 (P3.0)
	P3DIR &= ~BIT0;
	P3SEL |= BIT0;

	// Wait ~ 2 sec. for the crystal oscillator to stabilize.
	for (i = 0; i < 20; i++)
		pause(0);

	eint();			// Enable interrupts
	#ifdef DCO
	dco.startup = 1;
	dco.updateEnable = 1;
	dco.count = DCOUPDATECOUNT;
	startDCOUpdate();
	while (1)
	{
		if (dcoStatus() & DCO_STABLE)
			break;
		pause(0);		// Wait a bit
	}
	#endif
	sendMsg((char*)startMsg, 0);
	delayMilliseconds(50);
	sendToHp((char*)hpMsg);
	while (1)
	{
		backgroundProcessing();
		ch = getCh();
		switch(ch)
		{
			case 0:
				break;
			case 'A':
				readLine(msg);
				sendToHp(msg);
				break;
			case 'B':
			{
				int v;
				v = measureADC();
				bin2dec((char*)&v, 2, UNSIGNED, msg, APPENDCRLF);
				sendMsg(msg, 0);
				break;
			}
			case 'a':
				sendMsg("Test ADC: ", 0);
				readLine(msg);
				i = str2bin(msg, 0);
				testADC(i);
				break;
			case 'b':
				break;
			case 'c':
				sendMsg("HP: ", 0);
				readLine(msg);
				strcpy(&msg[strlen(msg)], "\n");
				sendToHp(msg);
				break;
			case 'm':
				sendMsg("Waiting: ", 0);
				i = readLine(msg);
				sendMsg(msg, 0);
				break;
			case 'n':
				sendMsg("Now is the time\n\r", 0);
				break;
			case 'f':
				#if DCODEBUG == 1
				displayDCOFrequency(msg);
				#endif
				break;
			default:;
		}
	} //end while
}
