/* 
	afrecv.c - transmit data from PortAudio to network.  Use with afxmit.c.
	version: 0.8 (Linux / Windows)
	Copyright (c) 2012-2013 Martin S Ewing

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Contact author: ewing@alum.mit.edu, 
		28 Wood Rd, Branford, CT, USA 06405

	Portions of code incorporated from examples in PortAudio, Speex, and XAudio2 documentation:
		www.portaudio.com
		www.speex.org
*/

#include "af.h"

#ifdef WIN32
#include "xgetopt.h"
#endif

/*  Audio sampling & network transmission scheme:
 *
 *  1. Sampling: 16 bit mono, 8 kHz
 *  2. Header precedes each audio block with byte count to follow, etc.
 *  3. Null codec: 256 samples (512 B) sent per 32 msec.
 *  4. Speex codec: input block 160 B, 4x buffered (640 B = 80 msec)
 *     output block is much smaller. [SPEEX NOT AVAILABLE IN WINDOWS.]
 *  5. afxmit sends at rate determined by soundcard input clock.
 *     No compensation for fast or slow clock.
 *  6. UDP version allows for framing and control of bit slip, less bandwidth.
 */

/* To do:
 * verify correct Linux compilation/operation
 * add Opus codec
 * reorganize VS 2010 C++ "solution"?
 * attempt VS 2010 C++ and/or VS 2012 C++ under Win7, 32/64?
 * better "deployment" scheme?
 * signalling from Python to start/stop link?
 */

/* Globals */

struct sockaddr_in server_addr, rx_addr;
int	net_sock=0;
int	punt_count=0, skip_count=0;
int	retry_count;
int	debug_print=FALSE;

/* Codec defaults (global) */
int codec_buffer_size;
int codec_chunk_size;
int	framesPerBuffer;
int	ncodec=CODEC_NULL;	/* default to null codec */
int	spx_buffering=4;	/* 4 buffers, 80 msec default */
int	spx_enhance=FALSE;	/* perceptual enhancer */
int	spx_quality=8;		/* quality, affects output bit rate and cpu time */
int	spx_complexity=3;	/* complexity, affects cpu time */

#ifdef WIN32
#pragma comment(lib, "Ws2_32.lib")

//--------------------------------------------------------------------------------------
// Callback structure
//--------------------------------------------------------------------------------------
struct StreamingVoiceContext : public IXAudio2VoiceCallback
{
    STDMETHOD_( void, OnVoiceProcessingPassStart )( UINT32 )
    {
    }
    STDMETHOD_( void, OnVoiceProcessingPassEnd )()
    {
    }
    STDMETHOD_( void, OnStreamEnd )()
    {
    }
    STDMETHOD_( void, OnBufferStart )( void* )
    {
    }
    STDMETHOD_( void, OnBufferEnd )( void* )
    {
        SetEvent( hBufferEndEvent );
    }
    STDMETHOD_( void, OnLoopEnd )( void* )
    {
    }
    STDMETHOD_( void, OnVoiceError )( void*, HRESULT )
    {
    }

    HANDLE hBufferEndEvent;

    StreamingVoiceContext() : hBufferEndEvent( CreateEvent( NULL, FALSE, FALSE, NULL ) )
    {
    } // c.f. http://www.cprogramming.com/tutorial/initialization-lists-c++.html (mse)
    virtual ~StreamingVoiceContext()
    {
        CloseHandle( hBufferEndEvent );
    }
};

// UDP packet receive function
int recv_win(int sock, char **out)
{
	static char	rdbuffer[NET_BUF_SIZE];		// network input
	static char	outputBuffer[NET_BUF_SIZE];	// output
	socklen_t	rxaddrlen;
	extern int	punt_count, retry_count;
	static int	first_success=FALSE;

	// Get a received audio data buffer from net.
	rxaddrlen = sizeof(rx_addr);
	int resp = recvfrom(net_sock, rdbuffer, NET_BUF_SIZE, NULL,
			(struct sockaddr *)&rx_addr, &rxaddrlen);
	if (resp == SOCKET_ERROR)
		switch (WSAGetLastError())
		{
			case (WSAEWOULDBLOCK):
				// Network data is not ready.
				resp = 0;
				break;
			default:
				perror("Reading audio data");
				exit (EXIT_FAILURE);
		}
	if (resp == 0)	// Nothing new available from net, send silence.
		memset(outputBuffer, 0, NET_BUF_SIZE);
	else
		memcpy(outputBuffer, rdbuffer, 2*framesPerBuffer);
	*out = outputBuffer;
	return (resp);			// return byte count
}

#else
/* The receive callback routine is entered from PortAudio when new audio data is required,
   every 32 msec for 8 kHz sampling and 256 samples per buffer -- for the null codec.  
   If a UDP packet from afxmit is ready for recvfrom, all is well.  If there is none, we 
   will give PortAudio a buffer full of zeroes (silence).  If the afxmit soundcard clock 
   is a little faster than 8 kHz, we will sometimes get extra packets.  These will be 
   discarded if there is no UDP recvfrom for them, before another packet is sent.
   For the Speex codec, the buffering is 160 samples times 4, and the callback is called every
   80 msec.
*/
static int recvCallback( const void *inputBuffer, void *outputBuffer,
                         unsigned long framesPerBuffer,
                         const PaStreamCallbackTimeInfo* timeInfo,
                         PaStreamCallbackFlags statusFlags,
                         void *userData )
{
	char		rdbuffer[NET_BUF_SIZE];/* network input */
	SAMPLE 		*rptr, *wptr;
	socklen_t	rxaddrlen;
	extern int	punt_count, retry_count;
	int		punt_buffer;

	static int	frst=TRUE;
	static SpeexBits	bits;
	static void	*dec_state;
	SAMPLE		output_frame[NET_BUF_SIZE];
	static int	output_frame_size;
	int		spbuflen;
	char		*inptr;
	static int	first_success=FALSE;

	(void) inputBuffer; /* Prevent unused variable warnings. */
	(void) timeInfo;
	(void) statusFlags;
	(void) userData;

	/* Get a received audio data buffer from net. Note nonblocking read. */
	if (debug_print) { printf("."); fflush(stdout); }
	punt_buffer=FALSE;
	rxaddrlen = sizeof(rx_addr);
	/* Note that actual received packet length will be different for different
	   codec choices and codec parameters. Also, since this is UDP, we could get
	   packets from anywhere of any size!  Worry? */
	if ( (recvfrom(net_sock, rdbuffer, NET_BUF_SIZE, MSG_DONTWAIT,
			(struct sockaddr *)&rx_addr, &rxaddrlen) ) == -1)
	{
		switch (errno)
		{
			case EWOULDBLOCK:
				/* There is no data ready - we are starved. We punt this
				   portaudio buffer - write silence.  Hopefully we'll
				   have data next time through. */
				punt_buffer = TRUE;
				punt_count++;
				if (debug_print) {printf("P%03d", punt_count); fflush(stdout);}
				/* Check if good data has started and if we've missed too many
				   packets in a row. Normal to have many misses at first.  */
				if ((++retry_count > RETRY_LIMIT) && first_success)
				{
					perror("Too many missed packets");
					return paAbort;
				}
				break;
			default:
				perror("Reading audio data");
				return paAbort;
		}
	}
	else { 				// we received a packet ok.
		first_success=TRUE; 	// make note when we start getting good data.
		retry_count = 0;
	}
	
	/* We have got the current af data. */
	
	if (punt_buffer)	// Nothing new available from net, send silence.
		memset(outputBuffer, 0, 2*framesPerBuffer);
	else {
		switch (ncodec) 	// sort out types of data
		{
			case CODEC_NULL:
				/* normally -> put audio samples into output stream */
				memcpy(outputBuffer, rdbuffer, 2*framesPerBuffer);
				break;
			case CODEC_SPEEX:
				if (frst)
				{	/* Initialize Speex decoder */
					frst = FALSE;
					speex_bits_init(&bits);
					dec_state = speex_decoder_init(&speex_nb_mode);
					speex_decoder_ctl(dec_state, SPEEX_GET_FRAME_SIZE,
						&output_frame_size);
					assert(output_frame_size == 160);
					speex_decoder_ctl(dec_state, SPEEX_SET_ENH, &spx_enhance);
				}
				if (punt_buffer)
					memset(outputBuffer, 0, 2*framesPerBuffer);
				else { /* normally -> put audio samples into output stream */
					/* rdbuffer will contain multiple speex packets, e.g. 4 */
					inptr = (char *)rdbuffer;
					wptr = outputBuffer; /* note s.b. double size */
					int i;
					for (i=0; i < spx_buffering; i++)
					{
						sscanf(inptr, "%4d", &spbuflen); /* 1st 4 bytes = e.g. 38 */
						inptr += 4;
						speex_bits_read_from(&bits, inptr, spbuflen);
						speex_decode_int(dec_state, &bits, output_frame);
						rptr = output_frame;
						int j;
						/* nb: memcpy doesn't work here because of byte ordering? */
						for (j=0; j < output_frame_size; j++) *wptr++ = *rptr++;
						inptr += spbuflen;
					}
				}
				break;
			default:
				perror("Invalid codec 1");
				return paAbort;
			}	// end switch
		}	// end else
	return paContinue;
}	// end RecvCallback
struct sigaction hup_action;
#endif
/* Catch SIGHUP signal to terminate incoming stream before quitting main program. */
int my_hup_handler(int signal)
{
	// Not checking signal - assume any interrupt is like CTRL-C.
	// Terminate stream, then die.
	extern int	net_sock;
	char		codec_command[L_CMD];
	memset(codec_command, 0, L_CMD);
	snprintf(codec_command, L_CMD, "%s", CMD_STOP);
	if ((sendto(net_sock, codec_command, L_CMD, 0,
			(const struct sockaddr *)&server_addr, sizeof(server_addr))) == -1)
	{
		perr("kill_stop_command"); 
	}
	printf("\n*** Received SIGHUP signal: afxmit paused, quitting afrecv. ***\n");
	exit(EXIT_SUCCESS);
}
#ifdef WIN32
BOOL w_hup_handler(DWORD signal) // version for Win32 (could be merged w/ Linux?)
{
	my_hup_handler(signal);
	return (TRUE);				// VS requires, though control never reaches here.
}
#else
void hup_handler(int signal)	// version for Linux
{
	my_hup_handler(signal);
}
#endif

// ******REAL ENTRY**********************************************

int main(int argc, char *argv[])
{
#ifdef WIN32
	// may need adjustments!
	HRESULT hr;
	static int obuf_count = 0;
	char osver[32] = "Win32\0";
	extern const SpeexMode speex_nb_mode;
#else
	PaStreamParameters	outputParameters;
	PaStream		*stream;
	PaError			err = paNoError;
	char osver[32] = "Linux\0";
#endif
	struct hostent *host;
	int			opt;
	int			port = PORT_DEFAULT; 			/* default port */
	extern int 	optind, opterr, optopt;
	char		codec_command[L_CMD];
	extern int	codec_chunk_size, codec_buffer_size;	/* samples */
	extern int	framesPerBuffer;						// sames as chunk/buffer size?
	extern int	spx_buffering, spx_enhance, spx_quality, spx_complexity;
	extern int	retry_count;
	static int	run_forever = FALSE;
	int			kill_stream = FALSE;
	static SpeexBits	bits;
	static void	*dec_state;
	SAMPLE		output_frame[NET_BUF_SIZE];
	static int	output_frame_size;

	printf("afrecv build %s %s os=%s\n", __TIMESTAMP__, __DATE__, osver);

	/* afrecv [-c codec_number] [-b buffer factor] [-e enhance 1/0] [-q spx_quality] 
			[-x spx_complexity][-p port] [-k kill afxmit & term.]  host_name -s -> run forever */
	while ((opt = getopt(argc, argv, "skc:p:b:e:q:x:")) != -1) 
	{
		switch(opt) 
		{
			case 'k':	/* set up AFXMIT stream kill (WIN, prob ok for Linux) */
				kill_stream = TRUE;
				break;
			case 'p':	/* port */
				port = atoi(optarg);
				break;
			case 'c':	/* codec number */
				ncodec = atoi(optarg);
				break;
			case 'b':	/* speex buffering factor */
				spx_buffering = atoi(optarg);
				break;
			case 'e':	/* Perceptual Enhancer */
				spx_enhance = atoi(optarg)==0 ? FALSE : TRUE;
				break;
			case 'q':	/* speex quality 0:9 */
				spx_quality = atoi(optarg);
				break;
			case 'x':	/* speex complexity 1:10 */
				spx_complexity = atoi(optarg);
				break;
			case 's':	/* run forever, no stop on kb interrupt */
				run_forever = TRUE;
				break;
			default:	/* error */
				fprintf(stderr, 
					"usage: %s [-k] [-t] -s secs -c codec_no -p port host -e {1|0} percept."
					" -b buff -q qual -x complex.\n", argv[0]);
				exit(EXIT_FAILURE);
		}
	}
	/* process argv[optind:] = host name or addr. */
	if (argv[optind] == 0) 
	{
		printf("Host name is required.\n");
		exit(EXIT_FAILURE);
	}
	printf("Using codec %d, b %d, e %d, q %d, x %d, host %s\n", ncodec, spx_buffering,
		spx_enhance, spx_quality, spx_complexity, argv[optind]);
	char *hn = argv[optind];
#ifdef WIN32
	// Windows: negotiate a WinSock interface (!)
	WORD wVersionRequested = MAKEWORD(2, 2); // try for v 2.2
	WSADATA wsaData;
	int werr = WSAStartup(wVersionRequested, &wsaData);
	if (werr != 0)
	{
		printf("WSAStartup failed with error: %d\n", werr);
		return EXIT_FAILURE;
	}
	if (LOBYTE(wsaData.wVersion) != 2 ||HIBYTE(wsaData.wVersion) != 2) 
	{
		printf("Cannot find Winsock.dll v 2.2\n");
		WSACleanup();
		return EXIT_FAILURE;
	}
	// Windows: set ctrl-c handler
	// The handler will be invoked if user types CTRL+C or clicks console close box,
	// but, alas, not if we are killed by taskkill /f.  In that case, we have an
	// immediate kill, and the AFXMIT stream is left running!
	if ( !SetConsoleCtrlHandler((PHANDLER_ROUTINE) w_hup_handler, TRUE) )
		perr("Can't set ctrl-c handler");
#endif
	host = gethostbyname(argv[optind]);
	if (host==NULL)
	{
#ifdef WIN32
		char wmsgbuf[50];
		_snprintf_s(wmsgbuf, sizeof(wmsgbuf), _TRUNCATE, 
			"gethostbyname, error = %d", WSAGetLastError() );
		perr(wmsgbuf);
#else
		perr("gethostbyname");
#endif
	}
	/* prepare for chosen codec */
				//char sendBuf[ XA2_MAX_BUFF_CT * (CODEC_NULL_CHUNK * SAMPLE_WIDTH) ];
			//char *sendBufP = sendBuf;

	switch(ncodec) 
	{
		case CODEC_NULL:	/* null codec */
			codec_chunk_size = codec_buffer_size = 256; // samples = bytes/2
			framesPerBuffer = codec_chunk_size;			// historic from PortAudio
			break;

		case CODEC_SPEEX:	/* speex */
			/* NB: 1 network audio buffer = multiple Speex data buffers */
			codec_chunk_size = 160; /* frames = 16 bits */
			framesPerBuffer = codec_chunk_size;
			codec_buffer_size = spx_buffering * codec_chunk_size;
			speex_bits_init(&bits);
			dec_state = speex_decoder_init(&speex_nb_mode);
			speex_decoder_ctl(dec_state, SPEEX_GET_FRAME_SIZE, &output_frame_size);
			assert(output_frame_size == 160);
			//printf("output frame size = %d\n", output_frame_size);
			speex_decoder_ctl(dec_state, SPEEX_SET_ENH, &spx_enhance);
			break;

		default:
			fprintf(stderr, "Unknown codec %d\n", ncodec);
			exit(EXIT_FAILURE);
	}

	/* Initialize network */
#ifdef WIN32
	// NB: In Windows, this makes an overlapped socket by default?
	net_sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (net_sock == INVALID_SOCKET)
	{
		char wmsgbuf[50];
		_snprintf_s(wmsgbuf, sizeof(wmsgbuf), _TRUNCATE, 
			"Invalid socket, error = %d", WSAGetLastError() );
		perr(wmsgbuf);
	}
	// Set socket non-blocking
	DWORD bytesrtnd, FIONinbuffer=1;
	hr = WSAIoctl(net_sock,
				FIONBIO,				// IOCTL code (block/non-block)
				&FIONinbuffer,			// Input buffer: Set non-blocking, non-zero
				sizeof(FIONinbuffer),	// size of input buffer ??
				NULL,					// output buffer (out) ??
				0,						// size of output buffer
				&bytesrtnd,				// bytes returned
				NULL,					// lpOverlapped
				NULL					// lpCompletionRoutine
				);
	if (hr == SOCKET_ERROR) 
	{
		char wmsgbuf[50];
		_snprintf_s(wmsgbuf, sizeof(wmsgbuf), _TRUNCATE, 
			"Can't set non-blocking, error = %d", WSAGetLastError() );
		perr(wmsgbuf);
	}

#else
	net_sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (net_sock == -1)
		perr("Socket");
	// need more Linux error checking?
#endif
	server_addr.sin_family = AF_INET;
	server_addr.sin_port = htons(port);
	server_addr.sin_addr = *((struct in_addr *)host->h_addr);
        memset(&(server_addr.sin_zero), 0, 8); 
	// Special case: -k switch means we are here only to kill
	// the current AFXMIT stream.  (Since killtask /f won't do it.)
	if (kill_stream)
		my_hup_handler(0);			// and exit from there. 
	/* Initialize audio output */
#ifdef WIN32
	// Windows xaudio2 initialize...
    CoInitializeEx( NULL, COINIT_MULTITHREADED );
    IXAudio2* pXAudio2 = NULL;
    UINT32 flags = 0;
#ifdef _DEBUG
    flags |= XAUDIO2_DEBUG_ENGINE;
#endif
    if( FAILED( hr = XAudio2Create( &pXAudio2, flags ) ) )
    {
        printf( "Failed XAudio2 init: %#X\n", hr );
        CoUninitialize();
        return EXIT_FAILURE;
    }
    IXAudio2MasteringVoice* pMasteringVoice = NULL;
    if( FAILED( hr = pXAudio2->CreateMasteringVoice( &pMasteringVoice ) ) )
    {
        printf( "Failed creating voice: %#X\n", hr );
        SAFE_RELEASE( pXAudio2 );
        CoUninitialize();
        return EXIT_FAILURE;
    }
#else
	// Linux Portaudio initialize...
	err = Pa_Initialize();
	if( err != paNoError ) goto done;

	outputParameters.device = Pa_GetDefaultOutputDevice(); /* default output device */
	if (outputParameters.device == paNoDevice) 
	{
		fprintf(stderr,"Error: No default output device.\n");
		goto done;
	}
	outputParameters.channelCount = NUM_CHANNELS; 		/* mono */
	outputParameters.sampleFormat =  PA_SAMPLE_TYPE;
	outputParameters.suggestedLatency = 
		Pa_GetDeviceInfo( outputParameters.device )->defaultLowOutputLatency;
	outputParameters.hostApiSpecificStreamInfo = NULL;
	/* Set up SIGHUP handler to pause remote server when we are terminated. */
	hup_action.sa_handler = hup_handler;
	if ( sigaction(SIGHUP, &hup_action, NULL) < 0 )
		perr("SIGHUP handler");
#endif

	/* Send commands to server */
	/* ***CODEC*** */
	memset(codec_command, 0, L_CMD);
	snprintf(codec_command, L_CMD, "%s %d %d %d %d", CMD_CODEC, ncodec,
		spx_buffering, spx_quality, spx_complexity);
	printf("Sending: %s\n", codec_command);
	if ( sendto(net_sock, codec_command, L_CMD, 0,
			(const struct sockaddr *)&server_addr, sizeof(server_addr)) == -1)
		perr("codec_command");

	/* ***START*** */
	memset(codec_command, 0, L_CMD);
	snprintf(codec_command, L_CMD, "%s", CMD_START);
	printf("Sending: %s\n", codec_command);
	if ( sendto(net_sock, codec_command, L_CMD, 0,
			(const struct sockaddr *)&server_addr, sizeof(server_addr)) == -1)
		perr("start_command");
	printf("\n=== receiving. ===\n"); fflush(stdout);
	
#ifdef WIN32
	// Main loop to receive a packet from the network and send it to xaudio
	// For first XA2_MAX_BUFF_CT packets, just SubmitSourceBuffer.
	// Thereafter, wait for a buffer to finish, then receive a packet and
	// submit.  We assume UDP will provide new data.  If not ready, a
	// null buffer will be submitted (silence).  There is also possibility
	// of a missing packet.  In that case, output will jitter.  There is
	// nothing to be gained by further input buffering.  Note, however,
	// that Windows may silently be buffering incoming UDP packets.  We
	// don't know whether that is a concern.

	WAVEFORMATEX wfxs, *wfx = &wfxs;
	wfxs.wFormatTag =		WAVE_FORMAT_PCM;	// plain samples
	wfxs.nChannels =		1;					// mono
	wfxs.nSamplesPerSec =	8000;				// sample rate
	wfxs.nAvgBytesPerSec =	16000;				// bytes/sec
	wfxs.nBlockAlign =		2;					// double byte block = 1 sample
	wfxs.wBitsPerSample =	16;					// sample width
	wfxs.cbSize =			0;					// extra format info
    //
    // Create an XAudio2 voice to stream this wave
    //
    StreamingVoiceContext voiceContext;
    IXAudio2SourceVoice* pSourceVoice;
    if( FAILED( hr = pXAudio2->CreateSourceVoice( &pSourceVoice, wfx, 0, 1.0f, &voiceContext ) ) )
    {
        printf("\nError %#X creating voice\n", hr);
        exit(EXIT_FAILURE);
    }
    pSourceVoice->Start( 0, 0 );
	// Note we need a cyclical buffer, because sound data must be held
	// unchanged in the buffer until it has been transmitted by xaudio.
	int sendBufL = codec_chunk_size * SAMPLE_WIDTH;
	char *sendBufBase = (char *) malloc( XA2_MAX_BUFF_CT * sendBufL);
	char *sendBufP = sendBufBase;

	int packetCount = 0;
	int bufferPhase = 0;		// packet no. modulo number of buffers
	char *recvBuf;
	int recvCount, nullpktcount = 0;

	XAUDIO2_VOICE_STATE state;
	XAUDIO2_BUFFER xbuf = {0};
			sendBufP = sendBufBase;		// We will start filling a new send buffer.
	for(;;)								// ********* MAIN LOOP **********
	{
		if (packetCount < XA2_MAX_BUFF_CT)
		{	// Get initial packets just by looping until ready.
			do {	// Loop until we get non-null packet.
				recvCount = recv_win(net_sock, &recvBuf);
				memset(recvBuf, 0, NET_BUF_SIZE);	// Convert it to nulls, minimizing initial transient. (DC bias?)
				if (recvCount == 0) Sleep(30);		// 30 msec
			} while (recvCount == 0);
			Sleep(50);		// Need a little time before WaitForSingleObject below, else we get spurious event. (?)
			packetCount++;
		}
		else
		{	// Later packets must wait for a buffer to become ready.
			for (;;)
			{	// Wait for output buffer available.
				pSourceVoice->GetState( &state );
				if( state.BuffersQueued < XA2_MAX_BUFF_CT - 1 )
						break;
				WaitForSingleObject ( voiceContext.hBufferEndEvent, INFINITE );
			}
			// Get another network packet and send to XAudio or Speex decoder
			// If data not here yet, we get nulls.  Spin for non-null return.
			int recvCount=0;
			do {
				recvCount = recv_win(net_sock, &recvBuf);
				Sleep(5);
				//if (recvCount == 0)
				//{	// If we got zero, maybe should stuff buffer with nulls and send?
				//	int ll = codec_chunk_size * SAMPLE_WIDTH;
				//	if (ncodec == CODEC_SPEEX) ll *= spx_buffering;
				//	memset(recvBuf, 0, ll);						// 512 or 640 b  CHECK?
				//	recvCount = ll;
				//	printf("received null packet %4d\n", ++nullpktcount);
				//}
			} while(recvCount == 0);

			if (ncodec == CODEC_SPEEX)
			{	// Typ. input buffer will have spx_buffering Speex packets @ 42 bytes (4 header + 38 data)
				char *inptr = recvBuf;
				SAMPLE *wptr = (SAMPLE *)sendBufP;		// Working with new send buffer (16 bit items)
				for( static int i=0; i < spx_buffering; i++)
				{
					int  spbuflen=0;
					sscanf_s(inptr, "%4d", &spbuflen);	// 1st 4 bytes are length of encoded data (38)
					inptr += 4;							// skip over header, take next 38 bytes as compressed audio
					speex_bits_read_from(&bits, inptr, spbuflen);
					speex_decode_int(dec_state, &bits, output_frame);
					SAMPLE * rptr = output_frame;
					for (int j=0; j< output_frame_size; j++)		// move 160 16-bit samples
						*(wptr++) = *(rptr++);
					inptr += spbuflen;
				}
				// Data have been stuffed into sendBuf (SPEEX CODEC).  Fix up the byte count.
				recvCount = (char *)wptr - (char *)sendBufP;		 //e.g. 1280
			}
		}
		//printf("submit\n");
		// submit this chunk to Xaudio
		// Place audio data chunk into cyclical sendBuf (NULL CODEC)
		if (ncodec == CODEC_NULL) 
			memcpy_s(sendBufP, sendBufL, recvBuf, recvCount);
		xbuf.AudioBytes = recvCount;
		xbuf.pAudioData = ((BYTE *) sendBufP) ;
		printf("recvCount=%d, audiodata*=%x\n", recvCount, sendBufP);
		pSourceVoice->SubmitSourceBuffer( &xbuf );

		// Rotate to new buffer element.
		bufferPhase++;
		sendBufP += recvCount;
		if (bufferPhase >= XA2_MAX_BUFF_CT)
		{
			bufferPhase = 0;
			sendBufP = sendBufBase;
		}
	}	// End main loop [NEED A WAY TO TERMINATE!]

	// Unfortunately, it appears the "normal" exit will be via
	// hard termination, which will not allow clean-up here.
	// (There is no Windows equivalent to SIGHUP, apparently.)
	// We can call this program with the -k switch, which
	// will stop the remote AFXMIT stream, among other things.
	// Better(?): A Python to C++ signal passing scheme.


// Graceful termination of Voice (prob. not needed)
//	memset(sendBuf,0,10);	// token null buffer
//	xbuf.AudioBytes = 10;
//	xbuf.pAudioData = (byte*)sendBuf;
//	xbuf.Flags = XAUDIO2_END_OF_STREAM;
//	pSourceVoice->SubmitSourceBuffer( &xbuf );
	for(;;)
	{
		pSourceVoice->GetState( &state );
		if( !state.BuffersQueued )
			break;
		printf (".");
		WaitForSingleObject( voiceContext.hBufferEndEvent, INFINITE );
	}
	int rc = pSourceVoice->Stop( 0 );
	pSourceVoice->DestroyVoice();
	free(sendBufBase);
#else
	/* Set up Portaudio output */
	retry_count = 0;
	err = Pa_OpenStream(
		&stream,
		NULL, /* no input */
		&outputParameters,
		SAMPLE_RATE,
		codec_buffer_size,	/* e.g., buffering_no * 160 */
		paClipOff, 
		recvCallback,
		NULL );
	if( err != paNoError ) 
	{
		perror("Pa_openstream");
		goto done;
	}
	/* Launch audio input if all checks out well. */
	if( stream )
	{
		err = Pa_StartStream( stream );
		if( err != paNoError )
		{
			perror("Pa_startstream");
			goto done;
		}

		/* Hang here, waiting for user key to kill receive session. */
		/* UNLESS run_forever flag is set -- then, sleep & loop */
		if (!run_forever)
		{
			printf("Press enter to quit...");
			getchar();
		}
		else 
		{	//sleep, loop forever
			for(;;) sleep(1);	// NB using Windows Sleep fn (af.h)
		}
#endif		
		/* ***STOP*** */
		memset(codec_command, 0, L_CMD);
		snprintf(codec_command, L_CMD, "%s", CMD_STOP);
		printf("Sending: %s\n", codec_command);
		if ((sendto(net_sock, codec_command, L_CMD, 0,
				(const struct sockaddr *)&server_addr, sizeof(server_addr))) == -1)
		{
			perror("stop_command"); 
			goto done;
		}
#ifndef WIN32
		err = Pa_CloseStream( stream );
		goto done;
	}
#endif

done:
	close(net_sock);
#ifdef WIN32
	pMasteringVoice->DestroyVoice();
	SAFE_RELEASE( pXAudio2 );
	CoUninitialize();
	WSACleanup();
#else
	Pa_Terminate();
#endif
	return(EXIT_SUCCESS);
}