#!/usr/bin/env python
# jedit info = :noTabs=true:tabSize=4:

# --------------------------------------------------------------------------- #
# SMETER displays a meter "needle" against a fixed image background.
# 1/14/2013
# adapted by Martin Ewing, AA6E, (with many functions disabled) from
# SPEEDMETER Control wxPython IMPLEMENTATION
# (Much clean-up is still needed!)
#
# Python Code By:
#
# Andrea Gavana, @ 25 Sep 2005
# Latest Revision: 14 Apr 2010, 12.00 GMT
#
import wx
import wx.lib.colourdb
import wx.lib.fancytext as fancytext
from math import pi, sin, cos, log, sqrt, atan2

#----------------------------------------------------------------------
# DC Drawing Options
#----------------------------------------------------------------------
# SM_NORMAL_DC Uses The Normal wx.PaintDC
# SM_BUFFERED_DC Uses The Double Buffered Drawing Style

SM_NORMAL_DC = 0
SM_BUFFERED_DC = 1

#----------------------------------------------------------------------
# SpeedMeter Styles
#----------------------------------------------------------------------
# SM_ROTATE_TEXT: Draws The Ticks Rotated: The Ticks Are Rotated
#                 Accordingly To The Tick Marks Positions
# SM_DRAW_SECTORS: Different Intervals Are Painted In Differend Colours
#                  (Every Sector Of The Circle Has Its Own Colour)
# SM_DRAW_PARTIAL_SECTORS: Every Interval Has Its Own Colour, But Only
#                          A Circle Corona Is Painted Near The Ticks
# SM_DRAW_HAND: The Hand (Arrow Indicator) Is Drawn
# SM_DRAW_SHADOW: A Shadow For The Hand Is Drawn
# SM_DRAW_PARTIAL_FILLER: A Circle Corona That Follows The Hand Position
#                         Is Drawn Near The Ticks
# SM_DRAW_SECONDARY_TICKS: Intermediate (Smaller) Ticks Are Drawn Between
#                          Principal Ticks
# SM_DRAW_MIDDLE_TEXT: Some Text Is Printed In The Middle Of The Control
#                      Near The Center
# SM_DRAW_MIDDLE_ICON: An Icon Is Drawn In The Middle Of The Control Near
#                      The Center
# SM_DRAW_GRADIENT: A Gradient Of Colours Will Fill The Control
# SM_DRAW_FANCY_TICKS: With This Style You Can Use XML Tags To Create
#                      Some Custom Text And Draw It At The Ticks Position.
#                      See wx.lib.fancytext For The Tags.
                                  
SM_ROTATE_TEXT = 1
""" Draws the ticks rotated: the ticks are rotated accordingly to the tick marks positions. """
SM_DRAW_SECTORS = 2
""" Different intervals are painted in differend colours (every sector of the""" \
""" circle has its own colour). """
SM_DRAW_PARTIAL_SECTORS = 4
""" Every interval has its own colour, but only a circle corona is painted near the ticks. """
SM_DRAW_HAND = 8
""" The hand (arrow indicator) is drawn. """
SM_DRAW_SHADOW = 16
""" A shadow for the hand is drawn. """
SM_DRAW_PARTIAL_FILLER = 32
""" A circle corona that follows the hand position is drawn near the ticks. """
SM_DRAW_SECONDARY_TICKS = 64
""" Intermediate (smaller) ticks are drawn between principal ticks. """
SM_DRAW_MIDDLE_TEXT = 128
""" Some text is printed in the middle of the control near the center. """
SM_DRAW_MIDDLE_ICON = 256
""" An icon is drawn in the middle of the control near the center. """
SM_DRAW_GRADIENT = 512
""" A gradient of colours will fill the control. """
SM_DRAW_FANCY_TICKS = 1024
""" With this style you can use xml tags to create some custom text and""" \
""" draw it at the ticks position. See `wx.lib.fancytext` for the tags. """

#----------------------------------------------------------------------
# Event Binding
#----------------------------------------------------------------------
# SM_MOUSE_TRACK: The Mouse Left Click/Drag Allow You To Change The
#                 SpeedMeter Value Interactively

SM_MOUSE_TRACK = 1


fontfamily = range(70, 78)
familyname = ["default", "decorative", "roman", "script", "swiss", "modern", "teletype"]

weights = range(90, 93)
weightsname = ["normal", "light", "bold"]

styles = [90, 93, 94]
stylesname = ["normal", "italic", "slant"]

#----------------------------------------------------------------------
# BUFFERENDWINDOW Class
# This Class Has Been Taken From The wxPython Wiki, And Slightly
# Adapted To Fill My Needs. See:
#
# http://wiki.wxpython.org/index.cgi/DoubleBufferedDrawing
#
# For More Info About DC And Double Buffered Drawing.
#----------------------------------------------------------------------

class BufferedWindow(wx.Window):
    """
    A Buffered window class.

    To use it, subclass it and define a `Draw(dc)` method that takes a `dc`
    to draw to. In that method, put the code needed to draw the picture
    you want. The window will automatically be double buffered, and the
    screen will be automatically updated when a Paint event is received.

    When the drawing needs to change, you app needs to call the
    L{BufferedWindow.UpdateDrawing} method. Since the drawing is stored in a bitmap, you
    can also save the drawing to file by calling the
    `SaveToFile(self, file_name, file_type)` method.
    """

    def __init__(self, parent, id=wx.ID_ANY, pos=wx.DefaultPosition, size=wx.DefaultSize,
                 style=wx.NO_FULL_REPAINT_ON_RESIZE, bufferedstyle=SM_BUFFERED_DC):
        """
        Default class constructor.

        :param `parent`: parent window. Must not be ``None``;
        :param `id`: window identifier. A value of -1 indicates a default value;
        :param `pos`: the control position. A value of (-1, -1) indicates a default position,
         chosen by either the windowing system or wxPython, depending on platform;
        :param `size`: the control size. A value of (-1, -1) indicates a default size,
         chosen by either the windowing system or wxPython, depending on platform;
        :param `style`: the window style;
        :param `bufferedstyle`: if set to ``SM_BUFFERED_DC``, double-buffering will
         be used.
        """

        wx.Window.__init__(self, parent, id, pos, size, style)

        self.Bind(wx.EVT_PAINT, self.OnPaint)
        self.Bind(wx.EVT_SIZE, self.OnSize)
        self.Bind(wx.EVT_ERASE_BACKGROUND, lambda x: None)

        # OnSize called to make sure the buffer is initialized.
        # This might result in OnSize getting called twice on some
        # platforms at initialization, but little harm done.
        self.OnSize(None)


    def Draw(self, dc):
        """
        This method should be overridden when sub-classed.

        :param `dc`: an instance of `wx.DC`.        
        """

        pass


    def OnPaint(self, event):
        """
        Handles the ``wx.EVT_PAINT`` event for L{BufferedWindow}.

        :param `event`: a `wx.PaintEvent` event to be processed.
        """

        if self._bufferedstyle == SM_BUFFERED_DC:
            dc = wx.BufferedPaintDC(self, self._Buffer)
        else:
            dc = wx.PaintDC(self)
            dc.DrawBitmap(self._Buffer,0,0)


    def OnSize(self,event):
        """
        Handles the ``wx.EVT_SIZE`` event for L{BufferedWindow}.

        :param `event`: a `wx.SizeEvent` event to be processed.
        """
        
        self.Width, self.Height = self.GetClientSizeTuple()

        # Make new off screen bitmap: this bitmap will always have the
        # current drawing in it, so it can be used to save the image to
        # a file, or whatever.

        # This seems required on MacOS, it doesn't like wx.EmptyBitmap with
        # size = (0, 0)
        # Thanks to Gerard Grazzini

        if "__WXMAC__" in wx.Platform:
            if self.Width == 0:
                self.Width = 1
            if self.Height == 0:
                self.Height = 1

        self._Buffer = wx.EmptyBitmap(self.Width, self.Height)
        self.UpdateDrawing()


    def UpdateDrawing(self):
        """
        This would get called if the drawing needed to change, for whatever reason.

        The idea here is that the drawing is based on some data generated
        elsewhere in the system. if that data changes, the drawing needs to
        be updated.
        """

        if self._bufferedstyle == SM_BUFFERED_DC:
            dc = wx.BufferedDC(wx.ClientDC(self), self._Buffer)
            self.Draw(dc)
        else:
            # update the buffer
            dc = wx.MemoryDC()
            dc.SelectObject(self._Buffer)

            self.Draw(dc)
            # update the screen
            wx.ClientDC(self).Blit(0, 0, self.Width, self.Height, dc, 0, 0)


#----------------------------------------------------------------------
# SPEEDMETER Class
# This Is The Main Class Implementation. See __init__() Method For
# Details.
#----------------------------------------------------------------------

class SMeter(BufferedWindow):
    """
    SpeedMeter tries to reproduce the behavior of some car controls (but not only),
    by creating an "angular" control (actually, circular).

    This is the main class implementation.
    """
    def __init__(self, parent, id=wx.ID_ANY, pos=wx.DefaultPosition,
                 size=wx.DefaultSize, agwStyle=SM_DRAW_HAND,
                 bufferedstyle=SM_BUFFERED_DC,
                 mousestyle=0):
        """
        Default class constructor.

        :param `parent`: parent window. Must not be ``None``;
        :param `id`: window identifier. A value of -1 indicates a default value;
        :param `pos`: the control position. A value of (-1, -1) indicates a default position,
         chosen by either the windowing system or wxPython, depending on platform;
        :param `size`: the control size. A value of (-1, -1) indicates a default size,
         chosen by either the windowing system or wxPython, depending on platform;
        :param `agwStyle`: this value specifies the L{SpeedMeter} styles, and can be a
         combination of the following bits:

         =========================== =========== ==================================================
         Window Styles               Hex Value   Description
         =========================== =========== ==================================================
         ``SM_ROTATE_TEXT``                  0x1 Draws the ticks rotated: the ticks are rotated accordingly to the tick marks positions.
         ``SM_DRAW_SECTORS``                 0x2 Different intervals are painted in differend colours (every sector of the circle has its own colour).
         ``SM_DRAW_PARTIAL_SECTORS``         0x4 Every interval has its own colour, but only a circle corona is painted near the ticks.
         ``SM_DRAW_HAND``                    0x8 The hand (arrow indicator) is drawn.
         ``SM_DRAW_SHADOW``                 0x10 A shadow for the hand is drawn.
         ``SM_DRAW_PARTIAL_FILLER``         0x20 A circle corona that follows the hand position is drawn near the ticks.
         ``SM_DRAW_SECONDARY_TICKS``        0x40 Intermediate (smaller) ticks are drawn between principal ticks.
         ``SM_DRAW_MIDDLE_TEXT``            0x80 Some text is printed in the middle of the control near the center.
         ``SM_DRAW_MIDDLE_ICON``           0x100 An icon is drawn in the middle of the control near the center.
         ``SM_DRAW_GRADIENT``              0x200 A gradient of colours will fill the control.
         ``SM_DRAW_FANCY_TICKS``           0x400 With this style you can use xml tags to create some custom text and draw it at the ticks position. See `wx.lib.fancytext` for the tags.
         =========================== =========== ==================================================
         
        :param `bufferedstyle`: this value allows you to use the normal `wx.PaintDC` or the
         double buffered drawing options:

         =========================== =========== ==================================================
         Buffered Styles             Hex Value   Description
         =========================== =========== ==================================================
         ``SM_NORMAL_DC``                    0x0 Uses the normal `wx.PaintDC`
         ``SM_BUFFERED_DC``                  0x1 Uses the double buffered drawing style
         =========================== =========== ==================================================

        :param `mousestyle`: this value allows you to use the mouse to change the L{SpeedMeter}
         value interactively with left click/drag events. If set to ``SM_MOUSE_TRACK``, the mouse
         left click/drag allow you to change the L{SpeedMeter} value interactively.
        """

        self._agwStyle = agwStyle
        self._bufferedstyle = bufferedstyle
        self._mousestyle = mousestyle

        if self._agwStyle & SM_DRAW_SECTORS and self._agwStyle & SM_DRAW_GRADIENT:
            errstr = "\nERROR: Incompatible Options: SM_DRAW_SECTORS Can Not Be Used In "
            errstr = errstr + "Conjunction With SM_DRAW_GRADIENT."
            raise Exception(errstr)

        if self._agwStyle & SM_DRAW_PARTIAL_SECTORS and self._agwStyle & SM_DRAW_SECTORS:
            errstr = "\nERROR: Incompatible Options: SM_DRAW_SECTORS Can Not Be Used In "
            errstr = errstr + "Conjunction With SM_DRAW_PARTIAL_SECTORS."
            raise Exception(errstr)

        if self._agwStyle & SM_DRAW_PARTIAL_SECTORS and self._agwStyle & SM_DRAW_PARTIAL_FILLER:
            errstr = "\nERROR: Incompatible Options: SM_DRAW_PARTIAL_SECTORS Can Not Be Used In "
            errstr = errstr + "Conjunction With SM_DRAW_PARTIAL_FILLER."
            raise Exception(errstr)

        if self._agwStyle & SM_DRAW_FANCY_TICKS and self._agwStyle & SM_ROTATE_TEXT:
            errstr = "\nERROR: Incompatible Options: SM_DRAW_FANCY_TICKS Can Not Be Used In "
            errstr = errstr + "Conjunction With SM_ROTATE_TEXT."
            raise Exception(errstr)

        if self._agwStyle & SM_DRAW_SHADOW and self._agwStyle & SM_DRAW_HAND == 0:
            errstr = "\nERROR: Incompatible Options: SM_DRAW_SHADOW Can Be Used Only In "
            errstr = errstr + "Conjunction With SM_DRAW_HAND."
            raise Exception(errstr)

        if self._agwStyle & SM_DRAW_FANCY_TICKS:
            wx.lib.colourdb.updateColourDB()


        self.SetAngleRange()
        self.SetIntervals()
        self.SetSpeedValue()
        self.SetIntervalColours()
        self.SetArcColour()
        self.SetSpeedBackground()
        self.SetHandColour()
        self.SetShadowColour()
        self.SetFillerColour()
        self.SetDirection()
        self.SetNumberOfSecondaryTicks()
        self.SetFirstGradientColour()
        self.SetSecondGradientColour()
        self.SetHandStyle()
        self.DrawExternalArc()
        # Get the s-meter scale image and make a bitmap.
        mtrImage = wx.Image("Signal_Meter_M0MTJ_1.png",type=wx.BITMAP_TYPE_PNG)
        self.mtrImageBitmap = wx.BitmapFromImage(mtrImage)

        BufferedWindow.__init__(self, parent, id, pos, size,
                                style=wx.NO_FULL_REPAINT_ON_RESIZE,
                                bufferedstyle=bufferedstyle)

#        if self._mousestyle & SM_MOUSE_TRACK:
#            self.Bind(wx.EVT_MOUSE_EVENTS, self.OnMouseMotion)

    def Draw(self, dc):
        """
        Draws everything on the empty bitmap.
        Here all the chosen styles are applied.

        :param `dc`: an instance of `wx.DC`.        
        """

        size  = self.GetClientSize()

        if size.x < 21 or size.y < 21:
            return

        new_dim = size.Get()

        if not hasattr(self, "dim"):
            self.dim = new_dim

        self.scale = min([float(new_dim[0]) / self.dim[0],
                          float(new_dim[1]) / self.dim[1]])

        # Create An Empty Bitmap
        self.faceBitmap = wx.EmptyBitmap(size.width, size.height)

        dc.BeginDrawing()

        speedbackground = self.GetSpeedBackground()
        # Set Background Of The Control
        dc.SetBackground(wx.Brush(speedbackground))
        dc.Clear()
#testing MSE
        dc.DrawBitmap(self.mtrImageBitmap, 0, 0)
        
        centerX = self.faceBitmap.GetWidth()/2
        centerY = 137 #self.faceBitmap.GetHeight()

        self.CenterX = centerX
        self.CenterY = centerY

        # Get The Radius Of The Sector. Set It A Bit Smaller To Correct Draw After
        radius = 108 #  min(centerX, centerY) - 2

        self.Radius = radius

        # Get The Angle Of Existance Of The Sector
        anglerange = self.GetAngleRange()
        startangle = anglerange[1]
        endangle = anglerange[0]

        self.StartAngle = startangle
        self.EndAngle = endangle

        # Initialize The Colours And The Intervals - Just For Reference To The
        # Children Functions
        colours = None
        intervals = None

        if self._agwStyle & SM_DRAW_SECTORS or self._agwStyle & SM_DRAW_PARTIAL_SECTORS:
            # Get The Intervals Colours
            colours = self.GetIntervalColours()[:]

        textangles = []
        colourangles = []
        xcoords = []
        ycoords = []

        # Get The Intervals (Partial Sectors)
        intervals = self.GetIntervals()[:]

        start = min(intervals)
        end = max(intervals)
        span = end - start

        self.StartValue = start
        self.EndValue = end

        self.Span = span

        # Get The Current Value For The SpeedMeter
        currentvalue = self.GetSpeedValue()

        # Get The Direction Of The SpeedMeter
        direction = self.GetDirection()
        if direction == "Reverse":
            intervals.reverse()

            if self._agwStyle & SM_DRAW_SECTORS or self._agwStyle & SM_DRAW_PARTIAL_SECTORS:
                colours.reverse()

            currentvalue = end - currentvalue

        # This Because DrawArc Does Not Draw Last Point
        offset = 0.1*self.scale/180.0

        xstart, ystart = self.CircleCoords(radius+1, -endangle, centerX, centerY)
        xend, yend = self.CircleCoords(radius+1, -startangle-offset, centerX, centerY)

        # Calculate The Angle For The Current Value Of SpeedMeter
        accelangle = (currentvalue - start)/float(span)*(startangle-endangle) - startangle

        dc.SetPen(wx.TRANSPARENT_PEN)

        if self._agwStyle & SM_DRAW_PARTIAL_FILLER:

            # Get Some Data For The Partial Filler
            fillercolour = self.GetFillerColour()
            fillerendradius = radius - 10.0*self.scale
            fillerstartradius = radius

            if direction == "Advance":
                fillerstart = accelangle
                fillerend = -startangle
            else:
                fillerstart = -endangle
                fillerend = accelangle

            xs1, ys1 = self.CircleCoords(fillerendradius, fillerstart, centerX, centerY)
            xe1, ye1 = self.CircleCoords(fillerendradius, fillerend, centerX, centerY)
            xs2, ys2 = self.CircleCoords(fillerstartradius, fillerstart, centerX, centerY)
            xe2, ye2 = self.CircleCoords(fillerstartradius, fillerend, centerX, centerY)

            # Get The Sector In Which The Current Value Is
            intersection = self.GetIntersection(currentvalue, intervals)
            sectorradius = radius - 10*self.scale

        else:

            sectorradius = radius

        if self._agwStyle & SM_DRAW_PARTIAL_FILLER:
            # Draw The Filler (Both In "Advance" And "Reverse" Directions)

            dc.SetBrush(wx.Brush(fillercolour))
            dc.DrawArc(xs2, ys2, xe2, ye2, centerX, centerY)

            if self._agwStyle & SM_DRAW_SECTORS == 0:
                dc.SetBrush(wx.Brush(speedbackground))
                xclean1, yclean1 = self.CircleCoords(sectorradius, -endangle, centerX, centerY)
                xclean2, yclean2 = self.CircleCoords(sectorradius, -startangle-offset, centerX, centerY)
                dc.DrawArc(xclean1, yclean1, xclean2, yclean2, centerX, centerY)


        # This Is Needed To Fill The Partial Sector Correctly
        xold, yold = self.CircleCoords(radius, startangle+endangle, centerX, centerY)

        # Draw The Sectors
        for ii, interval in enumerate(intervals):

            if direction == "Advance":
                current = interval - start
            else:
                current = end - interval

            angle = (current/float(span))*(startangle-endangle) - startangle
            angletext = -((pi/2.0) + angle)*180/pi
            textangles.append(angletext)
            colourangles.append(angle)
            xtick, ytick = self.CircleCoords(radius, angle, centerX, centerY)

            # Keep The Coordinates, We Will Need Them After To Position The Ticks
            xcoords.append(xtick)
            ycoords.append(ytick)
            x = xtick
            y = ytick

            if self._agwStyle & SM_DRAW_SECTORS:
                if self._agwStyle & SM_DRAW_PARTIAL_FILLER:
                    if direction == "Advance":
                        if current > currentvalue:
                            x, y = self.CircleCoords(radius, angle, centerX, centerY)
                        else:
                            x, y = self.CircleCoords(sectorradius, angle, centerX, centerY)
                    else:
                        if current < end - currentvalue:
                            x, y = self.CircleCoords(radius, angle, centerX, centerY)
                        else:
                            x, y = self.CircleCoords(sectorradius, angle, centerX, centerY)
                else:
                    x, y = self.CircleCoords(radius, angle, centerX, centerY)


            if ii > 0:
                if self._agwStyle & SM_DRAW_PARTIAL_FILLER and ii == intersection:
                    # We Got The Interval In Which There Is The Current Value. If We Choose
                    # A "Reverse" Direction, First We Draw The Partial Sector, Next The Filler

                    dc.SetBrush(wx.Brush(speedbackground))

                    if direction == "Reverse":
                        if self._agwStyle & SM_DRAW_SECTORS:
                            dc.SetBrush(wx.Brush(colours[ii-1]))

                        dc.DrawArc(xe2, ye2, xold, yold, centerX, centerY)

                    if self._agwStyle & SM_DRAW_SECTORS:
                        dc.SetBrush(wx.Brush(colours[ii-1]))
                    else:
                        dc.SetBrush(wx.Brush(speedbackground))


                    dc.DrawArc(xs1, ys1, xe1, ye1, centerX, centerY)

                    if self._agwStyle & SM_DRAW_SECTORS:
                        dc.SetBrush(wx.Brush(colours[ii-1]))
                        # Here We Draw The Rest Of The Sector In Which The Current Value Is
                        if direction == "Advance":
                            dc.DrawArc(xs1, ys1, x, y, centerX, centerY)
                            x = xs1
                            y = ys1
                        else:
                            dc.DrawArc(xe2, ye2, x, y, centerX, centerY)

                elif self._agwStyle & SM_DRAW_SECTORS:
                    dc.SetBrush(wx.Brush(colours[ii-1]))

                    # Here We Still Use The SM_DRAW_PARTIAL_FILLER Style, But We Are Not
                    # In The Sector Where The Current Value Resides
                    if self._agwStyle & SM_DRAW_PARTIAL_FILLER and ii != intersection:
                        if direction == "Advance":
                            dc.DrawArc(x, y, xold, yold, centerX, centerY)
                        else:
                            if ii < intersection:
                                dc.DrawArc(x, y, xold, yold, centerX, centerY)

                    # This Is The Case Where No SM_DRAW_PARTIAL_FILLER Has Been Chosen
                    else:
                        dc.DrawArc(x, y, xold, yold, centerX, centerY)

            else:
                if self._agwStyle & SM_DRAW_PARTIAL_FILLER and self._agwStyle & SM_DRAW_SECTORS:
                    dc.SetBrush(wx.Brush(fillercolour))
                    dc.DrawArc(xs2, ys2, xe2, ye2, centerX, centerY)
                    x, y = self.CircleCoords(sectorradius, angle, centerX, centerY)
                    dc.SetBrush(wx.Brush(colours[ii]))
                    dc.DrawArc(xs1, ys1, xe1, ye1, centerX, centerY)
                    x = xs2
                    y = ys2

            xold = x
            yold = y

            if self._agwStyle & SM_DRAW_PARTIAL_SECTORS:

                sectorendradius = radius - 10.0*self.scale
                sectorstartradius = radius

                xps, yps = self.CircleCoords(sectorstartradius, angle, centerX, centerY)

                if ii > 0:
                    dc.SetBrush(wx.Brush(colours[ii-1]))
                    dc.DrawArc(xps, yps, xpsold, ypsold, centerX, centerY)

                xpsold = xps
                ypsold = yps


        textheight = 0

        # Requested To Draw The Hand
        if self._agwStyle & SM_DRAW_HAND:

            handstyle = self.GetHandStyle()
            handcolour = self.GetHandColour()

            # Calculate The Data For The Hand
            if textheight == 0:
                maxradius = radius-10*self.scale
            else:
                maxradius = radius-5*self.scale-textheight

            xarr, yarr = self.CircleCoords(maxradius, accelangle, centerX, centerY)

            x1, y1 = self.CircleCoords(maxradius, accelangle - 6.0/180, centerX, centerY)
            x2, y2 = self.CircleCoords(maxradius, accelangle + 6.0/180, centerX, centerY)
            x3, y3 = self.CircleCoords(maxradius+3*(abs(xarr-x1)), accelangle, centerX, centerY)

            newx = centerX + 4*cos(accelangle)*self.scale
            newy = centerY + 4*sin(accelangle)*self.scale

            if self._agwStyle & SM_DRAW_SHADOW:

                # Draw The Shadow
                shadowcolour = self.GetShadowColour()
                dc.SetPen(wx.Pen(shadowcolour, 5*log(self.scale+1)))
                dc.SetBrush(wx.Brush(shadowcolour))
                shadowdistance = 2.0*self.scale
                dc.DrawLine(newx + shadowdistance, newy + shadowdistance,
                            xarr + shadowdistance, yarr + shadowdistance)

                dc.DrawPolygon([(x1+shadowdistance, y1+shadowdistance),
                                (x2+shadowdistance, y2+shadowdistance),
                                (x3+shadowdistance, y3+shadowdistance)])
 
            # Always use "Arrow" mode.
            #dc.SetPen(wx.Pen(handcolour, 5*log(self.scale+1)))
            dc.SetPen(wx.Pen(handcolour, 3))
            # Draw The "Hand", An Arrow
            dc.DrawLine(newx, newy, xarr, yarr)

            # Draw The Arrow Pointer
            dc.SetBrush(wx.Brush(handcolour))
            dc.DrawPolygon([(x1, y1), (x2, y2), (x3, y3)])
            # Draw The Small Circle In The Center --> The Hand "Holder"
            dc.SetPen(wx.Pen(wx.BLACK, 1.5))
            dc.SetBrush(wx.Brush(wx.WHITE))
            dc.DrawCircle(centerX, centerY, 30*self.scale)

        dc.EndDrawing()

    def SetIntervals(self, intervals=None):
        """
        Sets the intervals for L{SpeedMeter} (main ticks numeric values).

        :param `intervals`: a Python list of main ticks to be displayed. If defaulted
         to ``None``, the list `[0, 50, 100]` is used.
        """

        if intervals is None:
            intervals = [0, 50, 100]

        self._intervals = intervals

    def GetIntervals(self):
        """ Returns the intervals for L{SpeedMeter}, a Python list of main ticks displayed. """
        return self._intervals

    def SetSpeedValue(self, value=None):
        """
        Sets the current value for L{SpeedMeter}.

        :param `value`: a floating point number representing the current value. If defaulted
         to ``None``, the L{SpeedMeter} value will be the middle point of the control range.
        """

        # translate input, S1 -- S9+60 (in dB rel. to S9) to 0 - 100 linear
        if value < -48: value = -48     # Should note range error?
        if value > 60 : value =  60     # ditto
        # Note scale compresses for values > S9 (dB from Rx > 0)
        if value < 0:
            x = (value + 48) * 1.19
        else:
            x = 57 + (value * 0.70)
        self._speedvalue = x
        try:
            self.UpdateDrawing()
        except:
            pass


    def GetSpeedValue(self):
        """ Returns the current value for L{SpeedMeter}. """

        return self._speedvalue


    def SetAngleRange(self, start=0, end=pi):
        """
        Sets the range of existence for L{SpeedMeter}.

        :param `start`: the starting angle, in radians;
        :param `end`: the ending angle, in radians.        
        """

        self._anglerange = [start, end]


    def GetAngleRange(self):
        """
        Returns the range of existence for L{SpeedMeter}.
        The returned values are in radians.
        """

        return self._anglerange


    def SetIntervalColours(self, colours=None):
        """
        Sets the colours for the intervals.

        :param `colours`: a Python list of colours. The length of this list should be
         the same as the number of circle sectors in L{SpeedMeter}. If defaulted to ``None``,
         all the intervals will have a white colour.
         
        :note: Every interval (circle sector) should have a colour.
        """

        if colours is None:
            if not hasattr(self, "_anglerange"):
                errstr = "\nERROR: Impossible To Set Interval Colours,"
                errstr = errstr + " Please Define The Intervals Ranges Before."
                raise Exception(errstr)
                return

            colours = [wx.WHITE]*len(self._intervals)
        else:
            if len(colours) != len(self._intervals) - 1:
                errstr = "\nERROR: Length Of Colour List Does Not Match Length"
                errstr = errstr + " Of Intervals Ranges List."
                raise Exception(errstr)
                return

        self._intervalcolours = colours

    def SetSpeedBackground(self, colour=None):
        """
        Sets the background colour outside the L{SpeedMeter} control.

        :param `colour`: a valid `wx.Colour` object. If defaulted to ``None``, the
         L{SpeedMeter} background will be taken from the system default.
        """

        if colour is None:
            colour = wx.SystemSettings_GetColour(0)

        self._speedbackground = colour


    def GetSpeedBackground(self):
        """ Returns the background colour outside the L{SpeedMeter} control."""

        return self._speedbackground


    def SetHandColour(self, colour=None):
        """
        Sets the hand (arrow indicator) colour.

        :param `colour`: a valid `wx.Colour` object. If defaulted to ``None``, the arrow
         indicator will be red.
        """

        if colour is None:
            colour = wx.RED

        self._handcolour = colour


    def GetHandColour(self):
        """ Returns the hand (arrow indicator) colour."""

        return self._handcolour


    def SetArcColour(self, colour=None):
        """
        Sets the external arc colour (thicker line).

        :param `colour`: a valid `wx.Colour` object. If defaulted to ``None``, the arc
         colour will be black.
        """

        if colour is None:
            colour = wx.BLACK

        self._arccolour = colour


    def GetArcColour(self):
        """ Returns the external arc colour."""

        return self._arccolour


    def SetShadowColour(self, colour=None):
        """
        Sets the hand's shadow colour.

        :param `colour`: a valid `wx.Colour` object. If defaulted to ``None``, the shadow
         colour will be light grey.
        """

        if colour is None:
            colour = wx.Colour(150, 150, 150)

        self._shadowcolour = colour


    def GetShadowColour(self):
        """ Returns the hand's shadow colour."""

        return self._shadowcolour


    def SetFillerColour(self, colour=None):
        """
        Sets the partial filler colour.

        A circle corona near the ticks will be filled with this colour, from
        the starting value to the current value of L{SpeedMeter}.

        :param `colour`: a valid `wx.Colour` object.
        """

        if colour is None:
            colour = wx.Colour(255, 150, 50)

        self._fillercolour = colour


    def GetFillerColour(self):
        """ Returns the partial filler colour."""

        return self._fillercolour


    def SetDirection(self, direction=None):
        """
        Sets the direction of advancing L{SpeedMeter} value.

        :param `direction`: specifying "advance" will move the hand in clock-wise direction
         (like normal car speed control), while using "reverse" will move it counterclock-wise
         direction. If defaulted to ``None``, then "advance" will be used.
        """

        if direction is None:
            direction = "Advance"

        if direction not in ["Advance", "Reverse"]:
            raise Exception('\nERROR: Direction Parameter Should Be One Of "Advance" Or "Reverse".')

        self._direction = direction


    def GetDirection(self):
        """ Returns the direction of advancing L{SpeedMeter} value."""

        return self._direction


    def SetNumberOfSecondaryTicks(self, ticknum=None):
        """
        Sets the number of secondary (intermediate) ticks.

        :param `ticknum`: the number of intermediate ticks. If defaulted to ``None``,
         3 ticks are used.
        """

        if ticknum is None:
            ticknum = 3

        if ticknum < 1:
            raise Exception("\nERROR: Number Of Ticks Must Be Greater Than 1.")

        self._secondaryticks = ticknum


    def GetNumberOfSecondaryTicks(self):
        """ Returns the number of secondary (intermediate) ticks. """

        return self._secondaryticks




    def GetMiddleIconDimens(self):
        """ Used internally. """

        return self._middleicon.GetWidth(), self._middleicon.GetHeight()


    def CircleCoords(self, radius, angle, centerX, centerY):
        """
        Converts the input values into logical x, y coordinates.

        :param `radius`: the L{SpeedMeter} radius;
        :param `angle`: the angular position of the mouse;
        :param `centerX`: the `x` position of the L{SpeedMeter} center;
        :param `centerX`: the `y` position of the L{SpeedMeter} center.        
        """

        x = radius*cos(angle) + centerX
        y = radius*sin(angle) + centerY

        return x, y


    def GetIntersection(self, current, intervals):
        """ Used internally. """

        if self.GetDirection() == "Reverse":
            interval = intervals[:]
            interval.reverse()
        else:
            interval = intervals

        indexes = range(len(intervals))
        try:
            intersection = [ind for ind in indexes if interval[ind] <= current <= interval[ind+1]]
        except:
            if self.GetDirection() == "Reverse":
                intersection = [len(intervals) - 1]
            else:
                intersection = [0]

        return intersection[0]


    def SetFirstGradientColour(self, colour=None):
        """
        Sets the first gradient colour (near the ticks).

        :param `colour`: a valid `wx.Colour` object.
        """

        if colour is None:
            colour = wx.Colour(145, 220, 200)

        self._firstgradientcolour = colour


    def GetFirstGradientColour(self):
        """ Returns the first gradient colour (near the ticks). """

        return self._firstgradientcolour


    def SetSecondGradientColour(self, colour=None):
        """
        Sets the second gradient colour (near the center).

        :param `colour`: a valid `wx.Colour` object.
        """

        if colour is None:
            colour = wx.WHITE

        self._secondgradientcolour = colour


    def GetSecondGradientColour(self):
        """ Returns the first gradient colour (near the center). """

        return self._secondgradientcolour


    def SetHandStyle(self, style=None):
        """
        Sets the style for the hand (arrow indicator).

        :param `style`: by specifying "Hand", L{SpeedMeter} will draw a polygon
         that simulates the car speed control indicator. Using "Arrow" will force
         L{SpeedMeter} to draw a simple arrow. If defaulted to ``None``, "Hand" will
         be used.
         """

        if style is None:
            style = "Hand"

        if style not in ["Hand", "Arrow"]:
            raise Exception('\nERROR: Hand Style Parameter Should Be One Of "Hand" Or "Arrow".')
            return

        self._handstyle = style


    def GetHandStyle(self):
        """ Returns the style for the hand (arrow indicator)."""

        return self._handstyle


    def DrawExternalArc(self, draw=True):
        """
        Specify wheter or not you wish to draw the external (thicker) arc.

        :param `draw`: ``True`` to draw the external arc, ``False`` otherwise.
        """

        self._drawarc = draw

# Testing code follows below
if __name__ == "__main__":
    class MyFrame(wx.Frame):
        def __init__(self, vval, *args, **kwds):
            kwds["style"] = \
                wx.MINIMIZE_BOX|wx.CAPTION|wx.CLOSE_BOX|wx.CLIP_CHILDREN|wx.RESIZE_BORDER
            wx.Frame.__init__(self, *args, **kwds)
            
            self.menubar = wx.MenuBar()
            self.f_menu = wx.Menu()
            quitter = self.f_menu.Append(-1, "Quit", "", wx.ITEM_NORMAL)
            self.menubar.Append(self.f_menu, "File")
            self.SetMenuBar(self.menubar)
            self.SetFont(wx.Font(32, wx.SWISS, wx.NORMAL, wx.NORMAL, 0, ""))
                
            self.SetTitle("smeter test")

            self.Bind(wx.EVT_MENU, self.DoQuit, quitter)
            self.Bind(wx.EVT_CLOSE, self.DoQuit)    # (event from close box)

            self.meter = SMeter(self )
            self.meter.SetAngleRange(0.75, pi-.75)
            self.meter.SetHandStyle("Arrow")
            self.meter.SetHandColour(wx.RED)
            self.meter.DrawExternalArc(False)
            self.meter.SetSpeedValue(vval)
            self.meter.SetSize((300,300))
        
        def Value(x):
            self.meter.SetSpeedValue(x)
        
        def DoQuit(self, event):
            wx.Exit()

# end of class MyFrame

    app = wx.PySimpleApp(0)
    wx.InitAllImageHandlers()
    frame_FSpin = MyFrame(10, None, -1, "", size=(212,125))
    app.SetTopWindow(frame_FSpin)
    frame_FSpin.Show()
    app.MainLoop()
