#!/usr/bin/env python
# -*- coding: utf-8 -*-
# jedit info = :noTabs=true:tabSize=4:

# Remote.py version 0.28devel 2/2013 MSE
# Remote rig control using Hamlib rigctld protocol

#   Copyright (C) 2012, 2013 Martin S. Ewing, AA6E
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

# TO DO
# Find problem that kills control loop.
# Other T/R options: electrical from xcvr? -- Hamlib ctrl of xcvr?
# Implement more controls: PBT, ...

# Note: we require wxPython from wxpython.org (a Graphical User Interface system)
import wx
import sys, subprocess, time, os, sysconfig, argparse
from hamlibio import *
from FSpin import *
import wx.lib.agw.knobctrl as KC
import wx.lib.throbber as throb
from smeter import *
from wx.lib.wordwrap import wordwrap

# Audio Link enums (enumerated values)
LNK_STOP    = 0
LNK_RUN     = 1

# Hamlib interface enums
HL_VALUE    = 0
HL_SELECT   = 1
HL_NULL     = 2     # no control

# Codec enums
CODEC_NULL  = 0
CODEC_SPEEX = 1

# Should we launch remote servers for control (Hamlib rigctld) and audio (afxmit)?
# Normally yes, remote.py will kill any existing servers and start new ones.
AUTOLAUNCH = True

# Are we are running in a Windows 32/64 environment?
# If not (WIN32==False) - we must be on Linux.  This affects the scripts that
# are called to (re)start audio & control servers, start & stop audio, etc.
WIN32 = (sysconfig.get_platform() in ["win32", "win-amd64"])

# Get the story on what kind of Linux (32 or 64 bit) we may have.
LINUX = (sysconfig.get_platform().find('linux') >= 0)      # e.g., linux-i686 -i586 etc
if LINUX:
    LINUX64 = (sysconfig.get_platform().find('_64') >= 0)  # e.g., linux-x86_64
else:
    LINUX64 = False

RIGNAME = 'Ten-Tec Jupiter'	    # Other rigs will require some recoding.
VERSION = "0.28_jupiter"

# This list is for convenient removal of features, depending on rig capability.
# Possibilities: afgain, rfgain, agc, filter, mode, s-meter, if, nr, attn, anf, nb
# (These have not been uses consistently!)
SUPPORTED = [ 'mode', 'rfgain', 'afgain',  'agc', 
                'filter', 's-meter' , 'nb', 'anf', 'nr', 'if', 'attn' ]  

# Required capabilities: vfo, band set

MODES = [ 'USB', 'LSB', 'CW', 'CWR', 'AM', 'RTTY' ]
DEFAULT_MODE = 1    # We start up with default mode, freq, etc.

# Choice of band places limits on VFO setting.  Use "max" if general
# coverage is desired.
# When a band is selected, we jump to the center freq. and a default mode,
# unless we have used this band before in this session.  In that case, the
# previously used freq. and mode are reloaded.
BANDS = [ '160', '80', '60', '40', '30', '20', '17', '15',
            '12', '10', '6' , 'max']
DEFAULT_BAND = 3    # 40 M
DEFAULT_FREQ = 7150000
BAND_LIMITS = [ (1800, 2000),(3500, 4000),(5330, 5403),(7000, 7300),
    (10100, 10150),(14000, 14350),(18068, 18168),(21000, 21450),
    (24890, 24990),(28000, 29700),(50000, 54000),(500,30000) ]
BAND_MODES = [ 
    'LSB', 'LSB', 'USB', 'LSB', 
    'CW',  'USB', 'USB', 'USB',
    'USB', 'USB', 'USB', 'AM' ]

# Filters: for each mode (USB, LSB, etc), give (low, med, high) BW option
# Hopefully, Hamlib & Jupiter produce something close to these values.
FILTERS = [ (1800, 2400, 2800), (1800, 2400, 2800), # USB, LSB
            (200, 500, 1000), (200, 500, 1000), # CW, CWR
            (3000, 4500, 6000), (300, 800, 1200) ]  # AM, RTTY
FILTER_NAMES = [ 'NAR', 'MED', 'WIDE' ]
DEFAULT_FILTER = 1      # i.e., medium BW for the mode

AGCSET= [ 'OFF', 'MED', 'FAST', 'SLOW' ]    # Hamlib enum - don't change!
NBSET = [ 'off', '1', '2', '3', '4', '5', '6']
DEFAULT_AGC = 1

# Repetition rate of timer events in msec.  If too low, you may lose GUI responsiveness.
# It should be longer than longest typical Hamlib update, plus local update processing. 
# Best setting will depend on local CPU speed, network, etc.
TIMER_INTVL = 1000

# Parameters relating to S-meter
METER_RANGE = (-60,60)          # meter range in dB
FIRCOEFFS = [ 0.1, 0.2, 1.0 ]   # A little smoothing for S-meter
SMETER_RETRY_LIMIT = 3          # Max. times to repeat query before despairing.

if WIN32:
    # This routine is needed for Windows. It will call plink (putty cli) that
    # uses unix-like parameters.  Can't use usual Python subprocess.call(?)  Windows
    # is slower about all this than Linux.  (That's why we use mute/unmute for 
    # prompt T/R switching.)
    
    def win_call(wlist, wshell):
        # wlist[0] is complete desired command line
        # Make a temporary batch file
        NAME = "tmp.bat"
        tfile = open(NAME,"w")
        tfile.write(wlist[0]+"\n")
        tfile.write("EXIT")
        tfile.close()
        # ... and execute.  START is needed to suppress stdout msg from bat file.
        e = subprocess.call("START /MIN /WAIT "+NAME, shell=True)
        os.remove(NAME)         # delete temporary
        return e

class Fir(object):
    # Simple FIR filter for S-meter, etc.
    def __init__(self, coeff):
        self.coeff = coeff      # list of coefficients, size determines filter length
        self.length = len(coeff)
        self.history = len(coeff) * [ 0 ]
        self.normalize = 1.0 / sum(coeff)

    def filter(self, x):
        self.history.pop(0)     # drop oldest sample (from left)
        self.history.append(float(x))  # add new sample (at right)
        resp = self.normalize * sum( [ self.history[i] * self.coeff[i] 
                        for i in range(self.length) ] )
        return resp

# MyFrame is the main class of remote.py, defining the GUI interface, handling
# interrupts, etc.

class MyFrame(wx.Frame):
    def __init__(self, *args, **kwds):
        kwds["style"] = \
            wx.MINIMIZE_BOX|wx.CAPTION|wx.CLOSE_BOX|wx.CLIP_CHILDREN|wx.RESIZE_BORDER
        wx.Frame.__init__(self, *args, **kwds)
        
        # Use the argparse package to handle command line arguments
        # http://docs.python.org/2/library/argparse.html
            # Usage:  remote.py
            #           -r or --hostname + my_remote_host.dyndns.org (required)
            #           -u or --username + user_account_name at remote host (required)
            #           -p or --password + user_password at remote host (required)
            #           -t or --ssh_port + ssh port at remote host (default 22)
            #           -q or --control_port + rigctrld port at remote (default 4532)
            #           -c or --codec + codec_name (raw or speex, default speex)
            #           -h or --help (this message, more or less)
            #
            # typical: python remote.py -h host.dyndns.org -u W1XYZ -p myPwd 
        
        parser = argparse.ArgumentParser(description="Run Control Panel for Remote Receiver",
            epilog="Typical command: python remote.py -h host.dyndns.org -u W1XYZ -p myPwd")
        parser.add_argument('-r', '--hostname', required=True)
        parser.add_argument('-u', '--username', required=True)
        parser.add_argument('-p', '--password', required=True)
        parser.add_argument('-t', '--ssh_port', default=22)
        parser.add_argument('-q', '--control_port', default=4532)
        parser.add_argument('-c', '--codec', choices=['null', 'speex'], default='speex')
        argd = vars( parser.parse_args() )  # completed arguments in dictionary form

        # Find numeric code for codec choice, then translate params to more
        # convenient instance variables.
        if    argd['codec'] == 'null':  self.ncodec = CODEC_NULL
        elif  argd['codec'] == 'speex': self.ncodec = CODEC_SPEEX
        else:                           self.ncodec = None
        self.username =     argd['username']
        self.hostname =     argd['hostname']
        self.password =     argd['password']
        self.ssh_port =     argd['ssh_port']
        self.control_port = argd['control_port']

        # Set up OS-specific functions
        #   LAUNCH_ALL  -   command to (re)start audio and rigctld servers on remote via SSH.
        #   START_RECEIVE - start afrecv subprocess, which will open the audio channel and
        #                   begin receiving audio data which is sent to local sound card.
        #   KILL_RECEIVE1 - terminate afrecv subprocess, gracefully if possible
        #   KILL_RECEIVE2 - clean up (needed in Windows) to terminate remote transmission
        #   CMD_MUTE -      local command that will mute local sound card
        #   CMD_UNMUTE -    local command that will unmute local sound card
        # (These were formerly global variables, but are now instance variables, 
        #  e.g. self.launch_all etc.)

        if WIN32:               # WINDOWS-SPECIFIC SETUP
            def _fixpath(s):    # quote the path name after the "C:", allow for spaces in dir. names.
                r = s[0:2]+'"'+s[2:]+'"'
                return r
            self.launch_all = ['plink -ssh -l %s -pw %s -P %s %s ./start_all' \
                %(self.username, self.password, self.ssh_port, self.hostname) ]
            # "Start /min" will still create an item in the taskbar, but no window.
            self.start_receive = 'START /MIN /REALTIME '+ \
                _fixpath(os.path.join(sys.path[0],"af","afrecv.exe"))+ \
                (' -c%d -s ' % self.ncodec) + self.hostname
            self.kill_receive1 = "TASKKILL /F /IM afrecv.exe /T"
            # Windows afrecv kill is non-graceful, leaving afxmit sending.
            # Need second call with "-k" switch to stop afxmit, wait for completion.
            self.kill_receive2 = _fixpath(os.path.join(sys.path[0],"af","afrecv.exe"))+' -k ' \
                + self.hostname
            self.cmd_mute    = "nircmdc mutesysvolume 1"
            self.cmd_unmute  = "nircmdc mutesysvolume 0"
        elif LINUX:                   # LINUX-SPECIFIC SETUP
            # Note: Need "sshpass" package to allow pwd use in command line.
            self.launch_all = ['sshpass -p %s ssh -o ConnectTimeout=10 -p %s %s@%s "./start_all"' \
                % (self.password, self.ssh_port, self.username, self.hostname)]
            if LINUX64: tag = '64'  # tag signals use of 64 bit version of afrecv
            else:       tag = '' 
            self.start_receive = os.path.join(sys.path[0],"af","afrecv"+tag)+ \
                " -c%d -s %s &" % (self.ncodec, self.hostname)
            self.kill_receive1 = '/usr/bin/killall -s SIGHUP afrecv'+tag
            # The Linux afrecv kill works gracefully, shutting down afxmit stream.
            # No second whack required.
            self.kill_receive2 = ''
            self.cmd_mute    = "pactl set-sink-mute 0 1"     # Linux Mute (PulseAudio)
            self.cmd_unmute  = "pactl set-sink-mute 0 0"     # Linux UnMute
        else:
            print "Unrecognized OS: ", sysconfig.get_platform()
            sys.exit()
        
        if AUTOLAUNCH:
            # start / restart control link and audio server via ssh or plink
            print "Launching rigctld and afxmit server daemons on remote..."
            cmd = self.launch_all
            if WIN32:
                err = win_call(cmd, True)
            else: 
                err = subprocess.call(cmd, shell=True)
            if err != 0:
                print "Tried", cmd
                print "Error: problem starting remote RX controller"
                sys.exit()
        # open connection to remote rig through Hamlib's rigctrld protocol.
        # If we can't do it, print an error and quit.
        try:
            self.myhamlib = HamlibIO(host=self.hostname, port=self.control_port)
        except:
            s = "Sorry, remote rig cannot be accessed at %s, port %d. "\
                "Please check network setup and remote status and try again."\
                % (self.hostname, self.control_port)
            print s
            d = wx.MessageDialog(self, s, "Remote Connection Error", wx.OK|wx.ICON_ERROR)
            d.ShowModal()
            d.Destroy()
            sys.exit()          # wx.Exit() doesn't work here.

        # Define a few menu options
        self.menubar = wx.MenuBar()
        self.f_menu = wx.Menu()
        quitter = self.f_menu.Append(wx.ID_EXIT, "Quit", "Quit application", wx.ITEM_NORMAL)
        self.menubar.Append(self.f_menu, "File")
        h_menu = wx.Menu()
        helper = h_menu.Append(wx.ID_HELP, "Status - Help", "Information")
        abouter = h_menu.Append(wx.ID_ABOUT, "About Remote.py", "version / status")
        self.menubar.Append(h_menu, "Help")
        self.SetMenuBar(self.menubar)

        # Some defaults for our GUI.
        self.SetFont(wx.Font(10, wx.SWISS, wx.NORMAL, wx.NORMAL, 0, ""))
        self.SetForegroundColour(wx.BLACK)
        self.SetBackgroundColour(wx.WHITE)
        self.SetTitle("Remote Rig Control "+VERSION)
        
        # DEFINE CONTROL PANEL LAYOUT

        # indicator panel, containing 2 "LED lamps" and captions
        self.indBox = wx.StaticBox(self, -1, "Activity")
        self.indPanel = wx.Panel(self, -1)
        self.indPanel.SetForegroundColour(wx.BLACK)
        self.indPanel.SetBackgroundColour(wx.WHITE)
        self.indPanel.SetFont(wx.Font(9, wx.SWISS, wx.NORMAL, wx.NORMAL))
        # Text titles for LEDs
        self.ledCtrlTxt =wx.StaticText(self.indPanel, -1, label="Control")
        self.ledAudioTxt=wx.StaticText(self.indPanel, -1, label="Audio")
        # "LED" for throb controls, Control & Audio indicators
        ledGrnImage = wx.Image("indicator-grn.png")
        ledGrnBitmap = wx.BitmapFromImage(ledGrnImage)
        # self.ledCtrl "throbs" with activity on the Hamlib control channel
        self.ledCtrl = throb.Throbber(self.indPanel, -1, ledGrnBitmap, size=(36,36), 
            frameDelay=1, frames=2, frameWidth=36)
        # self.ledAudio lights when the audio channel is running.
        self.ledAudio = throb.Throbber(self.indPanel, -1, ledGrnBitmap, size=(36,36), 
            frameDelay=1, frames=2, frameWidth=36)

        # VFO is the most complicated control, allowing +/- at each digit.
        self.vfobox = wx.StaticBox(self, -1, "VFO Freq. (Hz)")
        self.vfo = FSpin(self, -1, fvalue=14070000, fextend=0, fdigits=8, fsize=32)
        self.vfo.SetValue(DEFAULT_FREQ)
        self.vfo.SetFocus()         # Check if needed - enabling mute/unmute?
        
        # Allow user to type in desired frequency.
        self.DirectFreqEntryButton = wx.Button(self, -1, "Freq. from Keyboard", (20,20))

        # S-meter is a rough simulation, limited by the ~1 Hz update rate.
        self.meterbox = wx.StaticBox(self, -1, "")
        self.meter = SMeter(self, size=(214,125), agwStyle=SM_DRAW_HAND)
        self.meter.SetHandStyle("Arrow")
        self.meter.SetAngleRange(0.75, pi-.75)
        self.meter.SetSpeedValue(0)
        if 's-meter' not in SUPPORTED:
            self.meter.Disable()
            self.meter.SetHandColour('GRAY')
            self.meter.SetSpeedValue(-60)

        self.fir = Fir(FIRCOEFFS)           # S-meter smoothing filter

        # AF gain control uses receiver units -- could use a log taper here!
        self.afbox = wx.StaticBox(self, -1, "AF Gain")
        self.afgain = KC.KnobCtrl(self, -1, size=(80,80))
        self.afgain.SetKnobRadius(4)
        self.afgain.SetAngularRange(-45, 225)
        self.afgain.SetTags(range(0,100,10)) # Tags must be drawn last
        if 'afgain' not in SUPPORTED:
            self.afgain.Disable()
            self.afgain.SetTagsColour('GRAY')
        else:
            self.afgain.SetValue(40)

        self.rfbox = wx.StaticBox(self, -1, "RF Gain")
        self.rfgain = KC.KnobCtrl(self, -1, size=(80,80))
        self.rfgain.SetKnobRadius(4)
        self.rfgain.SetAngularRange(-45, 225)
        self.rfgain.SetTags(range(0,100,10))
        if 'rfgain' not in SUPPORTED:
            self.rfgain.Disable()
            self.rfgain.SetTagsColour('GRAY')
        else:
            self.rfgain.SetValue(100)

        self.mode = wx.RadioBox(self, -1, "Modes", choices=MODES, majorDimension=2,
            style=wx.RA_SPECIFY_ROWS)
        if 'mode' not in SUPPORTED:
            self.mode.Disable()
        else:
            self.mode.SetSelection(DEFAULT_MODE)
            self.currentMode = MODES[DEFAULT_MODE]

        self.band = wx.RadioBox(self, -1, "Bands", choices=BANDS, majorDimension=3,
            style=wx.RA_SPECIFY_ROWS)
        self.band.SetSelection(DEFAULT_BAND)    # initial band (select rbox)
        self.bandno = DEFAULT_BAND              # index value
        self.range = BAND_LIMITS[DEFAULT_BAND]  # [lower, upper] kHz
        rghz = [ k*1000 for k in self.range ]   # same, in Hz
        self.vfo.SetLimit(rghz)                 # tell VFO ctrl abt current range

        self.filter = wx.RadioBox(self, -1, "Bandwidth", choices=FILTER_NAMES,
                majorDimension=1, style=wx.RA_SPECIFY_ROWS)
        if 'filter' not in SUPPORTED:
            self.filter.Disable()
        else:
            self.filter.SetSelection(DEFAULT_FILTER)
            self.currentFilter = FILTERS[DEFAULT_MODE][DEFAULT_FILTER]        
                
        self.nb = wx.RadioBox(self, -1, "Noise Blank", choices=NBSET, majorDimension=4,
            style=wx.RA_SPECIFY_COLS)
        self.nb.SetSelection(0)                 # default = 0 = off
        
        self.autonotch = wx.RadioBox(self, -1, "Auto Notch", choices=['off', 'on'],
            majorDimension=1, style=wx.RA_SPECIFY_ROWS)
        self.autonotch.SetSelection(0)          # off
        
        self.noisereduction = wx.RadioBox(self, -1, "Noise Red.", choices=['off', 'on'],
            majorDimension=1, style=wx.RA_SPECIFY_ROWS)
        self.noisereduction.SetSelection(0)     # off

        self.attn = wx.RadioBox(self, -1, "Atten.", choices=['off', 'on'],
            majorDimension=1, style=wx.RA_SPECIFY_ROWS)
        self.attn.SetSelection(0)               # off
        
        # Audio link controls = start / stop / reset buttons (no gain settings)
        self.lnk_box = wx.StaticBox(self, -1, "Audio Link Controls")
        self.lnk_panel = wx.Panel(self, -1, style=wx.NO_BORDER)
        self.lnk_start_button = wx.Button(self.lnk_panel, -1, "Start")
        self.lnk_stop_button  = wx.Button(self.lnk_panel, -1, "Stop")
        self.lnk_stop_button.Disable()
        self.lnk_panel.SetBackgroundColour(wx.WHITE)
        self.lnk_status = LNK_STOP
        
        self.agc = wx.RadioBox(self, -1, "AGC", choices=AGCSET, majorDimension=2,
            style=wx.RA_SPECIFY_ROWS)
        if 'agc' not in SUPPORTED:
            self.agc.Disable()
        else:
            self.agc.SetSelection(DEFAULT_AGC)
        
        self.bquit = wx.Button(self,-1,"Quit")
        
        self.rignamebox = wx.StaticBox(self, -1, "Note")
        self.rigname = wx.StaticText(self, -1, "Press any key for audio mute/unmute.")
        self.rigname.SetForegroundColour(wx.Colour(128,0,0))
        
        self.mutebox = wx.StaticBox(self, -1, "Status")
        self.mutetxt = wx.StaticText(self, -1, "Normal")
        self.mutetxt.SetForegroundColour(wx.Colour(0,128,0))
        self.mute = False                               # Want initially unmuted
        err = subprocess.call(self.cmd_unmute, shell=True)   # tell Windows to unmute

        self.infobox = wx.StaticBox(self, -1, "")
        self.info = wx.StaticText(self, -1, 
            wordwrap(
            "W1HQ Remote Receiver System Developed by AA6E"
            " and the ARRL Laboratory, 2013",
            210, wx.ClientDC(self)))
        self.info.SetForegroundColour(wx.Colour(128, 128, 128))
        self.__do_layout()

        # END CONTROL PANEL LAYOUT

        self.first_scan = True
                
        # Command dictionary for Hamlib control interface
        # Each entry defines what has to be done for each supported control:
        # [wx control, stored value, type, rigctl string, (opt) 2nd rigctl command]]
        # type = HL_VALUE or HL_SELECT (knobs or buttons, more or less).
        #   HL_NULL items just send a sanity initialization periodically.

        self.HamlibCommandsDict = {
            "rfgain":   [self.rfgain, -1, HL_VALUE,     "'L RF %f' % float(x/100.)"],
            "afgain":   [self.afgain, -1, HL_VALUE,     "'L AF %f' % float(x/100.)"],
            "vfo":      [self.vfo,    -1, HL_VALUE,     "'F %d' % x"],
            "mode":     [self.mode,   -1, HL_SELECT, 
                            "'M %s %d' % (self.currentMode,self.currentFilter)"],
            "agc":      [self.agc,    -1, HL_SELECT,    "'L AGC %d' % x"],
            "filter":   [self.filter, -1, HL_SELECT,    
                            "'M %s %d' % (self.currentMode,self.currentFilter)"],
            "nb":       [self.nb,     -1, HL_SELECT,    "'U NB %d' % x"],
            "autonotch":[self.autonotch, -1, HL_SELECT, "'U ANF %d' % x"],
            "noisereduction":[self.noisereduction, -1, HL_SELECT, "'U NR %d' % x"],
            "attn":     [self.attn,   -1, HL_SELECT,    "'L ATT %d' % x"],
            "squelch":  [0,           -1, HL_NULL,      "'L SQL 0'"],
            "if":       [0,           -1, HL_NULL,      "'L IF 0'"],
            }
                
        # Set up timer that will control polling of control panel and
        # transmission to remote rig.
        self.timer = wx.Timer(self)
        self.timerPhase = 0
        self.Bind(wx.EVT_TIMER, self.OnTimerEvt, self.timer)
        
        # Bind control/menu events to action routines.
        self.Bind(wx.EVT_BUTTON, self.DoLnkStart, self.lnk_start_button)
        self.Bind(wx.EVT_BUTTON, self.DoLnkStop,  self.lnk_stop_button)
        self.Bind(wx.EVT_BUTTON, self.DoDirectFreqEntry, self.DirectFreqEntryButton)
        self.Bind(wx.EVT_RADIOBOX, self.DoBand, self.band)
        
        self.Bind(wx.EVT_MENU, self.DoQuit, quitter)
        self.Bind(wx.EVT_BUTTON, self.DoQuit, self.bquit)
        self.Bind(wx.EVT_CLOSE, self.DoQuit)        # (event from close box)
        self.Bind(wx.EVT_MENU, self.DoHelp, helper)
        self.Bind(wx.EVT_MENU, self.DoAbout, abouter)
        self.Bind(wx.EVT_CHAR_HOOK, self.DoKey)
        self.Bind(wx.EVT_RADIOBOX, self.DoMode, self.mode)
        self.Bind(wx.EVT_RADIOBOX, self.DoFilter, self.filter)
        
        # Initialize memory for last-used [freq, mode, filter] for each band
        # This memory lets user switch among bands and always come back to prior
        # settings.
        self.Band_Store = 12 * [ [ -1, -1, -1 ] ]       # freq == -1 indicates no value stored
        
        # Start a time to interrupt every ~1000 msec, determining control cycle rate.
        self.timer.Start(milliseconds=TIMER_INTVL, oneShot=False)
        
        return      # Initialization done

    def __do_layout(self):
        # Details of sizing and positioning controls on the frame.
        self.SetFont(wx.Font(8, wx.SWISS, wx.NORMAL, wx.NORMAL, 0, ""))
        
        sizer_vfobox = wx.StaticBoxSizer(self.vfobox, wx.VERTICAL)
        sizer_vfobox.Add(self.vfo, 0, wx.ALL|wx.EXPAND, 5)
        
        ledsSizer = wx.FlexGridSizer(2, 2, 5, 5)
        ledsSizer.Add(self.ledCtrl,  10, 0)
        ledsSizer.Add(self.ledAudio, 10, 0)
        ledsSizer.Add(self.ledCtrlTxt,  1, wx.BOTTOM, border=1)
        ledsSizer.Add(self.ledAudioTxt, 1, wx.BOTTOM, border=1)
        self.indPanel.SetAutoLayout(True)
        self.indPanel.SetSizer(ledsSizer)
        ledsSizer.Fit(self.indPanel)
        
        sizer_ledbox = wx.StaticBoxSizer(self.indBox, wx.VERTICAL)
        sizer_ledbox.Add(self.indPanel, 0, wx.ALL|wx.EXPAND, 5)
        
        sizer_indBox = wx.StaticBoxSizer(self.indBox, wx.VERTICAL)
        sizer_indBox.Add(self.indPanel, 0, wx.ALL|wx.EXPAND, 5)

        sizer_rtbuttons = wx.BoxSizer(wx.VERTICAL)
        sizer_rtbuttons.Add(sizer_ledbox, flag=wx.ALIGN_CENTER|wx.BOTTOM, border=15)
        sizer_rtbuttons.Add(self.DirectFreqEntryButton, flag= wx.ALIGN_CENTER)
        
        sizer_afbox = wx.StaticBoxSizer(self.afbox, wx.VERTICAL)
        sizer_afbox.Add(self.afgain, 0, wx.ALL|wx.EXPAND, 2)
        
        sizer_rfbox = wx.StaticBoxSizer(self.rfbox, wx.VERTICAL)
        sizer_rfbox.Add(self.rfgain, 0, wx.ALL|wx.EXPAND, 2)
        sizer_gainbox = wx.BoxSizer(wx.HORIZONTAL)  # for RF and AF, side by side
        sizer_gainbox.Add(sizer_rfbox, flag=wx.RIGHT, border=10)
        sizer_gainbox.Add(sizer_afbox)
        
        sizer_rignamebox = wx.StaticBoxSizer(self.rignamebox, wx.VERTICAL)
        sizer_rignamebox.Add (self.rigname, 0, wx.ALL|wx.EXPAND, 5)
        
        sizer_mutebox = wx.StaticBoxSizer(self.mutebox, wx.VERTICAL)
        sizer_mutebox.Add (self.mutetxt, 0, wx.ALL|wx.EXPAND, 5)
        
        sizer_infobox = wx.StaticBoxSizer(self.infobox, wx.VERTICAL)
        sizer_infobox.Add (self.info, 0, wx.ALL|wx.EXPAND, 5)
        
        bwagcsizer = wx.BoxSizer(wx.VERTICAL)
        bwagcsizer.Add(self.filter, flag=wx.ALIGN_CENTER)
        bwagcsizer.Add(self.agc, flag=wx.ALIGN_CENTER)
        
        sizer_meter = wx.StaticBoxSizer(self.meterbox, wx.VERTICAL)
        sizer_meter.Add(self.meter, 0, wx.ALL|wx.EXPAND, 5)
        
        lnk_sizer = wx.BoxSizer(wx.HORIZONTAL)
        lnk_sizer.Add(self.lnk_start_button, flag=wx.ALL, border=5)
        lnk_sizer.Add(self.lnk_stop_button, flag=wx.ALL, border=5)
        self.lnk_panel.SetAutoLayout(True)
        self.lnk_panel.SetSizer(lnk_sizer)
        lnk_sizer.Fit(self.lnk_panel)
        
        lnkbox_sizer = wx.StaticBoxSizer(self.lnk_box, wx.VERTICAL)
        lnkbox_sizer.Add(self.lnk_panel)
        
        sizer_autored = wx.BoxSizer(wx.VERTICAL)
        sizer_autored.Add(self.autonotch, flag=wx.ALIGN_CENTER)
        sizer_autored.Add(self.noisereduction, flag=wx.ALIGN_CENTER)

        # The screen is laid out with wx.GridBagSizer, which combines everything on a grid.        
        sizer = wx.GridBagSizer (hgap=5, vgap=10)
        rowct = 0
        sizer.Add(sizer_meter, pos=(rowct,0), span=(1,2), flag=wx.LEFT, border=8)
        sizer.Add(sizer_vfobox, pos=(rowct,2), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(sizer_rtbuttons, pos=(rowct,4), span=(1,2), flag=wx.ALIGN_CENTER)
        rowct += 1
        sizer.Add(sizer_infobox, pos=(rowct,0), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(self.band, pos=(rowct,2), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(self.mode, pos=(rowct,4), span=(1,2), flag=wx.ALIGN_CENTER|wx.RIGHT, border=10)
        rowct += 1
        sizer.Add(bwagcsizer, pos=(rowct,0), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(sizer_gainbox, pos=(rowct,2), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(sizer_autored, pos=(rowct,4), span=(1,2), flag=wx.ALIGN_CENTER)
        rowct += 1
        sizer.Add(self.nb, pos=(rowct,0), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(lnkbox_sizer, pos=(rowct,2), span=(1,2), flag=wx.ALIGN_CENTER)
        sizer.Add(self.bquit, pos=(rowct,4), span=(1,2), flag=wx.ALIGN_CENTER|wx.TOP, border=10)
        rowct += 1
        sizer.Add(self.attn, pos=(rowct,0), flag=wx.ALIGN_TOP|wx.LEFT, border=30)
        sizer.Add(sizer_mutebox, pos=(rowct,1), span=(1,1), flag=wx.ALIGN_CENTER|wx.BOTTOM, border=10)
        sizer.Add(sizer_rignamebox, pos=(rowct,2), span=(1,2), flag=wx.ALIGN_CENTER|wx.BOTTOM, border=10)
        if not WIN32:
            rowct += 1
            # Place a blank item on the next row -- Linux bug (or Ubuntu Unity bug?)
            # Eliminate this if your Linux system puts too much space at bottom.
            sizer.Add( wx.StaticText(self, -1, " "), pos=(rowct,0))
        self.SetSizer(sizer)
        self.Fit()

    def _UpdateCurrent(self):
        # Note current values of a couple of computed variables.  Call when we change
        # mode or filter settings.  Note that filter options differ for different modes.
        # string value
        self.currentMode = MODES[self.mode.GetSelection()]
        # integer Hz
        self.currentFilter = FILTERS[self.mode.GetSelection()][self.filter.GetSelection()]
        
    def DoMode(self, event):
        self._UpdateCurrent()
        
    def DoFilter(self, event):
        self._UpdateCurrent()
    
    def DoAbout(self, event):
        # Format some info about ourselves and display as new dialog.
        info = wx.AboutDialogInfo()
        info.SetDescription(wordwrap(
            "A remote receiver control panel for Amateur Radio."
            "Remote.py uses afxmit/afrecv to handle digital audio "
            "Portaudio or XAudio2 for audio input/output, and Speex for compression. "
            "\n\nDeveloped with support from ARRL Lab. "
            "\n\nDistributed under GPL v3. ",
            350, wx.ClientDC(self)))
        info.SetName("Remote.py")
        info.SetWebSite("http://www.aa6e.net")
        info.SetCopyright("Copyright 2012, 2013 Martin Ewing AA6E")
        info.SetVersion(VERSION)
        wx.AboutBox(info)
 
    def DoHelp(self, event):
        # Some (not very) useful info.  Could be expanded.
        msg =    "Target host = %s"         % self.hostname
        msg += "\nTarget SSH port = %s"     % self.ssh_port
        msg += "\nTarget rigctl port = %s"  % self.control_port
        msg += "\n\nComplete help at http://w1hq-help.dyndns.org ."
        help = wx.MessageDialog(None, msg, "Status & Help", wx.OK)
        help.ShowModal()

    def DoKey(self, event):
        # Responding to any keyboard key press to toggle audio mute/unmute.
        # Problem: If panel does not have focus, key presses are ignored. How to fix?
        if not self.mute:
            err = subprocess.call(self.cmd_mute, shell=True)
            self.mutetxt.SetLabel("Muted")
            self.mutetxt.SetForegroundColour(wx.Colour(128,0,0))
            self.mute = True
        else:
            err = subprocess.call(self.cmd_unmute, shell=True)
            self.mutetxt.SetLabel("Normal")
            self.mutetxt.SetForegroundColour(wx.Colour(0,128,0))
            self.mute = False;

    def DoLnkStart(self, event):
        # Start audio transfer -- start the afrecv process on this machine
        if self.lnk_status == LNK_RUN:
            return      # already running, do nothing more
        # ALSA errors spewed out are a 'feature' of PortAudio and may be ignored.
        # The start_receive process has to start the receiver process (afrecv) and return
        # immediately -- use '&' in Linux or 'start /min afrecv' in Windows.
        print "(start rx)", self.start_receive
        err = subprocess.call(self.start_receive, shell=True)
        if err != 0:
            print "Trouble starting afrecv on local machine..."
            return
        self.lnk_status = LNK_RUN
        self.lnk_start_button.Disable()
        self.lnk_stop_button.Enable()
        self.ledAudio.SetCurrent(1)
        print "Link Started"

    def DoLnkStop(self, event):
        # Stop audio stream -- terminate local afrecv.  
        # But the server (afxmit) stays ready for future restart.
        # This can be a little slow -- use "mute" function for T/R switching.
        if self.lnk_status == LNK_STOP:
            return      # already stopped
        print "(kill rx1)", self.kill_receive1
        err = subprocess.call(self.kill_receive1, shell=True)
        if err != 0:
            print "Trouble stopping recv.c on local machine, phase 1..."
            return
        if self.kill_receive2 <> '':       # requires 2nd whack to kill incoming stream
            print "(kill rx2)", self.kill_receive2
            err = subprocess.call(self.kill_receive2, shell=True)
            if err != 0:
                print "Trouble stopping recv.c on local machine, phase 2..."
                return
        self.lnk_status = LNK_STOP
        self.lnk_start_button.Enable()
        self.lnk_stop_button.Disable()
        self.ledAudio.SetCurrent(0)
        print "Link Stopped"
        
    def DoQuit(self, event):
        # Normal program exit from button, menu, or close box.
        self.timer.Stop()
        if self.lnk_status == LNK_RUN:
            self.DoLnkStop(event)   # Make sure audio is killed gracefully if running
        wx.Exit()
    
    def DoDirectFreqEntry(self, event):
        # Put up dialog asking for input of valid frequency string.
        # Entry must be within current band.
        dlg = wx.TextEntryDialog(self, "Enter frequency in MHz within current band limits", 
                "Frequency Entry", "14.250")
        if dlg.ShowModal() == wx.ID_OK:
            newf_ascii = dlg.GetValue()
            # need to check for conv. errors
            try:
                newf = int(1000000. * float(newf_ascii)) # convert to integer Hz
            except ValueError:
                dlgerr = wx.MessageDialog(self,"** Invalid frequency, try again **", 
                                        "Error", wx.ICON_ERROR|wx.OK)
                dlgerr.ShowModal()
                dlgerr.Destroy()
            else:
                self.vfo.SetValue(newf)

    def DoBand(self, event):
        # Store info about current band: freq / mode / filter, before moving to new band
        self.Band_Store[self.bandno] = [self.vfo.GetValue(), 
                                        self.mode.GetSelection(), 
                                        self.filter.GetSelection() ]
        newband = event.GetInt()
        self.bandno = newband                   # keep track of index - why?
        self.range = BAND_LIMITS[newband]       # kHz
        rghz = [ k*1000 for k in self.range ]   # Hz [ low, high ]
        self.vfo.SetLimit(rghz)
        bs = self.Band_Store[newband]
        if bs[0] > 0:   # If we have prev. stored info for this band, retrieve it.
            self.vfo.SetValue(          bs[0] )
            self.mode.SetSelection(     bs[1] )
            self.filter.SetSelection(   bs[2] )
        else:           # Otherwise, set freq to mid point of band.
            self.mode.SetSelection( MODES.index( BAND_MODES[newband] ) )
            self.vfo.SetValue( sum(rghz)/2 )
        self._UpdateCurrent()    # make sure we are all on the same page

    def OnTimerEvt(self, event):
        # Poll the GUI, looking for control changes, send any to rig.
        # Get rig's signal strength (dB rel. to S9 - by Hamlib) and update
        # the S-meter on screen.
        if self.meter.IsEnabled():
            ntry = 0
            while ntry < SMETER_RETRY_LIMIT:
                err, data = self.myhamlib.recv('l STRENGTH')  # data is str type
                ntry += 1
                if (err == 0) and (len(data) > 0):
                    # Look for good result up to SMETER_RETRY_LIMIT times
                    self.meter.SetSpeedValue(self.fir.filter(
                       max(METER_RANGE[0], min(METER_RANGE[1], int(data))) ))
                    break
            else:
                print "Warning: Excessive null S-meter readings", err, data

        # Every time through this routine causes LED to change state.
        self.ledCtrl.Increment()

        # Every time through, we pick one parameter and zap its stored value.
        # This will force it to update the rig's setting - just in case something
        # is out of sync.
        self.timerPhase = (self.timerPhase + 1) % len(self.HamlibCommandsDict)
        ourKey = self.HamlibCommandsDict.keys()[self.timerPhase]    # the command key
        self.HamlibCommandsDict[ourKey][1] = -1                     # zap its stored value

        # Go look for changed settings, and send updates as needed.
        # The first time we do this takes a few secs, since all controls need setting.
        self.ScanGUI()
        return

    # Test and possibly send command via Hamlib. (used in ScanGUI().)
    # hlc is control list for eg "rfgain"; return True if comm. sent
    def _DoHamlibCommand(self,hlc):
        try:
            # Check if this control is "null".  If so, send a canned command.
            if hlc[0] == 0:             # Works if null, otherwise throw exception.
                exec 'self.myhamlib.send(' + hlc[3] + ')'
                return True
        except:
            pass                        # Normal case - non-null control value
        if not hlc[0].IsEnabled(): 
            return False                # Control not enabled, nothing sent
        if hlc[2] == HL_VALUE:
            x = hlc[0].GetValue()       # Control's current value
        elif hlc[2] == HL_SELECT:
            x = hlc[0].GetSelection()   # Radiobox control setting
        elif hlc[2] == HL_NULL:
            x = 0                       # no control, always 0
        if not x == hlc[1]:
            hlc[1] = x                  # remember val. for next time
            exec 'self.myhamlib.send(' + hlc[3] + ')'
            if len(hlc) >= 5:           # have 2nd Hamlib command?
                exec 'self.myhamlib.send(' + hlc[4] + ')'
            return True                 # mission accomplished
        return False                    # nothing to send this time
        
    def ScanGUI(self):
        for q in self.HamlibCommandsDict:
            self._DoHamlibCommand(self.HamlibCommandsDict[q])
        return
        
# end of class MyFrame

class App(wx.App):
    def OnInit(self):
        frame = MyFrame(None, -1, "")   # instantiate our big frame
        self.SetTopWindow(frame)        # on top of desktop clutter
        frame.Show()                    # let the world see us
        return True

if __name__ == "__main__":  # Normally, we are running as main program, so...
    myApp = App()           # instantiate our wx.App
    myApp.MainLoop()        # run main loop forever

