#!/usr/bin/env python
# -*- coding: utf-8 -*-
# jedit info = :noTabs=true:tabSize=4:

# FSpin.py  1/2013 MSE
# wxWidget class for communications frequency displays

#   Copyright (C) 2012, 2013 Martin S. Ewing, AA6E
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

# to do:
#   Consider alternate fonts - esp. for "0"
#   improve separator ticks - use drawing
#   improve up/down buttons - image based?
#   check scaling of button appearance for small and large size displays
#   assure focus on panel for Windows mousewheel action
#   provide for a decimal point (dot or comma), proper separators (comma or dot)

import wx

DIGIT_COLOUR =      wx.Colour(255, 128, 0)      # orange
DIGIT_BACK_COLOUR = wx.Colour(0,0,50)           # deep blue

# table avoids having to compute 10**n (debatable savings!) It limits maximum
# number of digits.
MAXDIGITS = 30
PWR10 = [ 10**i for i in range(MAXDIGITS)]

UP_BUTTON_BASE = 800    # arbitrary base of button ID numbers
DN_BUTTON_BASE = 700

# Size tweaks
VDELTA = 24
HDELTA = 5

# FSpin is a control designed to display long numbers, such as a transceiver 
# frequency setting, allowing quick adjustment by incrementing or decrementing
# any chosen digit.  Up/down buttons are provided for each digit.  The mousewheel
# may also be used when pointing at a digit.
#
# NB: In Windows, the mousewheel only works when the control has 'focus', e.g.,
# after an up/down button has been used.
#
# Required: Python wxWidgets (wxPython).
# Tested with Python v 2.7 and Ubuntu's python-wxgtk2.8.  See http://wxpython.org.
# For Windows tips, see http://www.aa6e.net/wiki/Python_programs_under_Windows

class FSpin(wx.Control):
    """ FSpin implements a special spin control for integer variables.  The value
        can be adjusted by pointing to a digit and using up/down or mouse wheel 
        to modify."""

    def __init__(self, parent, id, fsize=32, fdigits=8, fextend=1, fvalue=9999999, 
            frange=(100000, 30000000), **kwds):
#       font size and number of digits determine size of this control
        self.fsize = fsize      # fsize = font size for digits
        self.fdigits = fdigits  # fdigits = no. of digits in value
        if self.fdigits <= 0 or self.fdigits >= MAXDIGITS:
            print "FSpin requested %d digits, out of range" % self.fdigits
            raise SystemExit(1)
        self.maxf = PWR10[self.fdigits] - 1    # assuming decimal base
        self.fextend = fextend      # fextend -> show leading zeroes
        self.fvalue = fvalue        # fvalue = initial value
        if self.fvalue < 0 or self.fvalue > self.maxf:
            print "FSpin initial value %d out of range" % self.fvalue
            raise SystemExit(1)
        self.frange = frange    # init. valid freq range
        wx.Control.__init__(self, parent, id, **kwds)
        
        self.panel = wx.Panel(self, -1, style=wx.NO_BORDER)
        self.panel.SetBackgroundColour(DIGIT_BACK_COLOUR)
        self.panel.SetForegroundColour(DIGIT_COLOUR)
        self.panel.SetFont(wx.Font(10, wx.FONTFAMILY_MODERN, wx.NORMAL, wx.BOLD, 0))
        self.text = wx.StaticText(self.panel, -1, "", pos=(0,24), style=wx.ALIGN_RIGHT)
        self.text.SetFont(wx.Font(self.fsize, wx.FONTFAMILY_MODERN, wx.NORMAL, wx.BOLD, 0))
        self.Display()
        # set up rows of up/down buttons for freq adj. (alternative to mousewheel)
        dwidth = float(self.text.GetClientSize()[0]) / self.fdigits
        iwidth = int(dwidth)
        # find vertical position of down buttons, offset 8 more to allow ticks (Win)
        vdnbut = 20 + self.text.GetClientSize()[1] + 8
        for ibut in range(self.fdigits):
            dposup = (int(ibut * dwidth ),  1)
            dposdn = (int(ibut * dwidth), vdnbut)
            wx.Button(self.panel, UP_BUTTON_BASE+ibut, label='+', pos=dposup, 
                size=(iwidth,18))
            wx.Button(self.panel, DN_BUTTON_BASE+ibut, label='-', pos=dposdn, 
                size=(iwidth,18))
        # Draw vertical separators (ticks), assuming decimal pt is at far right?
        # NB: StaticLine won't draw over text widget in Windows
        k = self.fdigits - 3
        while k > 0 :
            wx.StaticLine(self.panel, -1, pos=(k*self.digwidth,vdnbut-10),
                size=(-1,10), style=wx.LI_VERTICAL)
            k -= 3
        self.panel.SetClientSize((self.text.GetClientSize()[0]+HDELTA, vdnbut+VDELTA))
        self.SetMinSize((self.text.GetClientSize()[0]+HDELTA, vdnbut+VDELTA))
        
        self.panel.Bind(wx.EVT_MOUSEWHEEL, self.DoWheel)    # but not Wheel events!
        self.panel.Bind(wx.EVT_BUTTON, self.DoButtons)      # all buttons in panel

    def SetLimit(self, frange):      # establish current valid range (e.g. in band)
        self.frange = frange

    def Validate(self, freq):        # return freq, forced to be in current limits
       return min(max(freq, self.frange[0]), self.frange[1])

    def SetValue(self, fvalue):   # set the current frequency
        self.fvalue = self.Validate(fvalue)
        self.Display()

    def GetValue(self):     # get the current frequency
        return(self.fvalue)
 
    def DoButtons(self, event):
        """ Receive events from all the up/down buttons """
        myid = event.GetId()
        if (myid/100)*100 == UP_BUTTON_BASE :
            self.IncValue(myid-UP_BUTTON_BASE, 1)
        else:
            self.IncValue(myid-DN_BUTTON_BASE, -1)
        self.Display()       

    def Display(self):
        sd = "%d" % self.fdigits        # no. of digits
        if self.fextend:
            sfmt = "%0" + sd + "d"      # show leading 0s
        else:
            sfmt = "%" + sd + "d"       # no leading 0s
        svalue = sfmt % self.fvalue
        self.text.SetLabel(svalue)
        self.cursize = self.text.GetSize()
        self.digwidth = self.cursize[0] / self.fdigits
        # We might want to signal the world via custom event, but no, let's use
        # regular polling instead.

    def IncValue(self, dig_no, delta):
        """ Incr/decr display at dig_no position (leftmost = 0th digit) based on 
            sign of delta. """
        if delta >= 0:
            self.fvalue += PWR10[self.fdigits - dig_no - 1]
        else:
            self.fvalue -= PWR10[self.fdigits - dig_no - 1]
        self.fvalue = self.Validate(self.fvalue)

    def DoWheel(self, event):
        """ Mouse wheel allows rapid incr/decr of digits in number display.
            Point to digit, spin wheel."""
        wheelpos = event.GetPositionTuple()
        wheelpos1 = (wheelpos[0], wheelpos[1]-24)   # convert to text v coord.
        if wheelpos1[0] < self.cursize[0] and wheelpos1[1] < self.cursize[1]:
            k = int(float(self.fdigits * wheelpos1[0]) / self.cursize[0])
            wd = min(self.fdigits-1, k)
            self.IncValue(wd, event.GetWheelRotation()) # add or sub 1 to digit pos.
            self.Display()
        else:
            event.Skip()    # event was outside the panel (why did we get it?)

def SetFont(self, font):     # Choice of font may depend on OS in use.
        self.font = font

# end of class FSpin

# A class test application:

if __name__ == "__main__":
    class MyFrame(wx.Frame):
        def __init__(self, *args, **kwds):
            kwds["style"] = \
                wx.MINIMIZE_BOX|wx.CAPTION|wx.CLOSE_BOX|wx.CLIP_CHILDREN|wx.RESIZE_BORDER
            wx.Frame.__init__(self, *args, **kwds)
            
            self.menubar = wx.MenuBar()
            self.f_menu = wx.Menu()
            quitter = self.f_menu.Append(-1, "Quit", "", wx.ITEM_NORMAL)
            self.menubar.Append(self.f_menu, "File")
            self.SetMenuBar(self.menubar)
            self.SetFont(wx.Font(32, wx.SWISS, wx.NORMAL, wx.NORMAL, 0, ""))
                
            self.SetTitle("FSpin test")

            self.Bind(wx.EVT_MENU, self.DoQuit, quitter)
            self.Bind(wx.EVT_CLOSE, self.DoQuit)    # (event from close box)

            self.myfspin = FSpin(self, -1, fvalue=14070000, fextend=0, fdigits=8, fsize=48 )
        
        def DoQuit(self, event):
            wx.Exit()

# end of class MyFrame

    app = wx.PySimpleApp(0)
    wx.InitAllImageHandlers()
    frame_FSpin = MyFrame(None, -1, "", size=(400,400))
    app.SetTopWindow(frame_FSpin)
    frame_FSpin.Show()
    app.MainLoop()
