#define NUMBER_OF_SAMPLES  11  // an arbitrary number that fits
                               // the number of samples we want 
                               // to average

int sample_store[NUMBER_OF_SAMPLES];
int next_index;

int moving_average(int new value)
{
unsigned int i;
int          accumulator;

    // store the new sample in the ring buffer
    sample_store[next_index] = new_value;
    // set up the pointer to the next entry in the ring buffer
    next_index++;
    if (next_index ==  NUMBER_OF_SAMPLES)
    {
      next_index = 0;
    }
    // we do not care about the sample order
    // because of the commutitive property of addition
    accumulator = 0;
    for (i=0; i < NUMBER_OF_SAMPLES; i++)
    {
      accumulator += sample_store[i];
    }
    // the function result is the new moving average value
    return (accumulator / NUMBER_OF_SAMPLES);
}

#define ARRAY_SIZE   8       // This value must be a power of two
                             // for the logic to work
#define MASK         0x7     // the bit pattern to mask for the array
                             // size
#define BITS_IN_SIZE 3       // the number of bits that correspond to 
                             // the exponent of the array size
int binary_moving_average(int new value)
{
unsigned int i;
int          accumulator;

    // store the new sample in the ring buffer
    sample_store[next_index] = new_value;
    // set up the pointer to the next entry in the ring buffer
    // we save several instructions because the bit masking
    // replaces a compare and several load instructions
    next_index++;
    next_index &= MASK;
    // we do not care about the sample order
    // because of the commutitive property of addition
    accumulator = 0;
    for (i=0; i < NUMBER_OF_SAMPLES; i++)
    {
      accumulator += sample_store[i];
    }
    // the function result is the new moving average value
    // a shift operation takes an order of magnitude fewer
    // CPU cycles compared to an arbitrary divide
    return (accumulator >> BITS_IN_SIZE);
}

#define NUMBER_OF_TAPS      64
int   LMS_weights[NUMBER_OF_TAPS];
int   sample_data[NUMBER_OF_TAPS];

int   newest_sample;

void LMS_weight_adjuster(int new_sample, int error, int gain)
{
int   i, j, k;

   sample_data[newest_sample] = new_sample;
   // our pointer to input data in its circular buffer
   i = newest_sample;
   // our pointer into the weight array
   j = 0;
   // going backwards through an array is 
   // an excellent example of where a do- while
   // construct works best
   do
   {
      LMS_weights[j] = LMS_weights[j] + (2 * error * gain * sample_data[i]);
      j++;
      i--;
      if (i < 0)
      {
         i = NUMBER_OF_TAPS - 1;
      }
   } while (i != newest_sample);
   // point to the next position in the data circular buffer
   newest_sample++;
   if (newest_sample == NUMBER_OF_TAPS)
   {
      newest_sample = 0;
   }
}

