from myhdl import *
from lcdController import lcdController
from format import lcdFormat
from adjCount import adjustCount
from config import *

# lcdController
lcdStart, lcdDone, E, RS, RW = [Signal(bool(INACTIVE_HIGH)) for j in range(5)]
lcdMode = Signal(intbv(0)[2:])
lcdDbyte, lcdP1, lcdP2 = [ Signal(intbv(0)[8:]) for j in range(3)]

# lcdFormat
foStart, foDone = [Signal(bool(INACTIVE_HIGH)) for j in range(2)]
foResult = Signal(intbv(0)[80:])

# adjustCount
acStart, acDone = [Signal(bool(INACTIVE_HIGH)) for j in range(2)]
adjCount = Signal(intbv(0)[COUNT_WIDTH:])

# extractCharacter
exStart = Signal(bool(0))
exIndex = Signal(intbv(0)[4:])
exChar = Signal(intbv(0)[8:])

# udpateDisplayP
msgText = Signal(intbv(0)[80:])		# 8 * the longest text string that must be displayed

def updateDisplayP(clk, reset, start, done, dispMode, modeSw, intervalSw):
	"""
	Update the display with the current value of frequency or period.
	Display size is 2x16

						0123456789012345
		Frequency:		  dd,ddd,ddd Hz
						  dddd Sec

		Period:			dd,ddd,ddd uSec
					    ddd Cycles
	Inputs:
		dispMode:		0-Initialize LCD, 1-update with new count value
		modeSw			0-Display frequency, 1-Display period
	"""

	State = enum('STATE1', 'STATE2', 'STATE2A', 'STATE3', 'STATE3A', 'STATE4', 'STATE5',
				'STATE6', 'STATE7', 'STATE7A', 'STATE8', 'STATE8A', 'STATE9', 'STATE10', 'STATE11',
				'DISPLAY1', 'DISPLAY2', 'DISPLAY3', 'DISPLAY3A', 'DISPLAY4', 'DISPLAY5', 'DISPLAY6', 'IDLE')
	state, dispReturnState = [Signal(State.IDLE) for n in range(2)]

	#						Hz'			#'Sec'
	freqText = tuple([int(0x487a), int(0x536563)])
	freqNchars = tuple([1, 2])	# the number of characters less one
	freqRows = tuple([0, 1])
	freqCols = tuple([13, 8])
	freqValCol = 2
	numFreqText = 2
	#						'uSec'				'Cycles'
	periodText = tuple([int(0x75536563), int(0x4379636c6573)])
	periodNchars = tuple([3, 5])
	periodRows = tuple([0, 1])
	periodCols = tuple([11, 5])
	periodValCol = 0
	numPeriodText = 2
	#							" 0.1"					" 1.0"				"10.0"
	intervalText = tuple([int(0x20302e31), int(0x20312e30), int(0x31302e30)])
	intervalNchars = 3
	intervalValCol = 3
	#						"  1"				" 10"				"100"
	cycleText = tuple([int(0x202031), int(0x203130), int(0x313030)])
	cycleNchars = 2
	cycleValCol = 0

	j = Signal(intbv(0)[3:])
	msgIx = Signal(intbv(0)[4:])
	msgCol = Signal(intbv(0)[5:])
	msgRow = Signal(bool(0))
	msgNchars = Signal(intbv(0)[4:])

	@always(clk.negedge, reset.negedge)
	def logic():
		if reset == ACTIVE_LOW:
			done.next = INACTIVE_HIGH
			state.next = State.IDLE
		elif start == ACTIVE_LOW:
			done.next = INACTIVE_HIGH
			state.next = State.STATE1
		else:
			if state == State.STATE1:
				if dispMode == 0:
					# Initialize the display
					lcdMode.next = 0
					lcdStart.next = ACTIVE_LOW
					state.next = State.STATE2A
				else:
					state.next = State.STATE3

			elif state == State.STATE2A:
				lcdStart.next = INACTIVE_HIGH
				state.next = State.STATE2

			elif state == State.STATE2:
				if lcdDone == ACTIVE_LOW:
					done.next = ACTIVE_LOW
					state.next = State.IDLE

			elif state == State.STATE3:
				# Begin display update by clearing the display (also homes the cursor)
				lcdMode.next = 1		# command
				lcdDbyte.next = 1		# clear display
				lcdStart.next = ACTIVE_LOW
				state.next = State.STATE3A

			elif state == State.STATE3A:
				lcdStart.next = INACTIVE_HIGH
				state.next = State.STATE4

			elif state == State.STATE4:
				if lcdDone == ACTIVE_LOW:
					j.next = 0
					if modeSw == FREQUENCY_MODE:
						state.next = State.STATE5
					else:
						state.next = State.STATE6	# Period

			elif state == State.STATE5:
				if j < numFreqText:
					msgText.next = freqText[j]
					msgIx.next = freqNchars[j]
					msgRow.next = freqRows[j]
					msgCol.next = freqCols[j]
					j.next = j + 1
					dispReturnState.next = State.STATE5
					state.next = State.DISPLAY1
				else:
					state.next = State.STATE7

			elif state == State.STATE6:
				if j < numPeriodText:
					msgText.next = periodText[j]
					msgIx.next = periodNchars[j]
					msgRow.next = periodRows[j]
					msgCol.next = periodCols[j]
					j.next = j + 1
					dispReturnState.next = State.STATE6
					state.next = State.DISPLAY1
				else:
					state.next = State.STATE7

			elif state == State.STATE7:
				acStart.next = ACTIVE_LOW			# adjust the count value
				state.next = State.STATE7A

			elif state == State.STATE7A:
				acStart.next = INACTIVE_HIGH
				state.next = State.STATE8

			elif state == State.STATE8:
				if acDone == ACTIVE_LOW:
					foStart.next = ACTIVE_LOW		# format the value
					state.next = State.STATE8A

			elif state == State.STATE8A:
				foStart.next = INACTIVE_HIGH
				state.next = State.STATE9

			elif state == State.STATE9:
				if foDone == ACTIVE_LOW:
					# display the adjusted and formatted count value
					msgText.next = foResult
					msgRow.next = 0
					if modeSw == FREQUENCY_MODE:
						msgCol.next = freqValCol
					else:
						msgCol.next = periodValCol
					msgIx.next = 9				# this is one less than the number of characters
					dispReturnState.next = State.STATE10
					state.next = State.DISPLAY1

			elif state == State.STATE10:
				# display the interval or number of cycles
				if modeSw == FREQUENCY_MODE:
					msgText.next = intervalText[intervalSw]
					msgIx.next = intervalNchars
					msgCol.next = intervalValCol
				else:
					msgText.next = cycleText[intervalSw]
					msgIx.next = cycleNchars
					msgCol.next = cycleValCol
				msgRow.next = 1
				dispReturnState.next = State.STATE11
				state.next = State.DISPLAY1

			elif state == State.STATE11:
				done.next = ACTIVE_LOW
				state.next = State.IDLE

			# top of character display processing loop
			elif state == State.DISPLAY1:
				# Position the cursor
				lcdMode.next = 1	# command
				lcdDbyte.next = 3	# position cursor command
				lcdP1.next = msgRow
				lcdP2.next = msgCol
				lcdStart.next = ACTIVE_LOW
				state.next = State.DISPLAY2

			elif state == State.DISPLAY2:
				lcdStart.next = INACTIVE_HIGH
				state.next = State.DISPLAY3

			elif state == State.DISPLAY3:		# top of display ch loop
				if lcdDone == ACTIVE_LOW:
					exIndex.next = msgIx		# extract the first character from msgText
					exStart.next = ACTIVE_LOW
					state.next = State.DISPLAY3A

			elif state == State.DISPLAY3A:
				exStart.next = INACTIVE_HIGH
				state.next = State.DISPLAY4

			elif state == State.DISPLAY4:
				lcdMode.next = 2			# write data byte command
				lcdDbyte.next = exChar		# the extracted character
				lcdStart.next = ACTIVE_LOW
				state.next = State.DISPLAY5

			elif state == State.DISPLAY5:
				lcdStart.next = INACTIVE_HIGH
				state.next = State.DISPLAY6

			elif state == State.DISPLAY6:
				if lcdDone == ACTIVE_LOW:
					if msgIx > 0:
						msgIx.next = msgIx - 1
						state.next = State.DISPLAY3
					else:
						state.next = dispReturnState

			elif state == State.IDLE:
				pass
			else:
				state.next = State.IDLE
	return logic

def extractChar(clk, start, k, msg, char):
	@always(clk.negedge)
	def logic():
		if start == ACTIVE_LOW:
			if k == 0:
				char.next = msg[8:0]
			elif k == 1:
				char.next = msg[16:8]
			elif k == 2:
				char.next = msg[24:16]
			elif k == 3:
				char.next = msg[32:24]
			elif k == 4:
				char.next = msg[40:32]
			elif k == 5:
				char.next = msg[48:40]
			elif k == 6:
				char.next = msg[56:48]
			elif k == 7:
				char.next = msg[64:56]
			elif k == 8:
				char.next = msg[72:64]
			else:
				char.next = msg[80:72]
	return logic

def updateDisplay(clk, reset, start, done, dispMode, modeSw, intervalSw, count, lcdBus, E, RS, RW):

	LCD = lcdController(clk, reset, lcdStart, lcdDone, lcdMode, lcdDbyte, lcdP1, lcdP2, lcdBus, E, RS, RW)
	FO = lcdFormat(clk, reset, foStart, foDone, adjCount, foResult)
	AC = adjustCount(clk, reset, acStart, acDone, count, modeSw, intervalSw, adjCount)
	UPD = updateDisplayP(clk, reset, start, done, dispMode, modeSw, intervalSw)
	EXC = extractChar(clk, exStart, exIndex, msgText, exChar)

	return instances()

def testBench():

	clk, reset, start, done, dispMode, modeSw, E, RW, RS = [Signal(bool(INACTIVE_HIGH)) for k in range(9)]
	intervalSw = Signal(intbv(0)[2:])
	count = Signal(intbv(0)[COUNT_WIDTH:])
	lcdBus = Signal(intbv(0)[8:])

	UD = updateDisplay(clk, reset, start, done, dispMode, modeSw, intervalSw, count, lcdBus, E, RS, RW)

	HALF_PERIOD = delay(1)
	@always(HALF_PERIOD)
	def clockGen():
		clk.next = not clk

	@instance
	def testUpdateDisplay():
		reset.next = ACTIVE_LOW
		yield clk.negedge
		reset.next = INACTIVE_HIGH
		yield clk.negedge

		if 0:
			dispMode.next = 0		# initialize display
			start.next = ACTIVE_LOW
			yield clk.negedge
			start.next = INACTIVE_HIGH
			yield clk.negedge
			yield done.negedge
			print
		else:
			count.next = 1000000
			modeSw.next = 0
			intervalSw.next = 2
			dispMode.next = 1			# update
			start.next = ACTIVE_LOW
			yield clk.negedge
			start.next = INACTIVE_HIGH
			yield clk.negedge
			yield done.negedge

		print 'done', now()
		raise StopSimulation

	return instances()

def test1():
	tb = testBench()
	Simulation(tb).run(7000)


def test2():
	clk, reset, start, done, dispMode, modeSw, E, RW, RS = [Signal(bool(0)) for k in range(9)]
	intervalSw = Signal(intbv(0)[2:])
	count = Signal(intbv(0)[COUNT_WIDTH:])
	lcdBus = Signal(intbv(0)[4:])
	toVerilog(updateDisplay, clk, reset, start, done, dispMode, modeSw, intervalSw, count, lcdBus, E, RS, RW)

if __name__ == '__main__':
	test1()