//
//    Copyright 2004, Thomas C. McDermott, N5EG
//    This file is part of VNAR - the Vector Network Analyzer program.
//
//    VNAR is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    VNAR is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with VNAR, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#pragma once

// Utility routines to convert VNA measurements into other
// useful types and coordinates.
// 


#include "stdafx.h"
#using <mscorlib.dll>
#using <System.dll>
using namespace System;
using namespace System::ComponentModel;
using namespace System::IO;
using namespace System::Windows::Forms;

#define	VNA_XTAL_CONST	0xEE9BF				// translate VNA Xtal Osc to DDS divisor
	// the VNA crystal is ~12.000 MHz. It gets multiplied by 24 going to the DDS accumulator 
	// The clock rate is thus approximately 24 * ~12.000 MHz. = ~288.000 MHz.
	// The synthesizer accumulator value needed is:   N = Fdesired * 2^48 / Fclk
	// The constant 2^48/Fclk is 0xEE9BF, but can vary slightly for each oscillator module.

#define RAD2DEGR 180.0/3.14159265358		// Converts radians to degrees
#define DEGR2RAD 3.14159265358/180.0		// Converts degrees to radians

#define MAX_DDS_LEVEL	0xFFF				// Multiplier for DDS Amplitude, unsigned 12-bit int. Maximum = 0xFFF
#define ONEDB_DDS_LEVEL	0xE40				// -1 db below maximum (max * 0.891)
#define TWODB_DDS_LEVEL 0xCB3				// -2 db below maximum (max * 0.794)
#define TWENTYDB_DDS_LEVEL 0x19A			// -20 db below maximum (max * 0.100)

#define THREE_FIVEDB_LEVEL 0xAAA			// -3.5 db below maximum (used for Q-DAC ref level)

	// Convert Magnitude to rectangular Y display coordinate
int ToDisplayRectMag(double magnitude, int height, int dbScaleFactor, int refLevel);

	// Convert Phase to rectangular Y display coordinate
int ToDisplayRectPhs(double phase, int height);

	// Convert Group Delay to rectangular Y display coordinate
int ToDisplayRectGD(double groupdelay, int height, int scaleFactor);

	// Convert Magnitude and Phase to polar (X,Y) display coordinates
void ToDisplayPolar(double magnitude, double phase, int polarRad, int xoffset, int yoffset, int& X, int& Y);

	// Frequency Grid Class
__gc public class FrequencyGrid
{
private:
	int FrequencyIndex __gc[];
//	bool Valid;
	int startFreq, stopFreq;
	int indexer;
	int points;
	float delta;

public:
	FrequencyGrid(int numPoints);		// Constructor, allocate array
	void SetStartF(int start);			// Set start frequency of grid
	void SetStopF(int stop);			// Set stop frequency of grid
	int Frequency(int gridpoint);		// convert gridpoint to it's frequency
	long long int DDS(int Frequency);	// Derive DDS divisor value from Frequency
	int GridPoint(int Frequency);		// convert Frequency to gridpoint
	__property int get_Points();		// get number of points in grid
	__property int get_StartF();		// get start frequency of grid
	__property int get_StopF();			// get stop frequency of grid

private:
	void Build(void);
};


	// Analog Devices AD8302 device-dependent constants
__gc public class Detector
{
public:
	int lowerPhase, upperPhase;			// lower & upper I/Q phase crossover points
	int centerPhase;					// phase midpoint
	int maxMagnitude;					// maximum magnitude reading
	float unitsPerdB;				// magnitude ADC counts per dB. (~57.0)
	int cxFreqIdx;						// frequency index of 360 degrees
	int centerQindex, centerQ2index;
	int centerIindex, centerI2index, centerI3index;
	float quadratureError;				// I-Q phase quadrature error in degrees
	float phaseSlope;					// degrees phase per unit phase count
	double pherror __gc[];				// phase error for each degree
	int MagTable __gc[ , ];				// magnitude raw data table
	bool calibrated;					// true if values have been calibrated

	Detector(void);							// Constructor, allocate default values
	void PhaseCal(int Iphase __gc[], int Qphase __gc[]);	// construct phase points from raw data
	void AmpCal(int Mag __gc[ , ]);		// construct amplitude calibration table from raw data
	double IQtoDegrees(int I, int Q);			// convert I/Q ADC reading to degrees
	double MagTodB(int Freq, int Mag);		// convert detector magnitude count to dB. via piecewise linear fit
};


    // Object to hold Calibration data
__gc public class CalDataSet
{
public:
	// Holds raw and computed calibration data

	Detector* RxDet, * TxDet;		// Holds detector constants

	double EdReal __gc[], EdImag __gc[], EsReal __gc[], EsImag __gc[];
	double EtReal __gc[], EtImag __gc[], ThReal __gc[], ThImag __gc[];
	double S11shortReal __gc[], S11shortImag __gc[];
	double S11openReal __gc[], S11openImag __gc[];
	double S11termReal __gc[], S11termImag __gc[];

	CalDataSet(String *);
		// resolve reflected measured data set to Magnitude and Phase
void ResolveReflPolar(int ReflPI, int ReflPQ, int ReflMI, int Freq, double& rmag, double& rphs);
	// resolve transmitted measured data set to Magnitude and Phase
void ResolveTranPolar(int TranPI, int TranPQ, int TranMI, int Freq, double& rmag, double& rphs);
};

	// Derive error terms from cal measurements
void CalToErrorTerms(CalDataSet* Cal);

// Convert measured S11 into actual S11 via calibration
void CorrectS11(CalDataSet * Cal, int Frequency, double measmag, double measphs, double& rsltmag, double& rsltphs);

// Convert measured S21 into actual S21 via calibration
void CorrectS21(CalDataSet* Cal, int Frequency, double measmag, double measphs, double& rsltmag, double& rsltphs);

// Load calibration data from a previously saved file
void LoadCalDataSet(OpenFileDialog* infile, CalDataSet* Cal);

// Save newly acquired calibration data to a file
void SaveCalDataSet(SaveFileDialog* outfile, CalDataSet* Cal);

// Convert Log Tx Level to linear value needed by DAC register
unsigned short TxLevLinear(int);

// Export S Parameters to a file
void public	 ExportSParams(int format,  FrequencyGrid * FG,
						   double __gc[],  double __gc[],  double __gc[], double __gc[],
						   double __gc[],  double __gc[],  double __gc[], double __gc[]);

// Store S Parameters to temporary storage
void public	 StoreSParams(bool calmode, FrequencyGrid * FG, CalDataSet * CalData,
						   unsigned short __gc[], unsigned short __gc[], unsigned short __gc[],
						   unsigned short __gc[], unsigned short __gc[], unsigned short __gc[],
						   double __gc[],  double __gc[],  double __gc[], double __gc[]);
