{
  Remote control of transmitter and receiver via RS-232 serial I/O port

 Copyright 1999-2000 John B. Stephensen

 This software in source, object or binary form is licensed only for personal non-profit
 educational use in the Amateur Radio Service and the license is not transferrable. The
 software is provided as-is for experimental purposes and the author does not warranty
 its freedom from defects or its suitability for any specific application.

  Written for use with Borland Pascal for Windows version 7.0

  History:
  	6-11-99 created from xcvr()
        7-9-99  add IF Shift
	8-11-99	add transmit control and S-meter, change GetTickCount() to modulo 1 min.
        8-12-99 add T/R control, gain control, remove unneeded DSP-2232 support
        8-13-99 calibrate gain controls
        8-25-99 add noise blanker control
        8-27-99 reduce TX IF gain by 6 dB
        9-8-99  Change to new ALC circuit
        9-19-99 add S-mter display for use with NB and w/o 3-db pad on IF
        3-3-00  add rotator control (Axxx)
        3-14-00 modify to use serial input knob
        3-17-00 use transceiver status returned in getagc()
        4-16-00 20m default freqs.
}

program newxcvr(input,output);

uses WinCrt, WinDos, strings, radioinfo, {satdef,} convinfo, newradiodrv, wcom;

type
  vfo_mode = (rx,tx,both); {if mfc what VFOs controlled by knob}
  transmitter_mode = (off,on,spot);
  knob_state = (ifs,mfc,mgc,mlc,nbl,vox,azc); {what knob controls}

var
  temp: real;
  vfom: vfo_mode; {VFO mode for all VFO pairs}
  step: longint; {VFO tuning knob rate - Hz/count 128 count/rev.}
  tune: array[0..9] of tune_record;
  i,vfo: integer;     {vfo holds index to current VFO}
  rxf,txf: real;      {temp. for TX/RX freq.}
  cf: xcvr_freq;      {buffer for reading corrected freq. from RadioDRV}
  quit: boolean; {flag to exit this program}
  f,offset: real;  {others are temporaries}
  r,ok: boolean;
  command: char;
  old_time,count: longint;			{M001}
  name: Pchar;
  transmit: transmitter_mode;
  use_knob: boolean;  {ignore knob rotations if false}
  ttyin,ttyout: Text; {comm. port fot TNC}
  ttybuffer: string[80]; {TTY output buffer}
  chin: char; {tty input buffer}
  {virtual tty screen}
  x,y: integer; {next character position}
  screen: array[1..9] of string[80];
  adj_tx_ofs: integer;
  avg_agc,avg_alc: real; {moving average of AGC voltage}
  ifgain,clippinglevel,blankinglevel,voxlevel: real; {gain controls}
  agc_offset: integer; {offset oin 0.5 dB units to compensate for gain of xvtr, NB, etc.}
  knob_use: knob_state; {what parameter knob controls}
  azimuth: real; {for rotator control}
  receiver_enabled: boolean; {transceiver status}

const
  zero: double = 0.0;
  max_vfo: integer = 9; {10 VFOs}
  threshold1: real = 0.1; {counts/10 millisec.}
  threshold2: real = 1.0;
  window: longint = 100;
  tx_idle_offset = 0; {none}
  nolines: integer = 9;
  nochars: integer = 72;
  {control char}
  SOH: char = char(1);
  STX: char = char(2);
  ETX: char = char(3);

procedure wait;
var
  i: longint;
begin
  i := 200000;
  while i > 0 do
    i := i - 1;
end;

function asyncin: char;
var
  n: integer;
begin
  n := 10000;
  while (not AsyncCharWaiting) and (n > 0) do n := n-1;
  {return null if no response else return character}
  asyncin := AsyncRecieve;
end;

{get current time as longint}
function CurrentTime:longint;
var
  t: TDateTime;
  dow,ms: Word;
  time: longint;
begin
  GetTime(t.hour,t.min,t.sec,ms);
  GetDate(t.year,t.month,t.day,dow);
  PackTime(t,time);
  CurrentTime := time;
end; {CurrentTime}

{wait a variable number of seconds}
procedure delay(s:integer);
var
  alarm: longint;
  c: char;
begin
  alarm := CurrentTime + s;
  while CurrentTime < alarm do
    if KeyPressed then c := ReadKey;
end; {delay}

procedure writefreq(i: integer);
begin
  {display frequencies}
  write(tune[i].rx/1000000:12:6);
  case tune[i].mode of
    cw: write(' CW  ');
    lsb: write(' LSB ');
    usb: write(' USB ');
    lsbc: write(' AML ');
    usbc: write(' AMU ');
    pm: write(' PSK ');
    rtty: write(' TTY ');
    fm: write(' FSK ');
    pac: write(' PAC ');
  end; {case}
  write(tune[i].tx/1000000:12:6);
  if tune[i].invert
  then write(' I ')
  else write('   ');
  {display antenna polarity}
  case tune[i].polarization of
    rhcp: write(' R ');
    lhcp: write(' L ');
    horizontal: write(' H ');
    vertical: write(' V ');
  end; {case}
end; {writefreq}

{write status line to screen}
procedure writestatus;
begin
  {display transceiver mode}
  gotoxy(1,12);
  write('Xmt:');
  case transmit of
  on: write('ON  ');
  off: write('off ');
  spot:write('SPT');
  end; {case}
  write(' VFO:');
  case vfom of
  rx: write('RX');
  tx: write('TX');
  both: write('XC');
  end;
  {display operator controls}
  if knob_use=ifs then write(' IF:') else write(' if:');
  write(tune[vfo].shift:4);
  write(' BW:');
  write(tune[vfo].filter:4);
  if knob_use=mgc then write(' GN:') else write(' gn:');
  write(ifgain/10:2:0);
  if knob_use=mlc then write(' AL:') else write(' al:');
  write(clippinglevel/10:3:0);
  if knob_use=nbl then write(' NB:') else write(' nb:');
  write(blankinglevel/10:2:0);
  if knob_use=vox then write(' VX:') else write(' vx:');
  if voxlevel < 1000.0
  then write(voxlevel/10:3:0)
  else write('OFF');
  if use_knob
  then write(' Knob:',step:4)
  else write(' Knob: OFF');
  if knob_use=azc then write(' AZ:') else write(' az:');
  write(azimuth:3:0);
  writeln;
end; {writestatus}

procedure smeter(agcv: integer);
var
  agc,i,s,offset:integer;
  dbuv:real;
begin
 {add any offset for NB and 3-dB pad out}
 agc := agcv - 6; {AGC floor = -17 dBuv}
 {take moving average - store as real to prevent rounding error}
 avg_agc := (3*avg_agc + agc)/4;
 {calculate signal strength}
 dbuv := avg_agc/2 - 14; {scale starts at -14 dBuv}
 {display result}
 gotoxy(1,13);
 {dBuv}
 write(dbuv:4:0,' dBuV  S:');
 {bar graph, S1 = -14 dBuV, S9 = +34 dBuV}
 if agc <= 96
 then s := agc div 6 {3 dB per division to S9}
 else s := ((agc-96) div 10) + 16; {5 dB/div. above S9}
 if agc < 0 then write('                                                ');
 if (agc >= 0) and (s <= 16) then
  begin
   case s of
   0:write('1                ');
   1:write('1-               ');
   2:write('1-2              ');
   3:write('1-2-             ');
   4:write('1-2-3            ');
   5:write('1-2-3-           ');
   6:write('1-2-3-4          ');
   7:write('1-2-3-4-         ');
   8:write('1-2-3-4-5        ');
   9:write('1-2-3-4-5-       ');
   10:write('1-2-3-4-5-6      ');
   11:write('1-2-3-4-5-6-     ');
   12:write('1-2-3-4-5-6-7    ');
   13:write('1-2-3-4-5-6-7-   ');
   14:write('1-2-3-4-5-6-7-8  ');
   15:write('1-2-3-4-5-6-7-8- ');
   16:write('1-2-3-4-5-6-7-8-9');
   end; {case}
   write('                                ')
  end; {if}
 if (s >= 17) then
  begin
   write('1-2-3-4-5-6-7-8-9');
   case s of
   17:write('+                  ');
   18:write('+1                 ');
   19:write('+1+                ');
   20:write('+1+2               ');
   21:write('+1+2+              ');
   22:write('+1+2+3             ');
   23:write('+1+2+3+            ');
   24:write('+1+2+3+4           ');
   25:write('+1+2+3+4+          ');
   26:write('+1+2+3+4+5         ');
   27:write('+1+2+3+4+5+        ');
   28:write('+1+2+3+4+5+6       ');
   29:write('+1+2+3+4+5+6+      ');
   30:write('+1+2+3+4+5+6+7     ');
   31:write('+1+2+3+4+5+6+7+    ');
   32:write('+1+2+3+4+5+6+7+8   ');
   33:write('+1+2+3+4+5+6+7+8+  ');
   34:write('+1+2+3+4+5+6+7+8+9 ');
   35:write('+1+2+3+4+5+6+7+8+9+');
   end; {case}
   write('                   ');
  end; {if}
end; {smeter}

procedure alcmeter(alc: integer);
var
  i,s:integer;
  db:real;
begin
 {take moving average - store as real to prevent rounding error}
 avg_alc := (3*avg_alc + alc)/4;
 {calculate signal strength}
 db := avg_alc/6;
 {display result}
 gotoxy(1,13);
 {dB}
 write(db:4:0,' dB  ALC:');
 {bar graph}
 s := alc div 6;
 if alc = 0 then write('                                                ');
 if (alc > 0) and (s <= 16) then
  begin
   case s of
   0:write('0                 ');
   1:write('01                ');
   2:write('012               ');
   3:write('0123              ');
   4:write('01234             ');
   5:write('012345            ');
   6:write('0123456           ');
   7:write('01234567          ');
   8:write('012345678         ');
   9:write('0123456789        ');
   10:write('0123456789 0      ');
   11:write('0123456789 01     ');
   12:write('0123456789 012    ');
   13:write('0123456789 0123   ');
   14:write('0123456789 01234  ');
   15:write('0123456789 012345 ');
   16:write('0123456789 0123456');
   end; {case}
   write('                                ')
  end; {if}
 if (s >= 17) and (s <= 32) then
  begin
   write('0123456789 0123456');
   case s of
   17:write('7                 ');
   18:write('78                ');
   19:write('789               ');
   20:write('789 0             ');
   21:write('789 01            ');
   22:write('789 012           ');
   23:write('789 0123          ');
   24:write('789 01234         ');
   25:write('789 012345        ');
   26:write('789 0123456       ');
   27:write('789 01234567      ');
   28:write('789 012345678     ');
   29:write('789 0123456789    ');
   30:write('789 0123456789 0  ');
   31:write('789 0123456789 01 ');
   32:write('789 0123456789 012');
   end; {case}
   write('                ');
  end; {if}
 if s >= 33 then
  begin
   write('0123456789 0123456789 0123456789 012');
   case s of
   33:write('3                   ');
   34:write('34                  ');
   35:write('345                 ');
   36:write('3456                ');
   37:write('34567               ');
   38:write('345678              ');
   39:write('3456789             ');
   40:write('3456789 0           ');
   41:write('3456789 01          ');
   42:write('3456789 012         ');
   43:write('3456789 0123        ');
   44:write('3456789 01234       ');
   45:write('3456789 012345      ');
   46:write('3456789 0123456     ');
   47:write('3456789 01234567    ');
   48:write('3456789 012345678   ');
   49:write('3456789 0123456789  ');
   50:write('3456789 0123456789 0');
   end; {case}
  end; {if}
end; {alcmeter}

{increment VFO frequency}
procedure adjfreq(vfo: integer; increment: longint);
begin
    {adjust rx freq. if not inhibited}
    if vfom <> tx then
    begin
      tune[vfo].rx := tune[vfo].rx + increment;
      if tune[vfo].rx < tune[vfo].min_rx then tune[vfo].rx := tune[vfo].min_rx;
      if tune[vfo].rx > tune[vfo].max_rx then tune[vfo].rx := tune[vfo].max_rx;
    end;
    {adjust tx freq. if not inhibited}
    if vfom <> rx then
    begin
      if tune[vfo].invert
      then tune[vfo].tx := tune[vfo].tx - increment
      else tune[vfo].tx := tune[vfo].tx + increment;
      if tune[vfo].tx < tune[vfo].min_tx then tune[vfo].tx := tune[vfo].min_tx;
      if tune[vfo].tx > tune[vfo].max_tx then tune[vfo].tx := tune[vfo].max_tx;
    end;
  sendfreq(tune[vfo],transmit = on);
end;

procedure knobmoved(i:integer);
var
 j: integer;
begin
  if i <> 0 then
  case knob_use of
  mfc:  {check tuning knob for rotation and adjust something}
   begin
    if i > 0 then adjfreq(vfo,i*i * step);
    if i = 0 then adjfreq(vfo,i * step);
    if i < 0 then adjfreq(vfo,-i*i * step);
    gotoxy(5,2+vfo);
    writefreq(vfo);
   end;
  ifs:
   begin
    for j := 0 to 9
    do tune[j].shift := tune[j].shift + i;
    writestatus;
    adjfreq(vfo,0);
   end;
  mgc:
   begin
    ifgain := ifgain + i;
    if ifgain < 0 then ifgain := 0; {set stops}
    if ifgain > 800.0 then ifgain := 800.0;
    writestatus;
    setifgain(ifgain/10);
   end;
  mlc:
   begin
    clippinglevel := clippinglevel + i;
    if clippinglevel < 0.0 then clippinglevel := 0.0; {set stops}
    if clippinglevel > 400.0 then clippinglevel := 400.0;
    writestatus;
    setclippinglevel(clippinglevel/10);
   end;
  nbl:
   begin
    blankinglevel := blankinglevel + i;
    if blankinglevel < 0.0 then blankinglevel := 0.0; {set stops}
    if blankinglevel > 630.0 then blankinglevel := 630.0;
    writestatus;
    setblankinglevel(blankinglevel/10);
   end;
  vox:
   begin
    voxlevel := voxlevel + i;
    if voxlevel < 0.0 then voxlevel := 0.0; {set stops}
    if voxlevel > 1000.0 then voxlevel := 1000.0;
    writestatus;
    setvox(voxlevel/10);
   end;
  azc:
   begin
    azimuth := azimuth + i;
    if azimuth < 0.0 then azimuth := 0.0; {set stops}
    if azimuth > 359.0 then azimuth := 359.0;
    writestatus;
   end;
  end; {case}
end; {knobmoved}

{identify input by calling parsing routines}
procedure process_input;
var
  c:char;
  i,agc: integer;
begin
 c := asyncin;
 case c of
 's':
  if getagc(agc,receiver_enabled) then
   if receiver_enabled
    then smeter(agc)
    else alcmeter(agc);
 'k':
 if c = 'k' then
  if getknobstate(i)
   then knobmoved(i);
 end; {case}
end; {process_input}

function GetTickCount: longint;
var
  h,m,s,s100: Word;
begin
  GetTime(h,m,s,s100);
  GetTickCount := round(((h*60.0 + m)*60.0 + s)*100.0 + s100);
end; {GetTickCount}

{idle loop - wait for keystroke, timeout or incoming character}
function idle: integer;
var
  alarm1,alarm2,alarm3: longint; {polling timers}
  count,rate:integer; {polling skip counters}
begin
  {clean up screen}
  gotoxy(1,4+max_vfo);
  {set polling interval}
  alarm1 := GetTickCount+3;
  alarm2 := GetTickCount+6;
  alarm3 := GetTickCount+9;
  count := 0;
  repeat
    {request S-meter readings every 360 msec.}
    if ((GetTickCount-alarm1) > 0) then
    begin
      alarm1 := GetTickCount+9;
      if (count mod 4) = 0 then reqagc;
    end;
    {get knob position every 150 msec.}
    if ((GetTickCount-alarm2) > 0) then
    begin
      count := count+1;
      alarm2 := alarm1+3;
      reqknobstate;
    end;
    {send azumuth every 1.44 sec.}
    if ((GetTickCount-alarm3) > 0) then
    begin
      alarm3 := alarm2+3;
      if (count mod 16) = 0 then send_azimuth(azimuth);
    end;
    {process serial input if it arrives}
    if AsyncCharWaiting then
      if AsyncRecieve = STX then process_input;
  until keypressed;
end; {idle}

{adjust IF shift with tuning knob}
procedure ifshift;
begin
  while not KeyPressed do
  begin
    knob_use := ifs;
    idle; {check tuning knob for rotation and adjust frequency (tune)}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {ifshift}

{adjust IF gain with tuning knob}
procedure adjustifgain;
begin
  while not KeyPressed do
  begin
    knob_use := mgc;
    idle; {check tuning knob for rotation and adjust IF gain}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {adjustifgain}

{adjust clipping level with tuning knob}
procedure adjustclippinglevel;
begin
  while not KeyPressed do
  begin
    knob_use := mlc;
    idle; {check tuning knob for rotation and adjust clipping level}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {adjustclippinglevel}

{adjust noise blanking level with tuning knob}
procedure adjustblankinglevel;
begin
  while not KeyPressed do
  begin
    knob_use := nbl;
    idle; {check tuning knob for rotation and adjust blanking level}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {adjustblankinglevel}

{adjust VOX setpoint}
procedure adjustvox;
begin
  while not KeyPressed do
  begin
    knob_use := vox;
    idle; {check tuning knob for rotation and adjust blanking level}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {adjustvox}

{adjust rotator setpoint}
procedure adjustazimuth;
begin
  while not KeyPressed do
  begin
    knob_use := azc;
    idle; {check tuning knob for rotation and adjust blanking level}
    knob_use := mfc;
  end;
  readkey; {discard keystroke}
end; {adjustazimuth}

function readnum(s: string; var num: real; min,max: real): boolean;
var
  buf: string[80];
  error: integer;
begin
 repeat
 begin
  write(s);
  readln(buf);
  Val(buf,num,error);
 end
 until (error = 0) and (((min <= num) and (num <= max)) or (num = 0));
 readnum := num <> 0;
end;

procedure swap(vf: integer);
var
  i: integer;
  temp: tune_record;
begin
  {swap tx and rx frequencies and band limits}
  temp := tune[vf];
  tune[vf].rx := temp.tx;
  tune[vf].rx_ofs := temp.tx_ofs;
  tune[vf].min_rx := temp.min_tx;
  tune[vf].max_rx := temp.max_tx;
  tune[vf].rxb := temp.txb;
  tune[vf].tx := temp.rx;
  tune[vf].tx_ofs := temp.rx_ofs;
  tune[vf].min_tx := temp.min_rx;
  tune[vf].max_tx := temp.max_rx;
  tune[vf].txb := temp.rxb;
end; {swap}

procedure setvfo(vf: integer; rx,tx: real);
var
  i: integer;
begin
  tune[vf].rx := rx;
  tune[vf].tx := tx;
  {adjust limits and VFO offsets in case new band selected}
  for i := 1 to nolimit do
  begin
    if (tune[vf].rx >= limits[i].min) and (tune[vf].rx <= limits[i].max) then
    begin
      tune[vf].rx_ofs := limits[i].lo;
      tune[vf].min_rx := limits[i].min;
      tune[vf].max_rx := limits[i].max;
      tune[vf].rxb := limits[i].sw;
    end;
    if (tune[vf].tx >= limits[i].min) and (tune[vf].tx <= limits[i].max) then
    begin
      tune[vf].tx_ofs := limits[i].lo;
      tune[vf].min_tx := limits[i].min;
      tune[vf].max_tx := limits[i].max;
      tune[vf].txb := limits[i].sw;
    end;
  end;
end; {setvfo}

procedure setall(v: integer; rx,tx: real; mode: xcvr_mode; filter:integer; invert: boolean; polarization: polarization_type);
begin
  setvfo(v,rx,tx);
  tune[v].mode := mode;
  tune[v].invert := invert;
  tune[v].polarization := polarization;
  tune[v].shift := 0;
  tune[v].filter := filter;
end; {setall}
{
procedure setsat(sat: integer);
var
  i,j: integer;
begin
  j := 0;
  for i := 1 to nofreqs do
    if satfreq[i].s = sat then
    begin
      setall(j,satfreq[i].r*1000000,satfreq[i].t*1000000,
        satfreq[i].m,2400,satfreq[i].i,satfreq[i].p);
      j := j+1;
      if j >= max_vfo then
        j := max_vfo;
    end;
end;
}
{process command to list commands}
procedure listcommands;
begin
  gotoxy(1,14); writeln;
  writeln('E Exit          Z Xmt/Rcv       B Knob on/off   S Step size    ');
  writeln('0-9 Select VFO  X Lock T/R VFOs R Adjust RX VFO T Adjust TX VFO');
  writeln('M Freq.(MHz)    W Swap VFOs     + Up 5 kHz      - Down 5 kHz   ');
  writeln('O Sync. Offset  N Normal Xpndr. I Inv. Xpndr    A Azimuth      ');
  writeln('Y IF shift      GI IF Gain      GA ALC Thresh.  GB Blank. Level');
  writeln('UW USB 2400 Hz  UN USB 1800 Hz  CW CW 500 Hz    CN CW 250 Hz   ');
  writeln('LW LSB 2400 Hz  LN LSB 1800 Hz  PW PSK 500 Hz   PN PSK 250 Hz  ');
  writeln('D RTTY 500 Hz   ');
end; {listcommands}

{main program}
begin
  {enable serial I/O}
  AsyncInit(1);
  {initialize tty screen}
  x := 1; y := 1;
  {set last sample time variable to 5 seconds ago}
  old_time := GetTickCount - 500;
  {initialize VFOs for middle of AO-10 passband}
  for i := 0 to max_vfo do
  begin
    setall(i,14250000,14250000,xcvr_mode(usb),2400,false,lhcp);
  end;
    setall(0,14062000,14250000,xcvr_mode(usb),2400,false,lhcp);
    setall(1,14069150,14250000,xcvr_mode(usb),2400,false,lhcp);
    setall(2,14230000,14250000,xcvr_mode(usb),2400,false,lhcp);
    setall(3,14347000,14250000,xcvr_mode(usb),2400,false,lhcp);
  agc_offset := 0; {assume NB on, no IF pad}
  adj_tx_ofs := 0;
  transmit := off;
  vfom := both;
  use_knob := true;
  knob_use := mfc;
  step := 5; {5 Hz steps for knob}
  ifgain := 800.0; {maximum}
  setifgain(80);
  clippinglevel := 200.0; {midrange}
  setclippinglevel(20);
  voxlevel := 1000.0; {disabled}
  vfo := 0;
  {main loop - this runs forever}
  quit := false;
  while not quit do
  begin
    {set frequency, mode and filter bandwidth}
    adjfreq(vfo,0);
    change_filter(tune[vfo].filter);
    {display VFOs and modes}
    gotoxy(1,1);
    writeln('VFO   RCVR FREQ  Mode  XMTR FREQ    Pol (NEWXCVR v0.8)');
    for i := 0 to max_vfo do
    begin
      if i = vfo then
        write('>')
      else
        write(' ');
      write(i:1,': ');
      writefreq(i);
      writeln;
    end;
    writestatus;
    {test for tuning knob motion}
    if use_knob then idle; {returns if key hit}
    {check for keyboard command}
    gotoxy(1,4+max_vfo);
    command := readkey;
    case command of
    'e','E','q','Q': quit := true;
    '?','h','H': listcommands; {help info.}
    'b','B': use_knob := not use_knob;
    'y','Y': ifshift; {adjust IF shift, gain, clip. level}
    'g','G': {adjust gain of IF, RF clipper or noise blanker}
     begin
       write('Select knob function: I-IF Gain, A-Antenna, B-Blanking, C-ALC, V-VOX');
      case readkey of
       'i','I': adjustifgain;
       'c','C': adjustclippinglevel;
       'a','A': adjustazimuth;
       'b','B': adjustblankinglevel;
       'v','V': adjustvox;
      end; {case}
     end;
    's','S':
      if readnum('Enter step size (Hz): ',temp,-99999,99999)
      then step := round(temp);
    'a','A':
      if readnum('Enter Azimuth (degrees): ',temp,0,359)
      then azimuth := round(temp);
    'c','C':
     begin
      tune[vfo].mode := cw;
      case readkey of
       'w','W': tune[vfo].filter := 500;
       'n','N': tune[vfo].filter := 250;
       else tune[vfo].filter := 250;
      end; {case}
     end;
    'l':
     begin
      tune[vfo].mode := lsb;
      case readkey of
       'w','W': tune[vfo].filter := 2400;
       'n','N': tune[vfo].filter := 1800;
       else tune[vfo].filter := 2400;
      end; {case}
     end;
    'u','U':
     begin
      tune[vfo].mode := usb;
      case readkey of
       'w','W': tune[vfo].filter := 2400;
       'n','N': tune[vfo].filter := 1800;
       else tune[vfo].filter := 2400;
      end; {case}
     end;
    'p','P':
     begin
      tune[vfo].mode := pm;
      case readkey of
       'w','W': tune[vfo].filter := 500;
       'n','N': tune[vfo].filter := 250;
       else tune[vfo].filter := 250;
      end; {case}
     end;
    'd','D':
     begin
      tune[vfo].mode := rtty;
      tune[vfo].filter := 500;
     end;
    'w','W': swap(vfo);
    'z','Z':
    begin
      case transmit of
      off: begin transmit := on; adjfreq(vfo,0); gotoxmt; end;
      on: begin transmit := off; gotorcv; adjfreq(vfo,0); end;
      spot: transmit := off;
      end; {case}
    end;
    'r': vfom := rx;
    't','T': vfom := tx;
    'x','X': vfom := both;
    'n','N': tune[vfo].invert := false;
    'i','I': tune[vfo].invert := true;
    '+': adjfreq(vfo,+5000);
    '-': adjfreq(vfo,-5000);
    'h','H': tune[vfo].polarization := horizontal;
    'v','V': tune[vfo].polarization := vertical;
    'R': tune[vfo].polarization := rhcp;
    'L': tune[vfo].polarization := lhcp;
    '0': vfo := 0;
    '1': vfo := 1;
    '2': vfo := 2;
    '3': vfo := 3;
    '4': vfo := 4;
    '5': vfo := 5;
    '6': vfo := 6;
    '7': vfo := 7;
    '8': vfo := 8;
    '9': vfo := 9;
{    '','':
    begin
      {give user list of sattelites - 4 per line}
{      for i := 0 to nosats do
      begin
        write(i:2,'.',satlist[i]:6,' ');
        if (i mod 4) = 3 then
          writeln;
      end;
      if (i mod 4) <> 3 then
        writeln;
      {select one}
{      readnum('Enter sattelite to receive: ',f,0,nosats);
      setsat(round(f));
    end; {sat. sel.}
    'm','M':
    begin
      repeat
        r := readnum('Enter Frequency (MHz): ',temp,1.8,54);
        f := temp * 1000000.0;
        ok := false;
        for i := 1 to nolimit do
          ok := ok or ((f >= limits[i].min) and (f <= limits[i].max));
      until ok;
      if r then
      begin
        rxf := tune[vfo].rx;
        txf := tune[vfo].tx;
        if vfom <> tx then
          rxf := f;
        if vfom <> rx then
          txf := f;
        setvfo(vfo,rxf,txf);
      end; {if}
    end;
    'o','O': {adjust all transmitter offsets to current VFO}
    begin
      {get offset}
      if tune[vfo].invert
      then offset := tune[vfo].tx + tune[vfo].rx
      else offset := tune[vfo].tx - tune[vfo].rx;
      {adjust other VFOs}
      for i := 0 to 9 do
        if (i <> vfo)
          and (tune[i].rxb = tune[vfo].rxb)
          and (tune[i].txb = tune[vfo].txb)
          and (tune[i].invert = tune[vfo].invert) then
        case vfom of
        tx:
        begin
          if tune[vfo].invert
          then tune[i].tx := offset - tune[i].rx
          else tune[i].tx := tune[i].rx + offset;
          tune[i].invert := tune[vfo].invert;
        end;
        rx:
        begin
          if tune[vfo].invert
          then tune[i].rx := offset - tune[i].tx
          else tune[i].rx := tune[i].tx - offset;
          tune[i].invert := tune[vfo].invert;
        end;
        end; {case}
      vfom := both;
    end;
    end; {case}
    {clean up screen}
    gotoxy(1,4+max_vfo);
    for i := 1 to 2 do
      writeln('                                                                ');
  end;
  AsyncShutdown; {disable COM port interrupt}
end.
{
 Copyright 1999-2000 John B. Stephensen
}