{
  procedures to drive DDS/PLL synthesizer for HF receiver control via serial I/O port

 Copyright 1999-2000 John B. Stephensen

 This software in source, object or binary form is licensed only for personal non-profit
 educational use in the Amateur Radio Service and the license is not transferrable. The
 software is provided as-is for experimental purposes and the author does not warranty
 its freedom from defects or its suitability for any specific application.

  Written for use with Borland Pascal for Windows version 7.0

  6-11-99    derive from hfrxcon()
  6-25-99    add BFO control
  6-30-99    add time delay between commands
  8-11-99    add AGC and filter control
  8-12-99    add T/R control
  8-25-99    add noise blanker control
  3-14-00    add routines to read knob and use SOH in commands
  3-17-00    added transceiver status to readagc1() and added voxlevel()
  4-6-00     change BFO offset to +/-1350 Hz for SSB
}
unit ddsctl;

interface

uses radioinfo, wcom;

procedure rxfreq(frequency: longint; mode: xcvr_mode; shift: longint);
procedure txfreq(frequency: longint; mode: xcvr_mode; shift: longint);
procedure selectfilter(f: integer);
procedure set_az(a:integer);
procedure readagc0;
function readagc1(var agc: integer; var rx: boolean): boolean;
procedure readknob0;
function readknob1(var i: integer): boolean;
procedure ifgain(a: integer);
procedure clippinglevel(a: integer);
procedure blankinglevel(a: integer);
procedure voxlevel(a: integer);
procedure transmit;
procedure receive;

implementation

uses sio;

var
  sendref: integer;

const
  {1st IF frequency}
  if_freq: Extended = 9000000.0;
  {frequency standard value - 30 MHz}
  freq_std: Extended = 30000000.0;
  {maximum value of DDS register = 2**32}
  dds_max: Extended = 4294967296.0;
  {offsets for different modes}
  ofs_cw: Extended = 0.0;      {tune to carrier freq.}
  bfo_cw: Extended = +750.0;   {750 Hz tone}
  ofs_rtty: Extended = -85.0;   {tune to mark frequency (+85Hz)}
  bfo_rtty: Extended = -1360.0; {1275/1445 Hz tones}
  ofs_psk: Extended = 0.0;     {indicate actual signal frequency}
  bfo_psk: Extended = +1000.0; {1000 Hz tone}
  {set SSB carrier for 250-2750 Hz audio with 2500 Hz filter}
  ofs_lsb: Extended = -1350.0; {1350 Hz below IF as signal inverted}
  ofs_usb: Extended = +1350.0; {1350 Hz above IF as signal inverted}
  bfo_lsb: Extended = -1350.0; {1350 Hz below IF as signal inverted}
  bfo_usb: Extended = +1350.0; {1350 Hz above IF as signal inverted}
  {AM signals cenered in 5 KHz passband}
  ofs_am: Extended = 0.0;
  {set AM synchronous detector carrier 250 Hz inside 5 KHz passband}
  ofs_aml: Extended = -2250.0; {2250 Hz below IF as signal inverted}
  ofs_amu: Extended = +2250.0; {2250 Hz above IF as signal inverted}
  {control char}
  SOH: char = char(1);
  STX: char = char(2);
  ETX: char = char(3);

procedure wait;
var
  i: longint;
begin
  i := 5000;
  while i > 0 do
    i := i - 1;
end;

{send characters to transceiver}
procedure sendstring(s: string);
var
  i,j,k:integer;
begin
  AsyncTransmit(char(255)); {pad}
  AsyncTransmit(char(255)); {pad}
  AsyncTransmit(#1); {SOH}
  i := 1;
  j := Length(s);
  k := 3; {parity, including ETX}
  while i <= j do
  begin
    k := k xor integer(s[i]);
    AsyncTransmit(s[i]); {send data}
    i := i+1;
  end;
  AsyncTransmit(#3); {ETX}
  AsyncTransmit(char(k)); {send longitudinal parity}
  wait;
end;

{convert binary value to hex and transmit}
procedure sndfrq(c:char; f:longint);
var
  i:integer;
  s:string[11];
begin
  s[0] := char(9);
  {insert command char}
  s[1] := c;
  {insert frequency nibbles, LS first}
  for i := 2 to 9 do
  begin
    s[i] := char((f and 15) + 48);
    f := f div 16;
  end;
  s[10] := char(0);
  {writeln('sending: ',s);}
  sendstring(s);
end; {sndfrq}

function asyncin: char;
var
  n: integer;
begin
  n := 10000;
  while (not AsyncCharWaiting) and (n > 0) do n := n-1;
  {return null if no response else return character}
  asyncin := AsyncRecieve;
end;

{position antenna}
procedure set_az(a:integer);
var
  s: string[13];
begin
  {send command}
  s[0] := char(11); {length}
  s[1] := char(13);
  s[2] := 'A'; {command byte}
  s[3] := 'Z'; {command byte}
  s[4] := char(integer('0') + (a div 100));
  s[5] := char(integer('0') + ((a div 10) mod 10));
  s[6] := char(integer('0') + (a mod 10));
  s[7] := ' '; {command byte}
  s[8] := 'E'; {command byte}
  s[9] := 'L'; {command byte}
  s[10] := '0'; {point at horizon}
  s[11] := char(13);
  s[12] := char(0);
  sendstring(s);
end; {set_az}

{request AGC voltage}
procedure readagc0;
var
  s: string[11];
begin
  {send command}
  s[0] := char(1); {length}
  s[1] := 'S'; {command byte}
  s[2] := char(0);
  sendstring(s);
end; {readagc0}

{process received AGC voltage}
function readagc1(var agc: integer; var rx:boolean): boolean;
var
  c1,c2,c3: char;
  n:integer;
begin
  {get response}
    c1 := asyncin; {MS nibble}
    c2 := asyncin; {LS nibble}
    c3 := asyncin; {status byte};
    {combine nibbles for AGC value}
    agc := 16*(integer(c1) and 15) + (integer(c2) and 15);
    {signal if in receive mode}
    rx := (integer(c3) and 1) = 1;
    readagc1 := true;
end; {readagc1}

{request knob position}
procedure readknob0;
var
  s: string[11];
begin
  {send command}
  s[0] := char(1); {length}
  s[1] := 'K'; {command byte}
  s[2] := char(0);
  sendstring(s);
end; {readknob0}

{process received knob position}
function readknob1(var i: integer): boolean;
var
  c1,c2,c3,c4: char;
  n:integer;
begin
  {get response}
    c1 := asyncin; {MS nibble}
    c2 := asyncin;
    {calculate ADC output value}
    n := 16*(integer(c1) and 15) + (integer(c2) and 15);
    if n > 127
    then i := -(256-n)
    else i := n;
    readknob1 := asyncin = ETX;
    asyncin; {discard LRC}
end; {readknob1}

{select filter}
procedure selectfilter(f:integer);
var
 s: string[11];
begin
  s[0] := char(2); {string length}
  s[1] := 'B'; {select filter command}
  s[2] := char(8); {2400 Hz filter is default}
  case f of
   250: s[2] := char(1);
   500: s[2] := char(2);
   1800: s[2] := char(4);
   2400: s[2] := char(8);
  end; {case}
  s[3] := char(0);
  sendstring(s);
end; {selectfilter}

{set minimum IF gain}
procedure ifgain(a:integer);
var
 s: string[11];
begin
  s[0] := char(2); {string length}
  s[1] := 'G'; {IF gain command}
  s[2] := char(a); {convert integer to character}
  s[3] := char(0);
  sendstring(s);
end; {ifgain}

{set RF clipper level}
procedure clippinglevel(a:integer);
var
 s: string[11];
begin
  s[0] := char(2); {string length}
  s[1] := 'C'; {clipping level command}
  s[2] := char(a); {convert integer to character}
  s[3] := char(0);
  sendstring(s);
end; {clippinglevel}

{set noise blanking level}
procedure blankinglevel(a:integer);
var
 s: string[11];
begin
  s[0] := char(2); {string length}
  s[1] := 'Z'; {blanking level command}
  s[2] := char(a); {convert integer to character}
  s[3] := char(0);
  sendstring(s);
end; {blankinglevel}

{set VOX level}
procedure voxlevel(a:integer);
var
 s: string[11];
begin
  s[0] := char(2); {string length}
  s[1] := 'V'; {blanking level command}
  s[2] := char(a); {convert integer to character}
  s[3] := char(0);
  sendstring(s);
end; {voxlevel}

{go to transmit mode}
procedure transmit;
var
  s: string[11];
begin
  {send command}
  s[0] := char(1); {length}
  s[1] := 'X'; {command byte}
  s[2] := char(0);
  sendstring(s);
end; {transmit}

{go to receive mode}
procedure receive;
var
  s: string[11];
begin
  {send command}
  s[0] := char(1); {length}
  s[1] := 'N'; {command byte}
  s[2] := char(0);
  sendstring(s);
end; {receive}

{set frequency - input is 0-54000000 Hz}
procedure rxfreq(frequency: longint; mode: xcvr_mode; shift: longint);
var
  freq: Extended;
  dds_reg: longint; {DDS phase accumulator increment}
  dds_freq: Extended;
  bfo: Extended;
  bfo_reg: longint;
begin
  if (frequency < 0) or (frequency >= 54000000)
  then writeln('ERROR: Receiver frequency ',frequency/1000000:10:6,' MHz!');
  case mode of
  lsb: freq := frequency + ofs_lsb;
  usb: freq := frequency + ofs_usb;
  cw: freq := frequency + ofs_cw;
  rtty: freq := frequency + ofs_rtty;
  pac: freq := frequency + ofs_usb;
  pm: freq := frequency + ofs_psk;
  end; {case}
  {add or subtract IF freq.}
  if freq < 40000000
  then freq := freq + if_freq + shift
  else freq := freq - if_freq - shift;
  {DDS frequency 1/5 LO frequency}
  dds_freq := freq/5.0;
  dds_reg := round(dds_max*dds_freq/freq_std);
  {send values to synthesizer}
  sndfrq('F',dds_reg);
  {writeln('RX LO:',dds_reg);}
  {set BFO frequency}
  case mode of
  lsb: bfo := if_freq + bfo_lsb;
  usb: bfo := if_freq + bfo_usb;
  cw: bfo := if_freq + bfo_cw;
  rtty: bfo := if_freq + bfo_rtty;
  pac: bfo := if_freq + bfo_usb;
  pm: bfo := if_freq + bfo_psk;
  end; {case}
  {add IF shift}
  bfo := bfo + shift;
  {send values to synthesizer}
  bfo_reg := round(dds_max*bfo/freq_std);
  sndfrq('f',bfo_reg);
  {writeln('RX BFO:',bfo_reg);}
end;

{set frequency - input is 0-54000000 Hz}
procedure txfreq(frequency: longint; mode: xcvr_mode; shift: longint);
var
  freq: Extended;
  dds_reg: longint; {DDS phase accumulator increment}
  dds_freq: Extended;
  bfo: Extended;
  bfo_reg: longint;
begin
  if (frequency < 0) or (frequency >= 54000000)
  then writeln('ERROR: Transmitter frequency ',frequency/1000000:10:6,' MHz!');
  case mode of
  lsb: freq := frequency + ofs_lsb;
  usb: freq := frequency + ofs_usb;
  cw: freq := frequency + ofs_cw;
  rtty: freq := frequency + ofs_rtty;
  pac: freq := frequency + ofs_usb;
  pm: freq := frequency + ofs_psk;
  end; {case}
  {add or subtract IF freq. and IF shift}
  if freq < 40000000.0
  then freq := freq + if_freq + shift
  else freq := freq - if_freq - shift;
  {DDS frequency 1/5 LO frequency}
  dds_freq := freq/5.0;
  dds_reg := round(dds_max*dds_freq/freq_std);
  {send values to synthesizer}
  sndfrq('F',dds_reg);
  {writeln('TX LO:',dds_reg);}
  {set BFO frequency}
  case mode of
  lsb: bfo := if_freq + bfo_lsb;
  usb: bfo := if_freq + bfo_usb;
  cw: bfo := if_freq + bfo_cw;
  rtty: bfo := if_freq + bfo_rtty;
  pac: bfo := if_freq + bfo_usb;
  pm: bfo := if_freq + bfo_psk;
  end; {case}
  {add IF shift}
  bfo := bfo + shift;
  {send values to synthesizer}
  bfo_reg := round(dds_max*bfo/freq_std);
  sndfrq('f',bfo_reg);
  {writeln('TX BFO:',bfo_reg);}
end;
{Copyright 1999-2000 John B. Stephensen}
end.
