/*******************************************************************************
* File Name: UART.h
* Version 1.50
*
* Description:
*  Contains the function prototypes and constants available to the UART
*  user module.
*
* Note:
*
********************************************************************************
* Copyright 2008-2010, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
********************************************************************************/


#include "cytypes.h"
#include "cyfitter.h"

#if !defined(CY_UART_UART_H)
#define CY_UART_UART_H


/***************************************
* Conditional Compilation Parameters
***************************************/

#define UART_RX_ENABLED                     (0u)
#define UART_TX_ENABLED                     (1u)
#define UART_HD_ENABLED                     (0u)
#define UART_RX_INTERRUPT_ENABLED           (0u)
#define UART_TX_INTERRUPT_ENABLED           (1u)
#define UART_INTERNAL_CLOCK_USED            (1u)
#define UART_RXHW_ADDRESS_ENABLED           (1u)
#define UART_OVER_SAMPLE_COUNT              (8u)
#define UART_PARITY_TYPE                    (0u)
#define UART_PARITY_TYPE_SW                 (0u)
#define UART_BREAK_DETECT                   (1u)
#define UART_BREAK_BITS_TX                  (13u)
#define UART_BREAK_BITS_RX                  (13u)
#define UART_TXCLKGEN_DP                    (1u)
#define UART_USE23POLLING                   (1u)

/* PSoC3 ES2 or early */
#define UART_PSOC3_ES2  ((CYDEV_CHIP_MEMBER_USED == CYDEV_CHIP_MEMBER_3A) && \
                                     (CYDEV_CHIP_REVISION_USED <= CYDEV_CHIP_REVISION_3A_ES2))

/* PSoC5 ES1 or early */
#define UART_PSOC5_ES1  ((CYDEV_CHIP_MEMBER_USED == CYDEV_CHIP_MEMBER_5A) && \
                                     (CYDEV_CHIP_REVISION_USED <= CYDEV_CHIP_REVISION_5A_ES1))
/* PSoC3 ES3 or later */
#define UART_PSOC3_ES3  ((CYDEV_CHIP_MEMBER_USED == CYDEV_CHIP_MEMBER_3A) && \
                                     (CYDEV_CHIP_REVISION_USED > CYDEV_CHIP_REVISION_3A_ES2))

/* PSoC5 ES2 or later */
#define UART_PSOC5_ES2  ((CYDEV_CHIP_MEMBER_USED == CYDEV_CHIP_MEMBER_5A) && \
                                     (CYDEV_CHIP_REVISION_USED > CYDEV_CHIP_REVISION_5A_ES1))

#if(UART_PSOC3_ES2 && (UART_RX_INTERRUPT_ENABLED || UART_TX_INTERRUPT_ENABLED))
    #include <intrins.h>
    #define UART_ISR_PATCH() _nop_(); _nop_(); _nop_(); _nop_(); _nop_(); _nop_(); _nop_(); _nop_();
#endif /* End PSOC3_ES2 */

#ifdef UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG
    #define UART_CONTROL_REG_REMOVED            (0u)
#else
    #define UART_CONTROL_REG_REMOVED            (1u)
#endif /* End UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG */


/***************************************
*      Data Struct Definition
***************************************/

/* Sleep Mode API Support */
typedef struct _UART_backupStruct
{
    uint8 enableState;

    #if(UART_CONTROL_REG_REMOVED == 0u)
        uint8 cr;            
    #endif /* End UART_CONTROL_REG_REMOVED */    
    #if( (UART_RX_ENABLED) || (UART_HD_ENABLED) )
        uint8 rx_period;
        #if (UART_PSOC3_ES2 || UART_PSOC5_ES1) /* PSoC3 ES2 or early, PSoC5 ES1*/
            uint8 rx_mask;
            #if (UART_RXHW_ADDRESS_ENABLED)
                uint8 rx_addr1;
                uint8 rx_addr2;
            #endif /* End UART_RXHW_ADDRESS_ENABLED */
        #endif /* End PSOC3_ES2 || PSOC5_ES1 */
    #endif  /* End (UART_RX_ENABLED) || (UART_HD_ENABLED)*/

    #if(UART_TX_ENABLED)
        #if(UART_TXCLKGEN_DP)
            uint8 tx_clk_ctr;
            #if (UART_PSOC3_ES2 || UART_PSOC5_ES1) /* PSoC3 ES2 or early, PSoC5 ES1*/
                uint8 tx_clk_compl;
            #endif  /* End PSOC3_ES2 || PSOC5_ES1 */
        #else
            uint8 tx_period;
        #endif /*End UART_TXCLKGEN_DP */
        #if (UART_PSOC3_ES2 || UART_PSOC5_ES1) /* PSoC3 ES2 or early, PSoC5 ES1*/
            uint8 tx_mask;
        #endif  /* End PSOC3_ES2 || PSOC5_ES1 */
    #endif /*End UART_TX_ENABLED */
} UART_BACKUP_STRUCT;


/***************************************
*       Function Prototypes
***************************************/

void UART_Start(void);
void UART_Stop(void) ;
uint8 UART_ReadControlRegister(void) ;
void UART_WriteControlRegister(uint8 control) ;

void UART_Init(void) ;
void UART_Enable(void) ;
void UART_SaveConfig(void);
void UART_RestoreConfig(void);
void UART_Sleep(void);
void UART_Wakeup(void);

/* Only if RX is enabled */
#if( (UART_RX_ENABLED) || (UART_HD_ENABLED) )

    #if(UART_RX_INTERRUPT_ENABLED)
        void  UART_EnableRxInt(void) ;
        void  UART_DisableRxInt(void) ;
        CY_ISR_PROTO(UART_RXISR);
    #endif /* UART_RX_INTERRUPT_ENABLED */

    #if (UART_RXHW_ADDRESS_ENABLED)
        void UART_SetRxAddressMode(uint8 addressMode) 
                                                            ;
        void UART_SetRxAddress1(uint8 address) ;
        void UART_SetRxAddress2(uint8 address) ;
    #endif /* End UART_RXHW_ADDRESS_ENABLED */

    void  UART_SetRxInterruptMode(uint8 intSrc) ;
    uint8 UART_ReadRxData(void);
    uint8 UART_ReadRxStatus(void) ;
    uint8 UART_GetChar(void);
    uint16 UART_GetByte(void);
    uint8 UART_GetRxBufferSize(void);
    void UART_ClearRxBuffer(void);

    /* Obsolete functions, defines for backward compatible */
    #define UART_GetRxInterruptSource   UART_ReadRxStatus

#endif /* End (UART_RX_ENABLED) || (UART_HD_ENABLED) */

/* Only if TX is enabled */
#if(UART_TX_ENABLED || UART_HD_ENABLED)

    #if(UART_TX_INTERRUPT_ENABLED)
        void UART_EnableTxInt(void) ;
        void UART_DisableTxInt(void) ;
        CY_ISR_PROTO(UART_TXISR);
    #endif /* UART_TX_INTERRUPT_ENABLED */

    void UART_SetTxInterruptMode(uint8 intSrc) ;
    void UART_WriteTxData(uint8 txDataByte);
    uint8 UART_ReadTxStatus(void) ;
    void UART_PutChar(uint8 txDataByte);
    void UART_PutString(uint8* string);
    void UART_PutArray(uint8* string, uint8 byteCount);
    void UART_PutCRLF(uint8 txDataByte);
    void UART_ClearTxBuffer(void);
    void UART_SetTxAddressMode(uint8 addressMode) ;
    void UART_SendBreak(uint8 retMode);
    uint8 UART_GetTxBufferSize(void);
    /* Obsolete functions, defines for backward compatible */
    #define UART_PutStringConst         UART_PutString
    #define UART_PutArrayConst          UART_PutArray
    #define UART_GetTxInterruptSource   UART_ReadTxStatus
    
#endif /* End UART_TX_ENABLED || UART_HD_ENABLED */

#if(UART_HD_ENABLED)
    void UART_LoadRxConfig(void) ;
    void UART_LoadTxConfig(void) ;
#endif /* End UART_HD_ENABLED */


/***************************************
*          API Constants
***************************************/
/* Parameters for SetTxAddressMode API*/
#define UART_SET_SPACE                              (0x00u)
#define UART_SET_MARK                               (0x01u)

/* Status Register definitions */
#if( (UART_TX_ENABLED) || (UART_HD_ENABLED) )
    #if(UART_TX_INTERRUPT_ENABLED)
        #define UART_TX_VECT_NUM                UART_TXInternalInterrupt__INTC_NUMBER
        #define UART_TX_PRIOR_NUM               UART_TXInternalInterrupt__INTC_PRIOR_NUM
    #endif /* UART_TX_INTERRUPT_ENABLED */
    #if(UART_TX_ENABLED) 
        #define UART_TX_STS_COMPLETE_SHIFT          (0x00u)
        #define UART_TX_STS_FIFO_EMPTY_SHIFT        (0x01u)
        #define UART_TX_STS_FIFO_FULL_SHIFT         (0x02u)
        #define UART_TX_STS_FIFO_NOT_FULL_SHIFT     (0x03u)
    #endif /* UART_TX_ENABLED */
    #if(UART_HD_ENABLED) 
        #define UART_TX_STS_COMPLETE_SHIFT          (0x00u)
        #define UART_TX_STS_FIFO_EMPTY_SHIFT        (0x01u)
        #define UART_TX_STS_FIFO_FULL_SHIFT         (0x05u)  /*needs MD=0*/
        #define UART_TX_STS_FIFO_NOT_FULL_SHIFT     (0x03u)
    #endif /* UART_HD_ENABLED */
    #define UART_TX_STS_COMPLETE                (0x01u << UART_TX_STS_COMPLETE_SHIFT)
    #define UART_TX_STS_FIFO_EMPTY              (0x01u << UART_TX_STS_FIFO_EMPTY_SHIFT)
    #define UART_TX_STS_FIFO_FULL               (0x01u << UART_TX_STS_FIFO_FULL_SHIFT)
    #define UART_TX_STS_FIFO_NOT_FULL           (0x01u << UART_TX_STS_FIFO_NOT_FULL_SHIFT)
#endif /* End (UART_TX_ENABLED) || (UART_HD_ENABLED)*/

#if( (UART_RX_ENABLED) || (UART_HD_ENABLED) )
    #if(UART_RX_INTERRUPT_ENABLED)
        #define UART_RX_VECT_NUM                UART_RXInternalInterrupt__INTC_NUMBER
        #define UART_RX_PRIOR_NUM               UART_RXInternalInterrupt__INTC_PRIOR_NUM
    #endif /* UART_RX_INTERRUPT_ENABLED */
    #define UART_RX_STS_MRKSPC_SHIFT            (0x00u)
    #define UART_RX_STS_BREAK_SHIFT             (0x01u)
    #define UART_RX_STS_PAR_ERROR_SHIFT         (0x02u)
    #define UART_RX_STS_STOP_ERROR_SHIFT        (0x03u)
    #define UART_RX_STS_OVERRUN_SHIFT           (0x04u)
    #define UART_RX_STS_FIFO_NOTEMPTY_SHIFT     (0x05u)
    #define UART_RX_STS_ADDR_MATCH_SHIFT        (0x06u)
    #define UART_RX_STS_SOFT_BUFF_OVER_SHIFT    (0x07u)

    #define UART_RX_STS_MRKSPC                  (0x01u << UART_RX_STS_MRKSPC_SHIFT)
    #define UART_RX_STS_BREAK                   (0x01u << UART_RX_STS_BREAK_SHIFT)
    #define UART_RX_STS_PAR_ERROR               (0x01u << UART_RX_STS_PAR_ERROR_SHIFT)
    #define UART_RX_STS_STOP_ERROR              (0x01u << UART_RX_STS_STOP_ERROR_SHIFT)
    #define UART_RX_STS_OVERRUN                 (0x01u << UART_RX_STS_OVERRUN_SHIFT)
    #define UART_RX_STS_FIFO_NOTEMPTY           (0x01u << UART_RX_STS_FIFO_NOTEMPTY_SHIFT)
    #define UART_RX_STS_ADDR_MATCH              (0x01u << UART_RX_STS_ADDR_MATCH_SHIFT)
    #define UART_RX_STS_SOFT_BUFF_OVER          (0x01u << UART_RX_STS_SOFT_BUFF_OVER_SHIFT)
    #define UART_RX_HW_MASK                     (0x7Fu)
#endif /* End (UART_RX_ENABLED) || (UART_HD_ENABLED) */

/* Control Register definitions */
#define UART_CTRL_HD_SEND_SHIFT                 (0x00u) /* 1 enable TX part in Half Duplex mode */
#define UART_CTRL_HD_SEND_BREAK_SHIFT           (0x01u) /* 1 send BREAK signal in Half Duplez mode */
#define UART_CTRL_MARK_SHIFT                    (0x02u) /* 1 sets mark, 0 sets space */
#define UART_CTRL_PARITY_TYPE0_SHIFT            (0x03u) /* Defines the type of parity implemented */
#define UART_CTRL_PARITY_TYPE1_SHIFT            (0x04u) /* Defines the type of parity implemented */
#define UART_CTRL_RXADDR_MODE0_SHIFT            (0x05u)
#define UART_CTRL_RXADDR_MODE1_SHIFT            (0x06u)
#define UART_CTRL_RXADDR_MODE2_SHIFT            (0x07u)

#define UART_CTRL_HD_SEND                       (0x01u << UART_CTRL_HD_SEND_SHIFT)
#define UART_CTRL_HD_SEND_BREAK                 (0x01u << UART_CTRL_HD_SEND_BREAK_SHIFT)
#define UART_CTRL_MARK                          (0x01u << UART_CTRL_MARK_SHIFT)
#define UART_CTRL_PARITY_TYPE_MASK              (0x03u << UART_CTRL_PARITY_TYPE0_SHIFT)
#define UART_CTRL_RXADDR_MODE_MASK              (0x07u << UART_CTRL_RXADDR_MODE0_SHIFT)

/* StatusI Register Interrupt Enable Control Bits. As defined by the Register map for the AUX Control Register */
#define UART_INT_ENABLE                         (0x10u)

/* Bit Counter (7-bit) Control Register Bit Definitions. As defined by the Register map for the AUX Control Register */
#define UART_CNTR_ENABLE                        (0x20u)

/*   Constants for SendBreak() "retMode" parameter  */
#define UART_SEND_BREAK                         (0x00u)
#define UART_WAIT_FOR_COMLETE_REINIT            (0x01u)
#define UART_REINIT                             (0x02u)
#define UART_SEND_WAIT_REINIT                   (0x03u)

#define UART_OVER_SAMPLE_8                      (8u)
#define UART_OVER_SAMPLE_16                     (16u)

#define UART_BIT_CENTER                         (UART_OVER_SAMPLE_COUNT - 1u)

#define UART_FIFO_LENGTH                        (4u)
#define UART_NUMBER_OF_START_BIT                (1u)

/* 8X always for count7 implementation*/
#define UART_TXBITCTR_BREAKBITS8X   (UART_BREAK_BITS_TX * UART_OVER_SAMPLE_8 - 1u)
/* 8X or 16X for DP implementation*/
#define UART_TXBITCTR_BREAKBITS   (UART_BREAK_BITS_TX * UART_OVER_SAMPLE_COUNT - 1u)

#if (UART_OVER_SAMPLE_COUNT == UART_OVER_SAMPLE_8)
    #define UART_HD_TXBITCTR_INIT   ((UART_BREAK_BITS_TX + \
        UART_NUMBER_OF_START_BIT) * UART_OVER_SAMPLE_COUNT - 1u)
    /* This parameter is increased on the 1 in 2 out of 3 mode to sample voting in the middle */
    #define UART_HD_RXBITCTR_INIT   ((UART_BREAK_BITS_RX + \
    UART_NUMBER_OF_START_BIT) * UART_OVER_SAMPLE_COUNT - 1u + \
    (UART_OVER_SAMPLE_COUNT / 2u) + (UART_USE23POLLING * 2u) - 1u)
#else /* UART_OVER_SAMPLE_COUNT == UART_OVER_SAMPLE_16 */
    #define UART_HD_TXBITCTR_INIT   (8u * UART_OVER_SAMPLE_COUNT - 1u)
    #define UART_HD_RXBITCTR_INIT   (7u * UART_OVER_SAMPLE_COUNT - 1u  +  \
       (UART_OVER_SAMPLE_COUNT / 2u) + (UART_USE23POLLING * 2u) - 1u)
#endif /* End UART_OVER_SAMPLE_COUNT */


/***************************************
* Enumerated Types and Parameters
***************************************/

#define UART__B_UART__AM_SW_BYTE_BYTE 1
#define UART__B_UART__AM_SW_DETECT_TO_BUFFER 2
#define UART__B_UART__AM_HW_BYTE_BY_BYTE 3
#define UART__B_UART__AM_HW_DETECT_TO_BUFFER 4
#define UART__B_UART__AM_NONE 0

#define UART__B_UART__NONE_REVB 0
#define UART__B_UART__EVEN_REVB 1
#define UART__B_UART__ODD_REVB 2
#define UART__B_UART__MARK_SPACE_REVB 3


/***************************************
*    Initial Parameter Constants
***************************************/

#define UART_TXBUFFERSIZE           (48u)
#define UART_RXBUFFERSIZE           (1u)
/* UART shifts max 8 bits, Mark/Space functionality working if 9 selected */
#define UART_NUMBER_OF_DATA_BITS    ((8u > 8u) ? 8u : 8u)
#define UART_NUMBER_OF_STOP_BITS    (1u)

#if (UART_RXHW_ADDRESS_ENABLED)
    #define UART_RXAddressMode      (1u)
    #define UART_RXHWADDRESS1       (0u)
    #define UART_RXHWADDRESS2       (0u)
#endif /* End EnableHWAddress */

#define UART_INIT_RX_INTERRUPTS_MASK \
                                          (0 << UART_RX_STS_FIFO_NOTEMPTY_SHIFT) \
                                        | (0 << UART_RX_STS_MRKSPC_SHIFT) \
                                        | (0 << UART_RX_STS_ADDR_MATCH_SHIFT) \
                                        | (0 << UART_RX_STS_PAR_ERROR_SHIFT) \
                                        | (0 << UART_RX_STS_STOP_ERROR_SHIFT) \
                                        | (0 << UART_RX_STS_BREAK_SHIFT) \
                                        | (0 << UART_RX_STS_OVERRUN_SHIFT)

#define UART_INIT_TX_INTERRUPTS_MASK \
                                          (0 << UART_TX_STS_COMPLETE_SHIFT) \
                                        | (0 << UART_TX_STS_FIFO_EMPTY_SHIFT) \
                                        | (0 << UART_TX_STS_FIFO_FULL_SHIFT) \
                                        | (0 << UART_TX_STS_FIFO_NOT_FULL_SHIFT)


/***************************************
*              Registers
***************************************/

#ifdef UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG
    #define UART_CONTROL_REG \
                            (* (reg8 *) UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG )
    #define UART_CONTROL_PTR \
                            (  (reg8 *) UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG )
#endif /* End UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG */

#if(UART_TX_ENABLED)
    #define UART_TXDATA_REG          (* (reg8 *) UART_BUART_sTX_TxShifter_u0__F0_REG)
    #define UART_TXDATA_PTR          (  (reg8 *) UART_BUART_sTX_TxShifter_u0__F0_REG)

    #define UART_TXSTATUS_REG        (* (reg8 *) UART_BUART_sTX_TxSts__STATUS_REG)
    #define UART_TXSTATUS_PTR        (  (reg8 *) UART_BUART_sTX_TxSts__STATUS_REG)
    #define UART_TXSTATUS_MASK_REG   (* (reg8 *) UART_BUART_sTX_TxSts__MASK_REG)
    #define UART_TXSTATUS_MASK_PTR   (  (reg8 *) UART_BUART_sTX_TxSts__MASK_REG)
    #define UART_TXSTATUS_ACTL_REG   (* (reg8 *) UART_BUART_sTX_TxSts__STATUS_AUX_CTL_REG)
    #define UART_TXSTATUS_ACTL_PTR   (  (reg8 *) UART_BUART_sTX_TxSts__STATUS_AUX_CTL_REG)

    /* DP clock */
    #if(UART_TXCLKGEN_DP)
        #define UART_TXBITCLKGEN_CTR_REG        \
                                        (* (reg8 *) UART_BUART_sTX_sCLOCK_TxBitClkGen__D0_REG)
        #define UART_TXBITCLKGEN_CTR_PTR        \
                                        (  (reg8 *) UART_BUART_sTX_sCLOCK_TxBitClkGen__D0_REG)
        #define UART_TXBITCLKTX_COMPLETE_REG    \
                                        (* (reg8 *) UART_BUART_sTX_sCLOCK_TxBitClkGen__D1_REG)
        #define UART_TXBITCLKTX_COMPLETE_PTR    \
                                        (  (reg8 *) UART_BUART_sTX_sCLOCK_TxBitClkGen__D1_REG)
    #else     /* Count7 clock*/
        #define UART_TXBITCTR_PERIOD_REG    \
                                        (* (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__PERIOD_REG)
        #define UART_TXBITCTR_PERIOD_PTR    \
                                        (  (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__PERIOD_REG)
        #define UART_TXBITCTR_CONTROL_REG   \
                                        (* (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__CONTROL_AUX_CTL_REG)
        #define UART_TXBITCTR_CONTROL_PTR   \
                                        (  (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__CONTROL_AUX_CTL_REG)
        #define UART_TXBITCTR_COUNTER_REG   \
                                        (* (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__COUNT_REG)
        #define UART_TXBITCTR_COUNTER_PTR   \
                                        (  (reg8 *) UART_BUART_sTX_sCLOCK_TxBitCounter__COUNT_REG)
    #endif /* UART_TXCLKGEN_DP */

#endif /* End UART_TX_ENABLED */

#if(UART_HD_ENABLED)

    #define UART_TXDATA_REG             (* (reg8 *) UART_BUART_sRX_RxShifter_u0__F1_REG )
    #define UART_TXDATA_PTR             (  (reg8 *) UART_BUART_sRX_RxShifter_u0__F1_REG )

    #define UART_TXSTATUS_REG           (* (reg8 *) UART_BUART_sRX_RxSts__STATUS_REG )
    #define UART_TXSTATUS_PTR           (  (reg8 *) UART_BUART_sRX_RxSts__STATUS_REG )
    #define UART_TXSTATUS_MASK_REG      (* (reg8 *) UART_BUART_sRX_RxSts__MASK_REG )
    #define UART_TXSTATUS_MASK_PTR      (  (reg8 *) UART_BUART_sRX_RxSts__MASK_REG )
    #define UART_TXSTATUS_ACTL_REG      (* (reg8 *) UART_BUART_sRX_RxSts__STATUS_AUX_CTL_REG )
    #define UART_TXSTATUS_ACTL_PTR      (  (reg8 *) UART_BUART_sRX_RxSts__STATUS_AUX_CTL_REG )
#endif /* End UART_HD_ENABLED */

#if( (UART_RX_ENABLED) || (UART_HD_ENABLED) )
    #define UART_RXDATA_REG             (* (reg8 *) UART_BUART_sRX_RxShifter_u0__F0_REG )
    #define UART_RXDATA_PTR             (  (reg8 *) UART_BUART_sRX_RxShifter_u0__F0_REG )
    #define UART_RXADDRESS1_REG         (* (reg8 *) UART_BUART_sRX_RxShifter_u0__D0_REG )
    #define UART_RXADDRESS1_PTR         (  (reg8 *) UART_BUART_sRX_RxShifter_u0__D0_REG )
    #define UART_RXADDRESS2_REG         (* (reg8 *) UART_BUART_sRX_RxShifter_u0__D1_REG )
    #define UART_RXADDRESS2_PTR         (  (reg8 *) UART_BUART_sRX_RxShifter_u0__D1_REG )

    #define UART_RXBITCTR_PERIOD_REG    (* (reg8 *) UART_BUART_sRX_RxBitCounter__PERIOD_REG )
    #define UART_RXBITCTR_PERIOD_PTR    (  (reg8 *) UART_BUART_sRX_RxBitCounter__PERIOD_REG )
    #define UART_RXBITCTR_CONTROL_REG   \
                                        (* (reg8 *) UART_BUART_sRX_RxBitCounter__CONTROL_AUX_CTL_REG )
    #define UART_RXBITCTR_CONTROL_PTR   \
                                        (  (reg8 *) UART_BUART_sRX_RxBitCounter__CONTROL_AUX_CTL_REG )
    #define UART_RXBITCTR_COUNTER_REG   (* (reg8 *) UART_BUART_sRX_RxBitCounter__COUNT_REG )
    #define UART_RXBITCTR_COUNTER_PTR   (  (reg8 *) UART_BUART_sRX_RxBitCounter__COUNT_REG )

    #define UART_RXSTATUS_REG           (* (reg8 *) UART_BUART_sRX_RxSts__STATUS_REG )
    #define UART_RXSTATUS_PTR           (  (reg8 *) UART_BUART_sRX_RxSts__STATUS_REG )
    #define UART_RXSTATUS_MASK_REG      (* (reg8 *) UART_BUART_sRX_RxSts__MASK_REG )
    #define UART_RXSTATUS_MASK_PTR      (  (reg8 *) UART_BUART_sRX_RxSts__MASK_REG )
    #define UART_RXSTATUS_ACTL_REG      (* (reg8 *) UART_BUART_sRX_RxSts__STATUS_AUX_CTL_REG )
    #define UART_RXSTATUS_ACTL_PTR      (  (reg8 *) UART_BUART_sRX_RxSts__STATUS_AUX_CTL_REG )
#endif /* End  (UART_RX_ENABLED) || (UART_HD_ENABLED) */

#if(UART_INTERNAL_CLOCK_USED)
    /* Register to enable or disable the digital clocks */
    #define UART_INTCLOCK_CLKEN_REG     (* (reg8 *) UART_IntClock__PM_ACT_CFG)
    #define UART_INTCLOCK_CLKEN_PTR     (  (reg8 *) UART_IntClock__PM_ACT_CFG)

    /* Clock mask for this clock. */
    #define UART_INTCLOCK_CLKEN_MASK    UART_IntClock__PM_ACT_MSK
#endif /* End UART_INTERNAL_CLOCK_USED */


/***************************************
* Renamed global variables or defines 
* for backward compatible
***************************************/

#define UART_initvar                    UART_initVar

#define UART_RX_Enabled                 UART_RX_ENABLED
#define UART_TX_Enabled                 UART_TX_ENABLED
#define UART_HD_Enabled                 UART_HD_ENABLED
#define UART_RX_IntInterruptEnabled     UART_RX_INTERRUPT_ENABLED
#define UART_TX_IntInterruptEnabled     UART_TX_INTERRUPT_ENABLED
#define UART_InternalClockUsed          UART_INTERNAL_CLOCK_USED
#define UART_RXHW_Address_Enabled       UART_RXHW_ADDRESS_ENABLED
#define UART_OverSampleCount            UART_OVER_SAMPLE_COUNT
#define UART_ParityType                 UART_PARITY_TYPE

#if( UART_TX_ENABLED && (UART_TXBUFFERSIZE > UART_FIFO_LENGTH))
    #define UART_TXBUFFER               UART_txBuffer
    #define UART_TXBUFFERREAD           UART_txBufferRead 
    #define UART_TXBUFFERWRITE          UART_txBufferWrite 
#endif /* End UART_TX_ENABLED */
#if( ( UART_RX_ENABLED || UART_HD_ENABLED ) && \
     (UART_RXBUFFERSIZE > UART_FIFO_LENGTH) )
    #define UART_RXBUFFER               UART_rxBuffer
    #define UART_RXBUFFERREAD           UART_rxBufferRead 
    #define UART_RXBUFFERWRITE          UART_rxBufferWrite 
    #define UART_RXBUFFERLOOPDETECT     UART_rxBufferLoopDetect
    #define UART_RXBUFFER_OVERFLOW      UART_rxBufferOverflow
#endif /* End UART_RX_ENABLED */

#ifdef UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG
    #define UART_CONTROL                UART_CONTROL_REG 
#endif /* End UART_BUART_sCR_AsyncCtl_CtrlReg__CONTROL_REG */

#if(UART_TX_ENABLED)
    #define UART_TXDATA                 UART_TXDATA_REG
    #define UART_TXSTATUS               UART_TXSTATUS_REG
    #define UART_TXSTATUS_MASK          UART_TXSTATUS_MASK_REG   
    #define UART_TXSTATUS_ACTL          UART_TXSTATUS_ACTL_REG
    /* DP clock */
    #if(UART_TXCLKGEN_DP)
        #define UART_TXBITCLKGEN_CTR        UART_TXBITCLKGEN_CTR_REG
        #define UART_TXBITCLKTX_COMPLETE    UART_TXBITCLKTX_COMPLETE_REG
    #else     /* Count7 clock*/
        #define UART_TXBITCTR_PERIOD        UART_TXBITCTR_PERIOD_REG
        #define UART_TXBITCTR_CONTROL       UART_TXBITCTR_CONTROL_REG
        #define UART_TXBITCTR_COUNTER       UART_TXBITCTR_COUNTER_REG
    #endif /* UART_TXCLKGEN_DP */
#endif /* End UART_TX_ENABLED */

#if(UART_HD_ENABLED)
    #define UART_TXDATA                 UART_TXDATA_REG
    #define UART_TXSTATUS               UART_TXSTATUS_REG
    #define UART_TXSTATUS_MASK          UART_TXSTATUS_MASK_REG   
    #define UART_TXSTATUS_ACTL          UART_TXSTATUS_ACTL_REG
#endif /* End UART_HD_ENABLED */

#if( (UART_RX_ENABLED) || (UART_HD_ENABLED) )
    #define UART_RXDATA                 UART_RXDATA_REG             
    #define UART_RXADDRESS1             UART_RXADDRESS1_REG
    #define UART_RXADDRESS2             UART_RXADDRESS2_REG
    #define UART_RXBITCTR_PERIOD        UART_RXBITCTR_PERIOD_REG
    #define UART_RXBITCTR_CONTROL       UART_RXBITCTR_CONTROL_REG
    #define UART_RXBITCTR_COUNTER       UART_RXBITCTR_COUNTER_REG
    #define UART_RXSTATUS               UART_RXSTATUS_REG
    #define UART_RXSTATUS_MASK          UART_RXSTATUS_MASK_REG
    #define UART_RXSTATUS_ACTL          UART_RXSTATUS_ACTL_REG
#endif /* End  (UART_RX_ENABLED) || (UART_HD_ENABLED) */

#if(UART_INTERNAL_CLOCK_USED)
    #define UART_INTCLOCK_CLKEN         UART_INTCLOCK_CLKEN_REG
#endif /* End UART_INTERNAL_CLOCK_USED */

#endif  /* CY_UART_UART_H */


/* [] END OF FILE */
