;
;     VFO2.asm - Program to set frequency of ADF4157 sigma-delta PLL
;
;            (C) Copyright 2008-2010 John B. Stephensen
;
; This assembly language source file and all its derivatives are licensed
; only for personal non-profit educational use in the Amateur Radio Service
; and the license is not transferrable. The information is provided as-is
; for experimental purposes and the author does not warranty its freedom
; from defects or its suitability for any specific application.
;
#pragma AVRPART ADMIN PART_NAME ATtiny2313
;
; I/O port definitions
;
.EQU	CLKPR = $26	; clock divider
.EQU	CLKPCE = 7	; enable clock divider change
;
; port B
;
.EQU	PB = $18	; port B data
.EQU	DDRB = $17	; port B direction
.EQU	SCK = 0		; PLL clock
.EQU	SDO = 1		; PLL data
.EQU	SLD = 2		; load PLL
.EQU	OUT1 = 3	; open collector outputs
.EQU	OUT2 = 4
;
; Port D
;
.EQU	PD = $12	; port D data
.EQU	DDRD = $11	; port D direction
.EQU	XMT = 4		; RS-485 driver enable
.EQU	LED = 5		; LED (0=off, 1=on)
.EQU	CE = 6		; PLL chip enable
;
; EEPROM control registers
;
.EQU	EEAR = $1E	; address
.EQU	EEDR = $1D	; data
.EQU	EECR = $1C	; control
.EQU	EEPM1 = 5	; programming mode
.EQU	EEPM0 = 4	; 
.EQU	EERIE = 3	; enable ready interrupt
.EQU	EEMPE = 2	; master program enable
.EQU	EEPE = 1	; program enable
.EQU	EERE = 0	; read enable
;
; UART
;
.EQU	UDR = $0C	; data
.EQU	UCSRA = $0B	; status register
.EQU	RXC = 7		; receiver ready
.EQU	TXC = 6		; transmit complete
.EQU	UDRE = 5	; transmitter empty
.EQU 	FE = 4		; framing error
.EQU	DOR = 3		; overrun error
.EQU	UPE = 2		; parity error
.EQU	U2X = 1		; double rate (8X clock)
.EQU	MPCM = 0	; multiprocessor comm. mode
.EQU	UCSRB = $0A	; control register 1
.EQU	RXCIE = 7	; enable receive complete interrupt
.EQU	TXCIE = 6	; enable transmit complete interrupt
.EQU	UDRIE = 5	; enable transmitter empty interrupt
.EQU	RXEN = 4	; enable receiver
.EQU	TXEN = 3	; enable transmitter
.EQU	UCSZ2 = 2	; 9-bit character size
.EQU	RXB8 = 1	; receive 9th data bit
.EQU	TXB8 = 0	; transmit 9th data bit
.EQU	UCSRC = $03	; control register 2
.EQU	UMSEL1 = 7	;  0=async, 1=sync, 3=master SPI
.EQU	UMSEL0 = 6
.EQU	UPM1 = 5	; parity: 0=none, 2=even, 3=odd
.EQU	UPM0 = 4
.EQU	USBS = 3	; stop bits: 0=1, 1=2
.EQU	UCSZ1 = 2	; char. size: 0=5, 1=6, 2=7, 3=8, 7=9
.EQU	UCSZ0 = 1
.EQU	UCPOL = 0	; 
.EQU	UBRRL = $09	; baud rate low and high
.EQU	UBRRH = $02
;
; Initialize I/O ports
;
START:	LDI		R25,$80		; enable CPU clock divider
		OUT		CLKPR,R25
		LDI		R25,$00		; CPU clock divider = 1
		OUT		CLKPR,R25
		LDI		R25,$1F		; enable PLL SPI and OC outputs
		OUT		DDRB,R25
		LDI		R25,$70		; enable RS-485, LED and CE
		OUT		DDRD,R25
		RCALL	USET115K2	; 115,200 baud RS-485 port
		SBI		PD,CE		; enable PLL
		RCALL	DEFPLL		; set PLL to default values
;
; Main Loop - process commands
;
LOOP:	RCALL	UGETLN		; fill buffer
		LD		R25,X+		; get first character
		CPI		R25,'F'		; frequency command
		BREQ	GETFRQ
		CPI		R25,'D'		; default frequency command
		BREQ	GETDEF
		CPI		R25,'N'		; N divider value command
		BREQ	GETN
		CPI		R25,'C'		; charge pump current command
		BREQ	GETCPC
		CPI		R25,'I'		; phase detector invert command
		BREQ	GETPDP
		CPI		R25,'P'		; prescaler/divider/doubler flags command
		BREQ	GETPDD
		CPI		R25,'R'		; R divider value command
		BREQ	GETR
		CPI		R25,'S'		; reference frequency command
		BREQ	GETREF
		RJMP	LOOP
;
; Get hexidecimal value and set N
;
GETN:	LDI		R24,REFDIV	; get reference divider
		CALL	EERD
		MOV		R6,R25
		RCALL	GETHEX		; read N into R7-12
		RCALL	SETFRQ
		RJMP	LOOP
;
; Get charge pump current
;
GETCPC:	RCALL	GETNUM		; read R into R0-3
		LDI		R24,CPCS	; point at charge pump current storage
		MOV		R25,R0
		RCALL	EEWR		; save
		RCALL	BLINK
		RJMP	LOOP
;
; Get phase detector polarity
;
GETPDP:	RCALL	GETNUM		; read R into R0-3
		LDI		R24,PDPF	; point at phase detector polarity flag
		MOV		R25,R0
		RCALL	EEWR		; save
		RCALL	BLINK
		RJMP	LOOP
;
; Get prescaler, divider and doubler flags
;
GETPDD:	RCALL	GETNUM		; read R into R0-3
		LDI		R24,PDDF	; point at flag storage
		MOV		R25,R0
		RCALL	EEWR		; save
		RCALL	BLINK
		RJMP	LOOP
;
; Get reference divider value (max.)
;
GETR:	RCALL	GETNUM		; read R into R0-3
		LDI		R24,REFDIV	; point at reference frequency storage
		MOV		R25,R0
		RCALL	EEWR		; save
		RCALL	BLINK
		RJMP	LOOP
;
; Get reference frequency
;
GETREF:	RCALL	GETNUM		; get frequency into R0-3
		LDI		R24,REFFRQ	; point at reference frequency storage
		RCALL	EEWR4		; save frequency
		RCALL	BLINK
		RJMP	LOOP
;
; Get default frequency
;
GETDEF:	RCALL	GETNUM		; get frequency into R0-3
		LDI		R24,DEFFRQ	; point at default frequency storage
		RCALL	EEWR4		; save frequency
		RCALL	BLINK
		RJMP	LOOP
;
; Get decimal frequency in Hertz and set PLL
;
GETFRQ:	RCALL	GETNUM		; get frequency into R0-3
		RCALL	CALCNR		; calculate N and R
		RCALL	SETFRQ
		RJMP	LOOP
;
; Calculate N
; Call:	R0-3 = frequency in Hertz
;		R6 = reference divider
; Uses:	R13-27
; Ret:	R6 = reference divider
;		R7-10 = fractional N
;		R11-12 = integer N
;
CALCNR:	LDI		R24,REFDIV	; get reference divider
		CALL	EERD
;		LDI		R25,10		; TEST
		MOV		R6,R25
		RCALL	CALCNR1		; calculate N for given R
		MOV		R24,R10
		CPI		R24,$00		; redo if < 1/256
		BREQ	CALCNR0
		CPI		R24,$FF
		BREQ	CALCNR0
		RET
CALCNR0:
		LDI		R24,REFDIV	; get reference divider
		CALL	EERD
;		LDI		R25,10		; TEST
		MOV		R6,R25
		DEC		R6			; use R-1
CALCNR1:
		MOV		R11,R6		; reference divider
		CALL	MUL32X8U	; multiply
		LDI		R24,REFFRQ	; get Fref
		CALL	EERD420
		RCALL	DIV40X32U	; divide by Fref
		RET
;
; Retreive default values from program memory
;
DEFPLL:	LDI		R24,DEFFRQ	; get default frequency
		CALL	EERD4
		CALL	CALCNR
;
; Initialize PLL registers 4 through 0
; Call:	R6 = reference divider
;		R7-10 = fractional frequency (LSB first)
;		R11-12 = integer frequency (LSB first)
; Uses:	R20-23 = 32-bit data for PLL
;		R24 = temp.
;
RSTPLL:	LDI		R20,4	; set test register
		LDI		R21,0	; clk div off
		LDI		R22,$80	; bleed currnet on
		LDI		R23,$01
		RCALL	SEND32
		LDI		R20,$83	; set function register, 40 LDP
		LDI		R24,PDPF
		RCALL	EERD	; get phase detector polarity
		COM		R25		; invert as 0=neg, 1=pos polarity
		ANDI	R25,$01	; only bit 0 used
		SWAP	R25		; move to bit 4
		LSL		R25		; move to bit 5
		LSL		R25		; move to bit 6
		OR		R20,R25	; add to lower byte
		LDI		R21,$40	; don't reset SD acc.
		LDI		R22,$00
		LDI		R23,$00
		RCALL	SEND32
;
; Update frequency only via registers 2, 1 and 0
; Call:	R6 = reference divider
;		R7-10 = fractional frequency (LSB first)
;		R11-12 = integer frequency (LSB first)
; Uses:	R20-23 = 32-bit data for PLL
;		R24-25 = temp.
;
SETFRQ:	LDI		R20,2	; set R divider
		LDI		R21,0
		MOV		R22,R6	; get R
		ANDI	R22,$1F	; 5 LSB used
		LSR		R22		; top 4 bit in DB19-16
		ROR		R21		; LSB to DB15
		LDI		R24,PDDF
		RCALL	EERD	; get prescaler, divider and doubler flags
		ANDI	R25,$07	; only bottom 3 bits used
		SWAP	R25		; put in bits 6-4
		OR		R22,R25	; set prescaler, div2, dblr bits
		LDI		R24,CPCS
		RCALL	EERD	; get charge pump currrent setting
		MOV		R23,R25
		ANDI	R23,$0F	; uses DB27-24
		ORI		R23,$00	; reset CSR bit
		RCALL	SEND32
		LDI		R20,1	; set LS frac register
		MOV		R21,R7	; get LS 20 bits fraction
		MOV		R22,R8
		MOV		R23,R9
		ANDI	R21,$80	; clear DB14-8
		ANDI	R23,$0F	; clear top 4 bits
		RCALL	SEND32
		MOV		R20,R9	; get MS 12 bits fraction
		MOV		R21,R10
		MOV		R22,R11	; get 12 bits integer
		MOV		R23,R12
		LSR		R23		; shift right 1 bit
		ROR		R22
		ROR		R21
		ROR		R20
		ANDI	R23,$07	; clear top 5 bits
		ANDI	R20,$F8	; clear bottom 3 bits
		ORI		R23,$70	; output is N/2
;
; Transmit 32 bits to PLL
; Call:	R20-23 = data to transmit (LSB first)
; Uses:	R24 = counter
;
SEND32:	LDI		R24,32	; send 32 bits
SEND1:	SBRC	R23,7	; place MSB on data line
		SBI		PB,SDO
		SBRS	R23,7
		CBI		PB,SDO
		LSL		R20		; shift all 32 bits left
		ROL		R21
		ROL		R22
		ROL		R23
		SBI		PB,SCK	; clock high
		NOP
		CBI		PB,SCK	; clock low
		DEC		R24
		BRNE	SEND1	; loop until 32 bits sent
		SBI		PB,SLD	; load high
		NOP
		CBI		PB,SLD	; load low
		RET
;
; Get 0-12 digit hexidecimal number from UART
; Call:	R27:26 = address of next character
; Uses:	R13 = zero
;		R25 = next character
; Ret:	R7-12 = binary number
;		R27:26 = address of next non-numeric character
;
GETHEX:	CLR		R7		; clear accumulator
		CLR		R8
		CLR		R9
		CLR		R10
		CLR		R11
		CLR		R12
GETHEX0:
		LD		R25,X+	; get next digit
		CPI		R25,$30	; check if '0'-'9'
		BRLT	GETHEX3	; return if non-numeric
		CPI		R25,$3A	; check for maximum
		BRLT	GETHEX1	; continue if numeric
		CPI		R25,$41	; check if 'A'-'F'
		BRLT	GETHEX3	; return if non-numeric
		CPI		R25,$47	; check for maximum
		BRGE	GETHEX3	; return if non-numeric
GETHEX1:
		ADD		R7,R7	; *2
		ADC		R8,R8
		ADC		R9,R9
		ADC		R10,R10
		ADC		R11,R11
		ADC		R12,R12
		ADD		R7,R7	; *4
		ADC		R8,R8
		ADC		R9,R9
		ADC		R10,R10
		ADC		R11,R11
		ADC		R12,R12
		ADD		R7,R7	; *8
		ADC		R8,R8
		ADC		R9,R9
		ADC		R10,R10
		ADC		R11,R11
		ADC		R12,R12
		ADD		R7,R7	; *16
		ADC		R8,R8
		ADC		R9,R9
		ADC		R10,R10
		ADC		R11,R11
		ADC		R12,R12
		CLR		R13
		SUBI	R25,'0'	; convert from ASCII to binary
		CPI		R25,10
		BRLT	GETHEX2
		SUBI	R25,7	; 'A'=>10
GETHEX2:
		ADD		R7,R25	; add next digit
		ADC		R8,R13	; propagate carry
		ADC		R9,R13
		ADC		R10,R13
		ADC		R11,R13
		ADC		R12,R13
		RJMP	GETHEX0	; check for next digit
GETHEX3:
		SBIW	R27:R26,1	; undo load of next digit
		RET
;
; Get 0-10 digit decimal number from UART
; Call:	R27:26 = address of next character
; Uses:	R20-23 = temp.
;		R24 = zero
;		R25 = next character
; Ret:	R7-10 = binary number
;		R27:26 = address of next non-numeric character
;
GETNUM:	CLR		R0		; clear accumulator
		CLR		R1
		CLR		R2
		CLR		R3
GETNUM0:
		LD		R25,X+	; get next digit
		CPI		R25,$30	; check if '0'-'9'
		BRLT	GETNUM1	; return if non-numeric
		CPI		R25,$3A	; check for maximum
		BRGE	GETNUM1	; return if non-numeric
		ADD		R0,R0	; *2
		ADC		R1,R1
		ADC		R2,R2
		ADC		R3,R3
		MOV		R20,R0	; save
		MOV		R21,R1
		MOV		R22,R2
		MOV		R23,R3
		ADD		R0,R0	; *4
		ADC		R1,R1
		ADC		R2,R2
		ADC		R3,R3
		ADD		R0,R0	; *8
		ADC		R1,R1
		ADC		R2,R2
		ADC		R3,R3
		ADD		R0,R20	; *10
		ADC		R1,R21
		ADC		R2,R22
		ADC		R3,R23
		CLR		R24
		SUBI	R25,'0'	; convert from ASCII to binary
		ADD		R0,R25	; add next digit
		ADC		R1,R24	; propagate carry
		ADC		R2,R24
		ADC		R3,R24
		RJMP	GETNUM0	; check for next digit
GETNUM1:
		SBIW	R27:R26,1	; undo load of next digit
		RET
;
; Configure UART
; Call:	R24-25 = baud rate divisor (LSB first)
;
USET115K2:
		LDI		R24,3		; 115.2 kbaud with 7.3728 MHz clock
		LDI		R25,0
USET:	OUT		UBRRH,R25	; set baud rate
		OUT		UBRRL,R24
		LDI		R25,(1<<RXEN)|(1<<TXEN)
		OUT		UCSRB,R25	; enable transmitter and receiver
		LDI		R25,(3<<UCSZ0)
		OUT		UCSRC,R25	; 8 data bits, 1 stop bit, no parity
		RET
;
; Receive a line terminated in LF or CRLF
; Uses:	R25 = received character
;		R27:R26 = pointer to buffer
; Ret:	R27:R26 = pointer to first character in buffer
;
UGETLN:	LDI		R26,LBUF & $FF
		LDI		R27,LBUF >> 8
UGETLN0:
		CALL	UGETC		; get next character
		ST		X+,R25		; save value at R27:R26
		CPI		R25,$0A		; check for LF
		BREQ	UGETLN1
		CPI		R25,$0D		; check for CR
		BRNE	UGETLN0
UGETLN1:
		LDI		R26,LBUF & $FF
		LDI		R27,LBUF >> 8
		RET
;
; Receive a byte
; Ret:	R25 = data
;
UGETC:	CBI		PD,XMT		; enable receiver
UGETC0:	SBIS	UCSRA,RXC	; data received?
		RJMP	UGETC0
		IN		R25,UDR		; get byte
		ANDI	R25,$7F		; strip top bit
		CPI		R25,0
		BREQ	UGETC0		; discard nulls
		RET
;
; Send a byte
; Call:	R25 = data
;
UCRLF:	LDI		R25,$0D
		CALL	UPUTC
		LDI		R25,$0A
UPUTC:	SBI		PD,XMT		; enable driver
UPUTC0:	SBIS	UCSRA,UDRE	; transmitter empty?
		RJMP	UPUTC0
		OUT		UDR,R25		; send byte
		RET
;
; Write 4 bytes to EEPROM
; Call:	R0-3 = data
;		R24 = address
; Uses:	R23 = temp.
;		R25 = temp. data
;
EEWR4:	MOV		R25,R0		; write bytes in sequence 0-3
		RCALL	EEWR
		INC		R24
		MOV		R25,R1
		RCALL	EEWR
		INC		R24
		MOV		R25,R2
		RCALL	EEWR
		INC		R24
		MOV		R25,R3
;
; Write a byte to EEPROM
; Call:	R24 = address
;		R25 = data
; Uses:	R23 = temp.
;
EEWR:	SBIC	EECR,EEPE	; wait for write completion
		RJMP	EEWR
		LDI		R23,(0<<EEPM1)|(0<<EEPM0)
		OUT		EECR,R23	; set programming mode to atomic
		OUT		EEAR,R24	; set up address
		OUT		EEDR,R25	; write data
		SBI		EECR,EEMPE	; write one
		SBI		EECR,EEPE	; start write
		RET
;
; Read 4 bytes from EEPROM
; Call:	R24 = address
; Uses:	R25 = temp. data
; Ret:	R0-3 or R20-23 = data
;
EERD4:	CALL	EERD		; read bytes into R0-3
		MOV		R0,R25
		INC		R24
		CALL	EERD
		MOV		R1,R25
		INC		R24
		CALL	EERD
		MOV		R2,R25
		INC		R24
		CALL	EERD
		MOV		R3,R25
		RET
EERD420:
		CALL	EERD		; read bytes into R20-23
		MOV		R20,R25
		INC		R24
		CALL	EERD
		MOV		R21,R25
		INC		R24
		CALL	EERD
		MOV		R22,R25
		INC		R24
		CALL	EERD
		MOV		R23,R25
		RET
;
; Read a byte from EEPROM
; Call:	R24 = address
; Ret.	R25 = data
;
EERD:	SBIC	EECR,EEPE	; wait for write completion
		RJMP	EERD
		OUT		EEAR,R24	; set up address
		SBI		EECR,EERE	; start read
		IN		R25,EEDR	; read data
		RET
;
; 32-bit by 8-bit Unsigned Multiply
; Call:	R0-3 = multiplicand
;		R11 = multiplier
; Uses:	R24 = counter
; Ret:	R11-15 = product
;
MUL32X8U:
		CLR		R15			; clear top 32-bits of product
		CLR		R14
		CLR		R13
		CLR		R12
		LDI		R24,8		; initialize bit counter
		LSR		R11			; shift out first multiplier bit
MUL32X8U0:
		BRCC	MUL32X8U2
		ADD		R12,R0		; add multiplicand to product
		ADC		R13,R1
		ADC		R14,R2
		ADC		R15,R3
MUL32X8U2:
		ROR		R15			; shift 40-bit product right
		ROR		R14
		ROR		R13
		ROR		R12
		ROR		R11
		DEC		R24			; more bits to process?
		BRNE	MUL32X8U0
		RET
;
; 40-bit by 32-bit unsigned divide with fractional quotient
; Call:	R11-15 = 40-bit dividend
;		R20-23 = 32-bit divisor
; Uses:	R24 = loop counter
; Ret:	R7-10 = 32-bit fractional quotient
;		R11-15 = 40-bit integer quotient
;		R16-19 = 32-bit remainder
;
DIV40X32U:
		CLR		R7			; zero quotient
		CLR		R8
		CLR		R9
		CLR		R10
		CLR		R16			; zero remainder
		CLR		R17
		CLR		R18
		SUB		R19,R19		; clear carry
		LDI		R24,73		; initialize loop counter
DIV40X32U1:
		ROL		R7			; shift quotient left
		ROL		R8
		ROL		R9
		ROL		R10
		ROL		R11			; shift dividend left
		ROL		R12
		ROL		R13
		ROL		R14
		ROL		R15
		DEC		R24			; decrement counter
		BRNE	DIV40X32U2	; jump if more bits to process
		RET
DIV40X32U2:
		ROL		R16			; shift dividend into remainder
		ROL		R17
		ROL		R18
		ROL		R19
		SUB		R16,R20		; subtract divisor from remainder
		SBC		R17,R21
		SBC		R18,R22
		SBC		R19,R23
		BRCC	DIV40X32U3	; jump if remainder still positive
		ADD		R16,R20		; add divisor to remainder
		ADC		R17,R21
		ADC		R18,R22
		ADC		R19,R23
		CLC					; clear carry (quotient bit is zero)
		RJMP	DIV40X32U1	; continue
DIV40X32U3:
		SEC					; set carry (quotient bit is one)
		RJMP	DIV40X32U1	; continue
;
; Blink LED
;
BLINK:	SBI		PD,LED
		CALL	WAIT
		CBI		PD,LED
		RET
;
; Wait 1/2 sec.
;
WAIT:	LDI		R29,16
		LDI		R30,0
		LDI		R31,0
WAIT0:	DEC		R31
		BRNE	WAIT0
		DEC		R30
		BRNE	WAIT0
		DEC		R29
		BRNE	WAIT0
		RET
;
; RAM
;
.DSEG
LBUF:	.BYTE	80	; line buffer
;
; EEPROM
;
.ESEG
DEFFRQ:	.BYTE	4	; default frequency in Hertz
REFFRQ:	.BYTE	4	; reference frequency in Hertz
REFDIV:	.BYTE	1	; reference divider
CPCS:	.BYTE	1	; charge pump current setting
PDDF:	.BYTE	1	; prescaler, divider and doubler flags
PDPF:	.BYTE	1	; phase detector polarity flag
;
