
//
//    Copyright  2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;
using System.Windows.Forms;

namespace ABCDmatrix
{
    /// <summary>
    /// Holds all four 2-port parameters at specified frequency.
    /// </summary>
    [Serializable]
    public class ParameterSet           // holds one line (one frequency) of a parameter set
    {
        /// <summary>
        /// The frequency of the point. -1 signals an error.
        /// </summary>
        public Double frequency;
        /// <summary>
        /// S11, Z11, Y11, or A
        /// </summary>
        public Complex P11A;
        /// <summary>
        /// S21, Z21, Y21, or C
        /// </summary>
        public Complex P21C;
        /// <summary>
        /// S12, Z12, Y12, or B
        /// </summary>
        public Complex P12B;
        /// <summary>
        /// S22, Z22, Y22, or D
        /// </summary>
        public Complex P22D;

        /// <summary>
        /// Default constructor
        /// </summary>
        public ParameterSet()
        {
            frequency = new Double();       // frequency as a double to provide large exponent range
            P11A = new Complex();
            P21C = new Complex();
            P22D = new Complex();
            P12B = new Complex();
        }

        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="copyfrom"></param>
        public ParameterSet(ParameterSet copyfrom)
        {
            frequency = copyfrom.frequency;
            P11A = new Complex(copyfrom.P11A);
            P21C = new Complex(copyfrom.P21C);
            P22D = new Complex(copyfrom.P22D);
            P12B = new Complex(copyfrom.P12B);
        }


        /// <summary>
        /// Linear interpolation between two ParameterSets at a frequency between the two sets.
        /// </summary>
        /// <param name="pslow">Lower Frequency ParameterSet.</param>
        /// <param name="pshigh">Higher Frequency ParameterSet.</param>
        /// <param name="Frequency">Frequency between low and high to interpolate parameters at.</param>
        /// <returns>Interpolated parameter set (using linear Re.Im interpolation).</returns>
        public static ParameterSet Interpolate(ParameterSet pslow, ParameterSet pshigh, Double Frequency)
        {
            ParameterSet result = new ParameterSet();
            result.frequency = Frequency;

            Double dx, dy;

            // dy varies from 0 to 1, low to high
            if (pshigh.frequency == pslow.frequency)    // supress interpolation if the two are the same freq
                dy = 0;
            else
                dy = (Frequency - pslow.frequency) / (pshigh.frequency - pslow.frequency);

            dx = pshigh.P11A.real - pslow.P11A.real;
            result.P11A.real = pslow.P11A.real + dy * dx;
            dx = pshigh.P11A.imag - pslow.P11A.imag;
            result.P11A.imag = pslow.P11A.imag + dy * dx;

            dx = pshigh.P12B.real - pslow.P12B.real;
            result.P12B.real = pslow.P12B.real + dy * dx;
            dx = pshigh.P12B.imag - pslow.P12B.imag;
            result.P12B.imag = pslow.P12B.imag + dy * dx;

            dx = pshigh.P21C.real - pslow.P21C.real;
            result.P21C.real = pslow.P21C.real + dy * dx;
            dx = pshigh.P21C.imag - pslow.P21C.imag;
            result.P21C.imag = pslow.P21C.imag + dy * dx;

            dx = pshigh.P22D.real - pslow.P22D.real;
            result.P22D.real = pslow.P22D.real + dy * dx;
            dx = pshigh.P22D.imag - pslow.P22D.imag;
            result.P22D.imag = pslow.P22D.imag + dy * dx;

            return result;
        }
        /// <summary>
        /// Invert a 2-port matrix (ABCD, S, Z, or Y) contained in a ParameterSet.
        /// </summary>
        /// <param name="p">ParameterSet holding 2-port (ABCD, S, Z, or Y) to invert.</param>
        /// <returns>inverted ParameterSet. Frequency = -1 and parameters = NaN indicate inversion error.</returns>
        public static ParameterSet Invert(ParameterSet p)
        {
            // Inverts the matrix.
            // Sets frequency to -1, and all matrix values to NotANumber
            //   to signal error.
            // Complex Divide throws an 'ArgumentException' on divide-by-zero error.

            Complex invDeterminant = new Complex();
            ParameterSet r = new ParameterSet();

            try             
            {
                invDeterminant = Complex.One / (p.P11A * p.P22D - p.P12B * p.P21C);
                r.P11A = p.P22D * invDeterminant;
                r.P12B = -p.P12B * invDeterminant;
                r.P21C = -p.P21C * invDeterminant;
                r.P22D = p.P11A * invDeterminant;
                r.frequency = p.frequency;
                return r;
            }
            catch (ArgumentException ae)        
            {
                MessageBox.Show("Can't invert 2-port matrix\n\r", ae.Message, MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                r.P11A = new Complex(Double.NaN, Double.NaN);
                r.P12B = new Complex(Double.NaN, Double.NaN);
                r.P21C = new Complex(Double.NaN, Double.NaN);
                r.P22D = new Complex(Double.NaN, Double.NaN);
                r.frequency = -1;       // signal error to caller
                return r;
            }
        }
        /// <summary>
        /// Tests p for singularity. A singular matrix cannot be inverted.
        /// </summary>
        /// <param name="p">Matrix to test</param>
        /// <returns>True if matrix is singular.</returns>
        public static Boolean IsSingular(ParameterSet p)
        {
            Complex determinant = (p.P11A * p.P22D - p.P12B * p.P21C);

            if((determinant.real == 0) && (determinant.imag == 0))
                return true;
            else
                return false;
        }

        /// <summary>
        /// Matrix multiplication p1 times p2.
        /// Matrix multiplication in not commutative, p1*p2 is not equal to p2*p1.
        /// </summary>
        /// <param name="p1">first matrix</param>
        /// <param name="p2">second matrix</param>
        /// <returns>resultant 2-port matrix</returns>
        public static ParameterSet operator *(ParameterSet p1, ParameterSet p2)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p1.frequency;

            r.P11A = p1.P11A * p2.P11A + p1.P12B * p2.P21C;
            r.P12B = p1.P11A * p2.P12B + p1.P12B * p2.P22D;
            r.P21C = p1.P21C * p2.P11A + p1.P22D * p2.P21C;
            r.P22D = p1.P21C * p2.P12B + p1.P22D * p2.P22D;

            return r;
        }

        /// <summary>
        /// Matrix multiplication by a complex scalar
        /// </summary>
        /// <param name="c">scal</param>
        /// <param name="p1">ParameterSet</param>
        /// <returns>resultant 2-port matrix</returns>
        public static ParameterSet operator *(Complex c, ParameterSet p1)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p1.frequency;

            r.P11A = p1.P11A * c;
            r.P12B = p1.P12B * c;
            r.P21C = p1.P21C * c;
            r.P22D = p1.P22D * c;

            return r;
        }
        /// <summary>
        /// Matrix division by a complex scalar
        /// </summary>
        /// <param name="p1">ParameterSet</param>
        /// <param name="c">scal</param>
        /// <returns>resultant 2-port matrix</returns>
        public static ParameterSet operator /( ParameterSet p1, Complex c)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p1.frequency;

            r.P11A = p1.P11A / c;
            r.P12B = p1.P12B / c;
            r.P21C = p1.P21C / c;
            r.P22D = p1.P22D / c;

            return r;
        }
       
        
        /// <summary>
        /// Matrix addition p1 plus p2.
        /// </summary>
        /// <param name="p1">first matrix</param>
        /// <param name="p2">second matrix</param>
        /// <returns>resultant 2-port matrix</returns>
        public static ParameterSet operator +(ParameterSet p1, ParameterSet p2)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p1.frequency;

            r.P11A = p1.P11A + p2.P11A;
            r.P12B = p1.P12B + p2.P12B;
            r.P21C = p1.P21C + p2.P21C;
            r.P22D = p1.P21C + p2.P22D;

            return r;
        }
        /// <summary>
        /// Matrix subtraction p1 minus p2.
        /// </summary>
        /// <param name="p1">first matrix</param>
        /// <param name="p2">second matrix</param>
        /// <returns>resultant 2-port matrix</returns>
        public static ParameterSet operator -(ParameterSet p1, ParameterSet p2)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p1.frequency;

            r.P11A = p1.P11A - p2.P11A;
            r.P12B = p1.P12B - p2.P12B;
            r.P21C = p1.P21C - p2.P21C;
            r.P22D = p1.P21C - p2.P22D;

            return r;
        }
        /// <summary>
        /// Matrix transpose p.
        /// </summary>
        /// <param name="p">matrix to transpose</param>
        /// <returns>Transposed 2-port matrix</returns>
        public static ParameterSet Transpose(ParameterSet p)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p.frequency;

            r.P11A = p.P11A;
            r.P12B = p.P21C;
            r.P21C = p.P12B;
            r.P22D = p.P22D;

            return r;
        }
        /// <summary>
        /// Unary negation of matrix p.
        /// </summary>
        /// <param name="p">matrix to negate</param>
        /// <returns>Negated 2-port matrix</returns>
        public static ParameterSet operator -(ParameterSet p)
        {
            ParameterSet r = new ParameterSet();

            r.frequency = p.frequency;

            r.P11A = -p.P11A;
            r.P12B = -p.P12B;
            r.P21C = -p.P21C;
            r.P22D = -p.P22D;

            return r;
        }

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
                return false;

            ParameterSet p = (ParameterSet)obj;
            return ((frequency == p.frequency)
                &&  (P11A == p.P11A)
                &&  (P12B == p.P12B)
                &&  (P21C == p.P21C)
                &&  (P22D == p.P22D));
        }

        public override int GetHashCode()
        {
            return ((int)frequency
                ^ P11A.GetHashCode()
                ^ P12B.GetHashCode()
                ^ P21C.GetHashCode()
                ^ P22D.GetHashCode());		// bitwise exclusive-or
        }

        /// <summary>
        /// Compare two ParameterSets for equality
        /// </summary>
        /// <param name="p1"></param>
        /// <param name="p2"></param>
        /// <returns>True if the two ParameterSets are equal, or refer to the same object.</returns>
        public static Boolean operator ==(ParameterSet p1, ParameterSet p2)
        {
            return object.Equals(p1, p2);
        }

        /// <summary>
        /// Compare two ParameterSets for inequality
        /// </summary>
        /// <param name="p1"></param>
        /// <param name="p2"></param>
        /// <returns>True is the two ParameterSets are not equal, or refer to the different objects.</returns>
        public static Boolean operator !=(ParameterSet p1, ParameterSet p2)
        {
            return !object.Equals(p1, p2);
        }
    }


}
