﻿
//
//    Copyright © 2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;
using System.Windows.Forms;
using System.ComponentModel;


namespace ABCDmatrix
{
    /// <summary>
    /// Provides constants for various coaxial and ladderline cables by cable model number
    /// </summary>
    [Serializable]
    public class CableSet
    {

        // k1 and k2 values from VK1OD transmission line calculator (March/2008)
        // Value of Z is a guess until better data is available; RG58 Z based on
        // personal measurements.

        /// <summary>
        /// Loss vs. Square-root-frequency coefficient
        /// </summary>
        [Description("Cable loss coefficient vs. sqrt-frequency."), Category("Cable Parameter")]
        public Double K1 { get; set; }
        /// <summary>
        /// Loss vs. Frequency Coefficient
        /// </summary>
        [Description("Cable loss coefficient vs. frequency."), Category("Cable Parameter")]
        public Double K2 { get; set; }

        /// <summary>
        /// Characteristic (surge) impedance of cable
        /// </summary>
        public Complex Zchar { get; set; }

        /// <summary>
        /// Real part of cable characteristic impedance
        /// </summary>
        [Description("Cable impedance, real part."), Category("Cable Parameter")]
        public Double Zreal
        {
            get
            {
                return Zchar.real;
            }
            set
            {
                Zchar.real = value;
            }
        }

        /// <summary>
        /// Imaginary part of cable characteristic impedance
        /// </summary>
        [Description("Cable impedance, imag part."), Category("Cable Parameter")]
        public Double Zimag
        {
            get
            {
                return Zchar.imag;
            }
            set
            {
                Zchar.imag = value;
            }
        }

        /// <summary>
        /// Cable velocity factor
        /// </summary>
        [Description("Cable velocity factor."), Category("Cable Parameter")]
        public Double Vf { get; set; }

        /// <summary>
        /// Cable type
        /// </summary>
        [Description("Cable type."), Category("Cable Parameter")]
        public CableType Ctype { get; set; }

        /// <summary>
        /// Default constructor
        /// </summary>
        public CableSet()
        {
            Ctype = CableType.User;
            Zchar = new Complex();
            Vf = 0;
            K1 = 0;
            K2 = 0;
        }
        /// <summary>
        /// Constructor
        /// </summary>
        public CableSet(CableType cable)
        {
            switch (cable)
            {
                case CableType.User:
                    K1 = 0;
                    K2 = 0;
                    Zchar = new Complex();
                    break;

                case CableType.RG8:
                    K1 = 5.966e-6;
                    K2 = 1.015e-10;
                    Vf = 0.66;
                    Zchar = new Complex(52, -0.6);
                    break;

                case CableType.RG58:
                    K1 = 1.392e-5;
                    K2 = 1.186e-10;
                    Vf = 0.66;
                    Zchar = new Complex(53.5, -0.83);
                    break;

                case CableType.RG6:
                    K1 = 4.658e-6;
                    K2 = 1.109e-10;
                    Vf = 0.82;
                    Zchar = new Complex(75, -0.6);
                    break;

                case CableType.RG213:
                    K1 = 5.966e-6;
                    K2 = 1.015e-10;
                    Vf = 0.66;
                    Zchar = new Complex(50, -0.32);
                    break;

                case CableType.LadderLine600:
                    K1 = 0.00000055;
                    K2 = 0;
                    Vf = 0.95;
                    Zchar = new Complex(600, 0);
                    break;

                case CableType.WindowLine450:
                    K1 = 1.358e-6;
                    K2 = 5.577e-11;
                    Vf = 0.92;
                    Zchar = new Complex(360, -0.69);
                    break; 

                default:
                    K1 = 0;
                    K2 = 0;
                    Zchar = new Complex();
                    break;
            }
        }

        /// <summary>
        /// Loss of cable vs. Frequency and Length
        /// </summary>
        /// <param name="Frequency">Frequency in Hz.</param>
        /// <param name="Length">Length in meters</param>
        /// <returns>Loss in dB.</returns>
        public static Double LossDB(Double Frequency, Double Length, Double k1, Double k2)
        {
            return (Length * (k1 * Math.Sqrt(Frequency) + k2 * Frequency));
        }
    }

    /// <summary>
    /// Cables that can be selected
    /// </summary>
    public enum CableType
    {
        /// <summary>
        /// User specified cable parameters
        /// </summary>
        User = 0,
        /// 
        /// <summary>
        /// typical RG58 Coaxial cable
        /// </summary>
        RG58,
        /// <summary>
        /// typical RG8 coaxial cable
        /// </summary>
        RG8,
        /// <summary>
        /// typical RG6 coaxial cable
        /// </summary>
        RG6,
        /// <summary>
        /// typical RG213 coaxial cable
        /// </summary>
        RG213,
        /// <summary>
        /// typical 600 ohm ladder line
        /// </summary>
        LadderLine600,
        /// <summary>
        /// typical 450 ohm window line
        /// </summary>
        WindowLine450
    }

}


