//////////////////////////////////////////////////////////////////////////////////
// Engineer:       James C. Ahlstrom
// 
// Create Date:    7 December 2009
// Design Name:    Digital receiver and transmitter
// Module Name:    Transmitter
// Project Name:   Transceiver
// Target Devices: Cyclone 3
// Tool versions: 
// Description: 
//
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
// Additional Comments: 
//
//////////////////////////////////////////////////////////////////////////////////

module Transmitter(
	input clock,
	input key_down,
	input eth_nRESET,
	input [31:0] tx_tune_phase,
	input [7:0] tx_ctrl,
	input signed [31:0] tx_sample,		// next sample to transmit
	output reg get_next_dac,			// toggle to get next sample
	output reg is_tx_bar,				// are we transmitting? (negative logic)
	output signed [13:0] dac_data
	);

	// Interpolate I/Q samples in memory from 48 kHz to the clock frequency
	wire req1, req2, req3;
	wire [19:0] y1_r, y1_i, y2_r, y2_i, y3_r, y3_i;
	CicInterpM5 #(.RRRR(20), .IBITS(16), .OBITS(20), .GBITS(18)) in1 (
		clock, req2, req1, tx_sample[15:0], tx_sample[31:16], y1_r, y1_i);
	CicInterpM5 #(.RRRR(16), .IBITS(20), .OBITS(20), .GBITS(16)) in2 (
		clock, req3, req2, y1_r, y1_i, y2_r, y2_i);
	CicInterpM5 #(.RRRR(8),  .IBITS(20), .OBITS(20), .GBITS(12)) in3 (
		clock, 1'd1, req3, y2_r, y2_i, y3_r, y3_i);

	// Tune transmitter with CORDIC
	wire [19:0] tx_real, tx_imag;	// input to cordic
	cordic1 tx_cordic (dac_data, clock, tx_tune_phase, tx_real, tx_imag);
	assign tx_real = is_cw ? cw_level[20:1] : y3_r;
	assign tx_imag = is_cw ? 20'd0          : y3_i;
	
	// Transmit CW.  Count cw_level up to 620000 takes 5.05 milliseconds
	parameter MAX_CW_LEVEL = 21'd620000;	// Maximum to CORDIC is 310000
	reg [20:0] cw_level;		// CW level to transmit
	reg is_cw;					// Are we in CW mode?

	reg [3:0] tx_state;			// state machine for transmit
	parameter sRx		= 0;	// Starting state is receive
	parameter sCwInc	= 1;
	parameter sCwDec	= 2;
	parameter sSsbTx	= 3;

	always @(posedge clock)		// Tx state machine
	begin
		if (eth_nRESET == 0)	// we are in reset
		begin
			tx_state = sRx;
		end
		else case (tx_state)
		sRx:	// Starting state is receive; key is up
		begin
			is_tx_bar <= 1'd1;
			cw_level <= 1'd0;
			is_cw <= 1'd1;
			if (key_down)
			begin
				case (tx_ctrl)
				1:		// Transmit CW
					tx_state <= sCwInc;
				2:		// Transmit samples in memory
					tx_state <= sSsbTx;
				endcase
			end
		end
		sCwInc:		// gradually increase carrier amplitude
		begin
			is_tx_bar <= 1'd0;
			if ( ! key_down)
				tx_state <= sCwDec;
			else if (cw_level != MAX_CW_LEVEL)
				cw_level <= cw_level + 1'd1;
		end
		sCwDec:		// gradually decrease carrier amplitude
		begin
			if (key_down)
				tx_state <= sCwInc;
			else if (cw_level == 0)
				tx_state <= sRx;
			else
				cw_level <= cw_level - 1'd1;
		end
		sSsbTx:
		begin
			is_cw <= 1'd0;
			if (key_down)
			begin
				if (req1)
				begin
					get_next_dac <= ~ get_next_dac;		// request next tx_sample
					is_tx_bar <= (tx_sample == 1'd0);
				end
			end
			else
				tx_state <= sRx;
		end
		endcase
	end

endmodule
