# Write the code for the fir filter


def main():
  fp = open("firR10.new", "w")
  N = 20

# line offset is 10 lines

  fp.write(
"""	`define PMBITS	18		// Multiplier bits
	`define PABITS  24		// Adder bits
	`define POBITS  24		// Output bits
	`define ADBITS	 5		// Address bits; must fit RRRR
	`define RRRR	20		// Our decimation

	// States for fir calculations
	`define sWait	5'b00000		// wait for x input
	`define sMult	5'b00001		// multiply x * coef
	`define sAccum	5'b00010		// accumulate product; move new x down the delay line
	`define sAddr	5'b00100		// increment address
	`define sOutput	5'b01000		// output the next y
	`define sClear	5'b10000		// clear accumulator

module firRR (		// FIR filter
	// The minimum input decimation is 7; one x_avail per 7 clocks.
	input clock,
	input x_avail,
	input signed [`PMBITS-1:0] x_real,		// x is the input
	input signed [`PMBITS-1:0] x_imag,
	output y_avail,
	output signed [`POBITS-1:0] y_real,		// y is the output
	output signed [`POBITS-1:0] y_imag);
	
	parameter MBITS	= `PMBITS;		// multiplier bits
	
	reg [`ADBITS-1:0] addr;	// memory address
	reg [4:0] macstate;		// state machine for multiply-accumulate
	reg mem_clock;			// fir memory clock
	
	initial
	begin
		macstate = `sWait;
		mem_clock = 0;
		addr = 1'd0;
	end
	
	always @(posedge clock)
	begin
		case (macstate)
			`sWait:
			begin		// wait for next x input
				if (x_avail)
					macstate <= `sMult;
			end
			`sMult:			// multiply happens here
			begin
				macstate <= `sAccum;
			end
			`sAccum:			// accumulate happens here
			begin
				macstate <= `sAddr;
				mem_clock <= 1'd1;	// move the new x down the delay line
			end
			`sAddr:			// increment the address
			begin
				mem_clock <= 1'd0;		// x must remain valid until here
				if (addr == `RRRR - 1'd1)
				begin
					macstate <= `sOutput;
					addr <= 0;
				end
				else
				begin
					macstate <= `sWait;
					addr <= addr + 1'd1;
				end
			end
			`sOutput:
			begin
				macstate <= `sClear;
			end
			`sClear:
			begin
				macstate <= `sWait;
			end
		endcase
	end

""")
  for i in range (0, N):
    fp.write('\twire signed [MBITS-1:0] coef%02d;  fir_rom #("c%02d.mif") c%02d (addr, mem_clock, coef%02d);\n' % (i, i, i, i))
  fp.write("""
	// Filter for real data
	firR10X fir_r (clock, mem_clock, macstate, addr, x_avail, x_real, y_avail, y_real,
	coef00, coef01, coef02, coef03, coef04, coef05, coef06, coef07, coef08, coef09, coef10,
	coef11, coef12, coef13, coef14, coef15, coef16, coef17, coef18, coef19
	);
	// Filter for imaginary data
	firR10X fir_i (clock, mem_clock, macstate, addr, x_avail, x_imag,        , y_imag,
	coef00, coef01, coef02, coef03, coef04, coef05, coef06, coef07, coef08, coef09, coef10,
	coef11, coef12, coef13, coef14, coef15, coef16, coef17, coef18, coef19
	);
endmodule
	
module firR10X(clock, mem_clock, macstate, addr, x_avail, x_in, y_avail, y,
	coef00, coef01, coef02, coef03, coef04, coef05, coef06, coef07, coef08, coef09, coef10,
	coef11, coef12, coef13, coef14, coef15, coef16, coef17, coef18, coef19
	);

	input clock;
	input mem_clock;			// clock for memory
	input [4:0] macstate;		// state machine for multiply-accumulate
	input [`ADBITS-1:0] addr;	// memory address
	input x_avail;
	input signed [IBITS-1:0] x_in;	// x is the input
	output reg y_avail;
	output signed [OBITS-1:0] y;
	input wire signed [MBITS-1:0]
	coef00, coef01, coef02, coef03, coef04, coef05, coef06, coef07, coef08, coef09, coef10,
	coef11, coef12, coef13, coef14, coef15, coef16, coef17, coef18, coef19;
	
	parameter IBITS	= `PMBITS;		// input size in bits
	parameter OBITS = `POBITS;		// output bits
	parameter MBITS	= `PMBITS;		// multiplier bits
	parameter ABITS = `PABITS;		// adder bits
	
	reg [3:0] outstate;			// state machine for output adder tree
	reg signed [ABITS-1:0] sum;	// filter output

	assign y = sum[ABITS-1 -: OBITS];

	initial
	begin
		outstate = 0;
	end

""")

  for i in range (0, N):
    fp.write("\twire signed [MBITS-1:0] data%02d; wire signed [ABITS-1:0] acc%02d; reg signed [ABITS-1:0] s%02d;\n" % (i, i, i))
  fp.write("\n\tfir_ram mem01 (x_in,   addr, 1'd1, mem_clock, data01);\n")
  for i in range (2, N):
    fp.write("\tfir_ram mem%02d (data%02d, addr, 1'd1, mem_clock, data%02d);\n" % (i, i-1, i))
  fp.write("\n\tmac mac00 (macstate, clock, x_in,   coef00, acc00);\n")
  for i in range (1, N):
    fp.write("\tmac mac%02d (macstate, clock, data%02d, coef%02d, acc%02d);\n" % (i, i, i, i))
  fp.write(

"""

	always @(posedge clock)
	begin
		case (outstate)
			0:
			begin
				if (macstate == `sOutput)
				begin
					outstate <= 1'd1;		// start the adder tree
""")
  for i in range (0, N):
    fp.write("\t\t\t\t\ts%02d <= acc%02d;\n" % (i,i))
  fp.write(
"""				end
			end
			1:
			begin
				outstate <= 4'd2;
""")
  for i in range (0, N / 2):
    fp.write("\t\t\t\ts%02d <= s%02d + s%02d;\n" % (i, i, N - 1 - i))
  fp.write (
"""			end
			2:
			begin
				outstate <= 4'd3;
				s00 <= s00 + s15;
				s01 <= s01 + s14;
				s02 <= s02 + s13;
				s03 <= s03 + s12;
				s04 <= s04 + s11;
				s05 <= s05 + s10;
				s06 <= s06 + s09;
				s07 <= s07 + s08;
			end	
			3:
			begin
				outstate <= 4'd4;
				s00 <= s00 + s07;
				s01 <= s01 + s06;
				s02 <= s02 + s05;
				s03 <= s03 + s04;
			end
			4:
			begin
				outstate <= 4'd5;
				s00 <= s00 + s01;
				sum <= s02 + s03;
			end
			5:
			begin
				outstate <= 4'd6;
				sum <= sum + s00;
				y_avail <= 1'd1;		// new output available
			end
			6:
			begin
				outstate <= 4'd0;
				y_avail <= 1'd0;
			end
		endcase
	end	
endmodule


// Quartus II Verilog Template
// Single port RAM with single read/write address 
module fir_ram 
(
	input [`PMBITS-1:0] data,
	input [`ADBITS-1:0] addr,
	input we, clk,
	output [`PMBITS-1:0] q
);

	// Declare the RAM variable
	reg [`PMBITS-1:0] ram [0:`ADBITS ** 2 - 1];

	// Variable to hold the registered read address
	reg [`ADBITS-1:0] addr_reg;

	always @ (posedge clk)
	begin
		// Write
		if (we)
			ram[addr] <= data;

		addr_reg <= addr;
	end

	// Continuous assignment implies read returns NEW data.
	// This is the natural behavior of the TriMatrix memory
	// blocks in Single Port mode.  
	assign q = ram[addr_reg];

endmodule

// Quartus II Verilog Template
// Single Port ROM

module fir_rom
(
	input [`ADBITS-1:0] addr,
	input clk, 
	output reg [`PMBITS-1:0] q
);
	parameter data = "xx.mif";
	
	(* ram_init_file = data *) reg [`PMBITS-1:0] rom [0:`ADBITS ** 2 - 1];
	
	always @ (posedge clk)
	begin
		q <= rom[addr];
	end
endmodule

module mac
(
	input [4:0] macstate,
	input clk,
	input signed [`PMBITS-1:0] dataa,
	input signed [`PMBITS-1:0] datab,
	output reg signed [`PABITS-1:0] adder_out);

	reg signed [`PMBITS*2-1:0] mult;

	always @ (posedge clk)
	begin
		case (macstate)
			`sMult:
				mult <= dataa * datab;
			`sAccum:
				adder_out <= adder_out + mult[`PMBITS*2-1 -: `PABITS];
			`sClear:
				adder_out <= 1'd0;
			default:
				;
		endcase
	end
endmodule
""")

main()

