//////////////////////////////////////////////////////////////////////////////////
// Engineer:       James C. Ahlstrom
// 
// Create Date:    25 June 2009
// Design Name:    Digital receiver and transmitter
// Module Name:    Transceiver
// Project Name:   Transceiver
// Target Devices: Cyclone 3
// Tool versions: 
// Description: 
//
// Dependencies: 
//
// Revision: 
// Revision 0.01 - File Created
// Additional Comments: 
//
//////////////////////////////////////////////////////////////////////////////////

module Transceiver(
	input clock,			// pin 150 from clock oscillator
	input key_down,			// pin 177
	output [7:0] LED,
	output led_red,			// Pin 4, negative logic
	output led_y1,			// Pin 46, negative logic
	output TestC,			// Pin 57, yellow led on pin, negative logic
	// ADC interface
	input adc_clock,		// Pin 91 from ADC
	input [13:0] adc_data,
	input adc_overrange,	// Pin 95
	output adc_reset,		// Pin 120
	// Ethernet controller interface bus
	output [6:0] eth_address_bus,
	inout  [15:0] eth_data_bus,
	output eth_nRD,
	output eth_nWR,
	output eth_nCS,
	output eth_FIFO_SEL,
	output eth_nRESET,
	input  eth_IRQ,
	// Output level control of DAC
	output [7:0] level_data,
	output level_nCS,
	output level_nWR,
	// DAC data; clock is "clock" on pin 150
	output signed [13:0] dac_data
	);
	
	// Use a phase locked loop PLL to generate the ADC clock based on adc_clock
	wire rx_clock;			// clock for ADC, etc.
	wire half_clock;		// half speed clock for ethernet
	wire pll_rx_locked;		// PLL lock indicator
	pll_rx prx (adc_clock, rx_clock, half_clock, pll_rx_locked);
	
	// Use a phase locked loop PLL to generate the DAC clock based on input clock
	wire tx_clock;			// clock for transmit
	wire slow_clock;		// slow clock for key delays, etc.
	wire pll_tx_locked;		// PLL lock indicator
	pll_tx ptx (clock, tx_clock, slow_clock, pll_tx_locked);
	
	// This module resets everything on power start.
	ethernet_reset ereset (slow_clock, eth_nRESET);
	
	// Generate transmit and receive keys with delays
	wire key_down_rx, key_down_tx;
	key_delays kd (slow_clock, eth_nRESET, key_down, key_down_rx, key_down_tx);

	assign led_y1 = (LED == 0);		// turn on for any LED bit true (errors)
	assign LED[7] = ~ (pll_rx_locked && pll_tx_locked);		// LED[7]: PLL not locked
	assign adc_reset = ~ eth_nRESET;	// the ADC requires a reset pulse 2 usec
	clip_led cl (slow_clock, adc_overrange, led_red);	// turn on LED for ADC clip
	
	// Memory for ADC data
	wire rx_mem_block;				// Index of available block 0 or 1
	wire [31:0] rx_mem_data;		// data to write
	wire [31:0] rx_mem_q;			// memory data at read address
	wire [31:0] rx_mem_checksum;	// must be unsigned
	wire [9:0]  rx_mem_read_addr;	// read address
	wire [9:0]  rx_mem_write_addr;	// write address
	wire rx_mem_wen;				// write enable
	eth_ram1 adc_ram (rx_mem_data, rx_mem_read_addr, half_clock,
		rx_mem_write_addr, rx_clock, rx_mem_wen, rx_mem_q);
	// read and register ADC data
	reg [13:0] reg_adc_data;
	always @(posedge rx_clock)
	begin
		reg_adc_data <= adc_data;
	end
	
	// Module to handle Ethernet bus transactions
	wire [31:0] rx_tune_phase;
	wire [7:0] rx_ctrl, tx_ctrl;
	wire [31:0] tx_tune_phase;
	ethernet mod_eth (half_clock, key_down_rx, LED[6:0], /*TestC*/ ,
		rx_ctrl, rx_tune_phase, tx_ctrl, tx_tune_phase,
		rx_mem_read_addr, rx_mem_q, rx_mem_checksum, rx_mem_block,
		tx_sample, get_next_dac,
		eth_address_bus, eth_data_bus, eth_nRD, eth_nWR,
		eth_nCS, eth_FIFO_SEL, eth_nRESET, eth_IRQ,
		level_data, level_nCS, level_nWR);

	// The Receiver
	Receiver rx (rx_clock, reg_adc_data, eth_nRESET, rx_tune_phase, rx_ctrl,
		rx_mem_data, rx_mem_write_addr, rx_mem_wen, rx_mem_checksum, rx_mem_block);

	// The Transmitter
	wire signed [31:0] tx_sample;	// next sample to transmit
	wire get_next_dac;				// toggle to get next sample
	Transmitter tx (tx_clock, key_down_tx, eth_nRESET, tx_tune_phase, tx_ctrl,
		tx_sample, get_next_dac,
		TestC, dac_data);
	
endmodule

module key_delays(		// generate two delayed keys from the input key
	input slow_clock,
	input eth_nRESET,
	input key_down,
	output reg key_down_rx,		// Key for Rx ADC samples: Delays are zero, key_delay_up
	output reg key_down_tx	// Key for FPGA Tx: Delays are key_delay_down, zero
	);
	
	// Key delays are in microseconds
	parameter key_delay_db		= 200;		// key debounce delay usec
	parameter key_delay_down	= 12000;	// Tx key delay on key down
	parameter key_delay_up		= 15000;	// Rx key delay on key up
	
	reg [23:0] key_timer;	// generate key delays
	reg [2:0] key_state;			// state machine for key
	parameter sKeyRx		= 0;
	parameter sKeyDbD		= 1;
	parameter sKeyDown		= 2;
	parameter sKeyTx		= 3;
	parameter sKeyDbU		= 4;
	parameter sKeyUp		= 5;
	
	`include "gparam.v"
	
	function [23:0] usec;	// return the number of clock ticks for input microseconds
		input integer usecs;
		integer x;
		x = (slow_clock_freq / 10000) * usecs / 100;
		usec = x[23:0];
	endfunction
	
	always @(posedge slow_clock)
	begin
		if (eth_nRESET == 0)	// we are in reset
		begin
			key_state <= sKeyRx;
		end
		else case (key_state)
		sKeyRx:		// Starting state is Rx; keys are up
		begin		// Debounce key down
			key_down_rx <= 1'd0;
			key_down_tx <= 1'd0;
			if ( ! key_down)
				key_timer <= usec (key_delay_db);
			else if (key_timer == 0)
				key_state <= sKeyDbD;
			else
				key_timer <= key_timer - 1'd1;
		end
		sKeyDbD:		// After debounce, key is down
		begin
			key_down_rx <= 1'd1;
			key_timer <= usec (key_delay_down);
			key_state <= sKeyDown;
		end
		sKeyDown:		// Key down delay
		begin
			if (key_timer == 0)
				key_state <= sKeyTx;
			else
				key_timer <= key_timer - 1'd1;
		end
		sKeyTx:			// We are transmitting
		begin			// Debounce key up
			key_down_tx <= 1'd1;
			if (key_down)
				key_timer <= usec (key_delay_db);
			else if (key_timer == 0)
				key_state <= sKeyDbU;
			else
				key_timer <= key_timer - 1'd1;
		end
		sKeyDbU:		// After debounce, key is up
		begin
			key_down_tx <= 1'd0;
			key_timer <= usec (key_delay_up);
			key_state <= sKeyUp;
		end
		sKeyUp:			// Key up delay
		begin
			if (key_timer == 0)
				key_state <= sKeyRx;
			else
				key_timer <= key_timer - 1'd1;
		end
		endcase
	end
endmodule

module clip_led (	// turn on LED for some milliseconds for clip
	input slow_clock,
	input adc_overrange,
	output led_red);
	
	reg [13:0] timer;
	assign led_red = (timer == 0);
	
	always @(posedge slow_clock)
	begin
		if (adc_overrange)
			timer <= 14'h3FFF;
		else if (timer != 0)
			timer <= timer - 1'd1;
	end
endmodule

module ethernet_reset(		// Reset the bus on power up
	input slow_clock,
	output eth_nRESET		// controller reset pin
	);
	
	reg [5:0] reset_timer;

	initial
	begin
		reset_timer = 6'h3F;	// time to release reset > 200 usec
	end
	
	assign eth_nRESET = (reset_timer == 0);
		
	always @(posedge slow_clock)
		if (reset_timer != 0)
			reset_timer <= reset_timer - 1'd1;
endmodule

