#! /usr/bin/octave -qf
#
#  TRANSMISSION LINE T-EQUIVALENT NETWORK CALCULATIONS
#
# Print header

printf("\n\n     *** TRANSMISSION LINE CALCULATIONS ***\n");

# Specify transmission line data

f = 7.01;   #   FREQUENCY IN MHz
d = 100;    #   LENGTH OF LINE IN FEET
a = 6.0;    #   ATTENUATION IN dB PER 100 FEET
v = 83;     #   VELOCITY FACTOR AS A PERCENTAGE
Zo = 75;    #   CHARACTERISTIC IMPEDANCE IN OHMS

# Specify source (transmitter) and load (antenna)

VS = 2.0;
RS = 50;
Rt = 50;
Xt = 0;

# Convert inputs as required

a = a ./ 1e2;  # convert dB per 100 feet to dB per foot
a = 0.1151 .* a;  # convert dB to nepers
c = 9.836e8;  # speed of light in feet per second
lambda = c ./ (1e6 .* f);  # wavelength of signal in vacuum
lambda = (v ./ 1e2) .* lambda; # adjust lambda for velocity
B = (2 .* pi) ./ lambda; # calculate Beta
ZL = Rt .+ j .* Xt;  # calculate complex terminating impedance

# Calculate elements of T-equivalent network

# ZA = input series element
# ZB = output series element
# ZC = shunt element

ZA = Zo .* tanh((a .+ j .*B) .* d ./ 2.);
ZB = ZA;
ZC = Zo ./ sinh((a .+ j .*B) .* d);

# Prepare matrices for solution of currents

right_side = [RS + ZA + ZC, -ZC; -ZC, ZB + ZC + ZL];
left_side = [VS; 0];

# Solve for currents

denom = det(right_side);
for m = 1: rows(right_side)
   numerator = right_side;
   for n = 1: rows(right_side)
      numerator(n, m) = left_side(n);
   endfor
   current(m) = det(numerator) / denom;
endfor

# Print out currents

for m = 1:rows(right_side)
   if imag(current(m)) < 0
      printf("\n\n  LOOP %d CURRENT = %8.5g - j%-8.5g\n", ...
      m, real(current(m)), -imag(current(m)));
   else
      printf("\n\n  LOOP %d CURRENT = %8.5g + j%-8.5g\n", ...
      m, real(current(m)), imag(current(m)));
   endif
printf("\n\n  LOOP %d CURRENT MAGNITUDE = %8.5g\n", ...
m, abs(current(m)));
endfor

# Calculate input impedance

Zd = ZA .+ (ZC .* (ZB + ZL)) ./ (ZC .+ ZB .+ ZL);

# Print input impdedance

for k = 1:columns(Zd)
   if imag(Zd(k)) < 0
      printf("\n\n  INPUT IMPEDANCE = %8.5g - j%-8.5g\n\n", ...
      real(Zd(k)), abs(imag(Zd(k))));
   else
      printf("\n\n  INPUT IMPEDANCE = %8.5g + j%-8.5g\n\n", ...
      real(Zd(k)), imag(Zd(k)));
   endif
endfor

# end program
