/*
  RTTY Reader by Glen Popiel - KW5GP

  Uses customized Sandbox Electronics SC16IS750 library
  Uses Adafruit ST7735 and GFX TFT display libraries
  Uses Wire library to communicate with I2C SC16IS750 UART module
  Uses SPI library to communicate with SPI TFT display

*/

//#define debug   // Enables diagnostic information

#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>  // SPI library required for SPI devices
#include <Wire.h> // Wire library required for I2C devices
#include <SC16IS750.h>  // SC16IS750 UART library

#define TFT_CS     10  // Assign the TFT CS to pin 10
#define TFT_RST    7  // Assign the TFT RST to pin 7
#define TFT_DC     8  // Assign the TFT DC to pin 8
#define tft_delay 10  // set the TFT command delay to 10ms

#define rtty_font_size 2  // The RTTY text display font size
#define status_font_size 1  // The font size for the status line

#define no_uart "UART:fail" // The UART connection failure message
#define uart_rdy "UART:OK"  // The UART ok message
#define baud_rate 45  // The UART baud rate

// Assign the translation values for the Baudot Letters character set
char baudot_ltrs[] = {'\0', 'E', '  ', 'A', ' ', 'S', 'I', 'U', ' ', 'D', 'R', 'J', 'N', 'F',
                      'C', 'K', 'T', 'Z', 'L', 'W', 'H', 'Y', 'P', 'Q', 'O', 'B', 'G', '\0',
                      'M', 'X', 'V', '\0'
                     };

// Assign the translation values for the Baudot Figures character set
char baudot_figs[] = {'\0', '3', '\0', '-', ' ', '\0', '8', '7', '\0', '\0', '4', '\0', ',', '!',
                      ':', '(', '5', '+', ')', '2', '$', '6', '0', '1', '9', '?', '&', '\0',
                      '.', '/', ';', '\0'
                     };

String status_msg;  // Variable for the current USB status message to display
String rtty_data = "";  // String variable to hold rtty text data
String rtty_char = "";  // Single character string variable to hold converted rtty char
bool system_status ;  // Flag indicating current system status - true = ok, false = bad

bool  figs = false; // The Figures/Letters toggle flag
bool unshift_on_space = true; // Enable shifting back to LTRS when a space character is received
int rx; // The received character
byte reg_data; // Variable for UART LCR Register data

SC16IS750 uart = SC16IS750(SC16IS750_PROTOCOL_I2C, SC16IS750_ADDRESS_BB); // Create the UART instance

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Create the TFT display instance

void setup()
{
#ifdef debug
  Serial.begin(9600);
#endif

  tft.initR(INITR_18BLACKTAB);   // initialize a 1.8" TFT with ST7735S chip, black tab
  delay(tft_delay);
  tft.fillScreen(ST7735_BLACK); // Clear the display - fill with BLACK background
  delay(tft_delay);
  tft.setRotation(1); // Set the screen rotation
  delay(tft_delay);
  tft.setTextWrap(true); // Turn off Text Wrap
  delay(tft_delay);
  tft.setTextSize(3); // Set the Font Size
  delay(tft_delay);
  tft.setTextColor(ST7735_GREEN); //Set the Text Color
  delay(tft_delay);
  tft.setCursor(40, 10);  //Set the Cursor and display the startup screen
  delay(tft_delay);
  tft.print("KW5GP");
  delay(tft_delay);
  tft.setTextSize(2);
  delay(tft_delay);
  tft.setCursor(15, 65);
  delay(tft_delay);
  tft.print("RTTY Reader");

  delay(5000);  //Wait 5 seconds
  clear_display(); // Clear the startup message

  uart.begin(baud_rate);  //Initialize SC16IS750 UART with defined baudrate setting
  if (uart.ping() != 1) // Check for a response form the UART
  {

#ifdef debug
    Serial.println("SC16IS750 UART not found");
#endif

    status_msg = no_uart; // set the status message to no UART found
    system_status = false;  // set the system status ok flag to false
    update_status();  // update the status line on the TFT display
    while (1);
  } else
  {
#ifdef debug
    Serial.println("SC16IS750 UART found");
#endif

    status_msg = uart_rdy; // set the status message to UART ok
    system_status = true;  // set the system status ok flag
    update_status();  // update the status line on the TFT display
  }

#ifdef debug
  Serial.println("Start Serial Communication");
  reg_data = uart.ReadRegister(SC16IS750_REG_LCR);  // Read the UART Line Control Register (LCR) register
  Serial.print("LCR Register = ");
  Serial.println(reg_data, BIN);
#endif

  uart.SetLine(5, 0, 0); // Set the UART Line Control Register (LCR) register fot 5 bit word, no parity, 1 stop bit

#ifdef debug
  reg_data = uart.ReadRegister(SC16IS750_REG_LCR);  // Read the UART Line Control Register (LCR) register to be sure it's set
  Serial.print("Modified LCR Register = ");
  Serial.println(reg_data, BIN);
#endif

}

void loop()
{
  get_rx_char();  // Get the received character and convert to ASCII
}


void clear_display()  // Function to clear the TFT display
{
  tft.fillScreen(ST7735_BLACK); // Clear the display - set to use a black background
  delay(tft_delay);
}

void update_status() // Prints status messages on bottom line of TFT display
{
  if (system_status)  // True if good, false if bad
  {
    //We're all good - update status in green
    tft.setTextColor(ST7735_GREEN);  // Green on black background color required to "clear" previous text
  } else
  {
    // There is a system error
    tft.setTextColor(ST7735_RED);  // Red on black background color required to "clear" previous text
  }
  tft.setTextSize(status_font_size);  // Set the font size to the status line font size
  tft.setCursor(5, 118); // Move to the bottom line of the display
  tft.print(status_msg);  // Display the status

  tft.setCursor(115, 118);  // Move to the end of the status line
  tft.print("Baud:");
  tft.print(baud_rate);  // Display the baud rate

  tft.setTextSize(rtty_font_size);  // Reset the text size for the rtty text area
  tft.setCursor(0, 5);  // Move to the top line of the display
}

void update_text() // Updates the RTTY text on the display
{
  tft.setTextSize(rtty_font_size);  // Set the rtty text font size
  delay(tft_delay);
  if (rtty_data.length() >= 78)  // Scroll bottom line up a line if at end of line 6
  {
    rtty_data = rtty_data.substring(13);  // Remove the first 13 characters (the top line)
    tft.setCursor(0, 5); // Move the cursor to the top line
    delay(tft_delay);
    tft.fillRect(0, 0, 159, 105, ST7735_BLACK); // Clear the rtty data area
    delay(tft_delay);
    tft.print(rtty_data); // Print the rtty data string variable
    tft.setCursor(0, 85);  // Set the cursor to the start of line 6
  }

  rtty_data = rtty_data + rtty_char;  // Add the character to the rtty data string
  tft.print(rtty_char); // display the received character

#ifdef debug
  if (rtty_data.length() != 0)
  {
    Serial.print("rtty_data string length: ");
    Serial.print(rtty_data.length());
  }
#endif

}

void get_rx_char()  // Get a received character from the UART
{
  if (uart.available() != 0)
  {
    // Read the received character
    rx = uart.read();
    if (rx != 0)
    {
      switch (rx)
      {
        case 4: // It's a space character. Unshift if flag is enabled
          if (unshift_on_space)
          {
            figs = false; // Set FIGS/LTRS flag to LTRS

#ifdef debug
            Serial.print("   Space - unshift");
#endif

          }
          break;

        case 27: // It's a FIGS key
          figs = true;  // Set the FIGS/LTRS flag to FIGS

#ifdef debug
          Serial.print("  FIGS");
#endif

          break;

        case 31:  // It's a LTRS key
          figs = false; // Set FIGS/LTRS flag to LTRS

#ifdef debug
          Serial.print("  LTRS");
#endif

          break;

      }

#ifdef debug
      Serial.print("   Baudot: ");
      Serial.print(rx);
#endif

      if (figs) // Choose which character decode array to use
      {
        rtty_char = baudot_figs[rx];  // Use the Figures array
      } else
      {
        rtty_char = baudot_ltrs[rx];  // Use the LTRS arry
      }

      // Decode it
#ifdef debug
      Serial.print("   Decode: ");
      Serial.println(rtty_char);
      //    delay(300);
#endif
      update_text();  // update the TFT display
    }
  }
}
