/*
Auto Power On/Off

Glen Popiel - KW5GP

Turns load power on after engine start and on delay
Turns load power off after engine stop and off delay
Bypass switch to turn on immediately and bypass start/stop delay

Released 2016 under GNU GPLv3

Uses Timer.h Library by Simon Monk http://www.simonmonk.org

*/

#define debug_mode 1  // Enables Debug Mode - Sends Debug output to Serial Monitor

#include "Timer.h"  // Timer Library so we can do timed interrupts

#define voltage_pin A0  // The input voltage analog pin
#define RELAY 3  // The Relay drive pin
#define LED_Red 4 // The RGB LED Red pin
#define LED_Green 5 // The RGB LED Green pin
#define LED_Blue 6  // The RGB LED Blue pin
#define Bypass_Switch 7 // The Bypass switch input pin

float voltage;  
float volt_drop = 1.05;  // The voltage drop across the 1N4001 diode and path to the analog input
float volt_threshold = 13.4;  // The turn-on threshold voltage
int volt_cal = 2450;  // Voltage calibration value for the MAP statement

#define on_delay 30000  // On delay in milliseconds - Turn on after 30 seconds
#define off_delay 600000 // Off delay in milliseconds - Turn off after 10 minutes

boolean current_state = false;  // The current power status
boolean bypass_Enabled = false; // Bypass active value
boolean timing = false;  // Variable to let us know that timing has started
boolean blinking = false; // Lets us know an LED is blinking so we don't try to start it blinking a second time
boolean status_change = false;  // Powering On/Off delay in progress flag

long seconds = 0;  // The number of seconds since the timer started

int tickEvent;  // The process number for the one second Timer tick
int On_ledEvent;  // The process number for the On LED Timer blink event
int Off_ledEvent; // The process number for the Off LED Timer blink event
int On_afterEvent;  // The process number for the On Delay Timer event
int Off_afterEvent; // The process number for the Off Delay Timer event
int Bypass_ledEvent;  // The process number for the Bypass LED Timer blink event

Timer t;  // Create Timer object as t

void setup() 
{
  // Initialize Serial for debug
  if (debug_mode)
  {
    Serial.begin(9600);
  }
  // initialize the I/O pins
  pinMode(RELAY, OUTPUT);
  pinMode(LED_Red, OUTPUT);
  pinMode(LED_Green, OUTPUT);
  pinMode(LED_Blue, OUTPUT); 
  pinMode(Bypass_Switch, INPUT_PULLUP);  // Enable the internal pullup resistor for the Bypass switch pin

  digitalWrite(RELAY, LOW);  // Turn off the power
  digitalWrite(LED_Red, HIGH);  // Turn on the Red status LED
  digitalWrite(LED_Green, LOW);
  digitalWrite(LED_Blue, LOW);
  
  if (debug_mode) // Start a one second Timer tick for debug
  {
    tickEvent = t.every(1000, Tick); 
    Serial.print("1 second tick started Process id= ");
    Serial.println(tickEvent);
  }
}

void loop() 
{

  // Check for Bypass, overrides all
  if (digitalRead(Bypass_Switch) == LOW) // If Low, Bypass switch is on
  {
    digitalWrite(RELAY, HIGH); // Turn on the power
    if (!bypass_Enabled) // If we weren't already in bypass, start blinking the Green LED
    {
      if (!blinking)  // Blink the Green LED if we're not already blinking it
      {
        Bypass_ledEvent = t.oscillate(LED_Green, 1000, LOW); // Blink the Green LED to indicate Bypass active and power on
        digitalWrite(LED_Red, LOW); // Turn off the Red LED
        if (debug_mode)
        {
          Serial.print("Bypass LED blink Process id: ");
          Serial.print(Bypass_ledEvent);
          Serial.println(" Bypass Enabled - power on");
        }
        bypass_Enabled = true;  // Set the Bypass on flag
        blinking = true;  // Set the LED blinking flag so we don't start another blink Timer process
      }
    }
  }
  else 
  {
    if (bypass_Enabled) // Bypass switch was on but now is off
    {
      if (blinking) // Stop the LED blink Timer process if it's blinking
      {
        t.stop(Bypass_ledEvent);   // Stop blinking the Green LED
        if (debug_mode)
        {
          Serial.print("Bypass LED Blink Timer event stopped, Process id: ");
        }
        blinking = false; // Turn off the LED blinking flag
      }
      bypass_Enabled = false;  // Turn off the Bypass active flag
            
      // Check the current status - set to proper on/off state
      readVoltage();

      if (voltage >= volt_threshold)  // If voltage is above the turn-on threshold we need to leave power on - set Green LED on - no delay
      {
        digitalWrite(LED_Green, HIGH);
        if (debug_mode)
        {
          Serial.println("   Bypass Off - Power remains on"); 
        }
        current_state = true;   
      }
      else
      {
        digitalWrite(RELAY, LOW);  // Turn off the power - no delay
        if (debug_mode)
        {
          Serial.print(Bypass_ledEvent);        
          Serial.println("   Bypass Off - power off");
        }
        digitalWrite(LED_Green, LOW); // Turn off the Green LED
        digitalWrite(LED_Red, HIGH);  // Turn on the Red LED
        current_state = false;  // Set the current power status to Off
      }
    }  
    
    else // Bypass was not enabled = handle normally

    {
      readVoltage();  // Read the input voltage
      if (!current_state && (voltage >= volt_threshold) && (!status_change)) // If we're not powered on, check to see if we need to be
      {
        // Initiate turn on delay
        if (!blinking)  // Start the Blue Led blink if we're not already blinking
        {
          On_ledEvent = t.oscillate(LED_Blue, 1000, LOW);  
          if (debug_mode)
          {
            Serial.print("Blinking Blue LED Timer Process id: ");
            Serial.print(On_ledEvent);
          }
          blinking = true;  // Turn on the LED blinking flag
          status_change = true; // Turn on the flag indicating we're in a power on/off delay
        }
        if (debug_mode)
        {
          Serial.println(" Was off - powering on");
        }
        digitalWrite(LED_Green, HIGH);  // Blinks Blue and Green LED to indicate power on delay 
        digitalWrite(LED_Red, LOW);  
        On_afterEvent = t.after(on_delay,PowerOn);  // Turn power on after start delay
        current_state = true; // Set the current power state flag to on
        blinking = true;  // Set the LED blink flag to on
      }
     
      if (current_state && (voltage < volt_threshold) && (!status_change))  // Turn off if below voltage threshold and we're powered on
      {
        // initiate turn off delay
        if (!blinking)  // Start blinking the Blue LED if it's not already blinking
        {
          Off_ledEvent = t.oscillate(LED_Blue, 1000, LOW);
          if (debug_mode)
          {
            Serial.print("Blink Blue LED Process id: ");
            Serial.print(Off_ledEvent);
          }
          blinking = true;  // set the LED blinking flag
          status_change = true; // Turn on the flag indicating we're in a power on/off delay
        }
        if (debug_mode)
        {
          Serial.println("   Was on - powering off");        
        }
        digitalWrite(LED_Green, LOW);
        digitalWrite(LED_Red, HIGH);
        Off_afterEvent = t.after(off_delay, PowerOff);  // Turn off after the turn-off delay
        current_state = false;  // set the current power status flag to off
      }    
    }    
  }

  t.update();  // Update the Interrupt handler - required for the Timer library to function 
 
}

void readVoltage()  // Function to read the input voltage
{
  float count = analogRead(voltage_pin);
  voltage = map(count,0,1023,0,volt_cal); // Map the A/D value to a voltage in millivolts
  voltage = (voltage/100)+ volt_drop;// Convert to volts and subtract the input circuit voltage loss
}

void PowerOn()  // Function to perform Power On operation after start delay
{
  // check to make sure we're still ready to power on
  if (blinking) // Turn off the blinking LED if it's blinking
  {
    t.stop(On_ledEvent);   // Stop blinking the Blue LED
    blinking = false; // Turn off the LED blinking flag
  } 
  digitalWrite(LED_Blue, LOW);  // Make sure the Blue LED is off    
  readVoltage();
  if (voltage >= volt_threshold)  // We're still ok to power on
  {
    // Turn it on
    digitalWrite(RELAY, HIGH); // Turn on the power
    if (debug_mode)
    {
      Serial.println("Start delay expired - Powering On");
    }    
    current_state = true; // Set the current power status to On
  }
  else
  {
    // Not ok to power on
    if (debug_mode)
    {
      Serial.println("Start delay expired - Not ok to power on");
    }    
    digitalWrite(LED_Green, LOW); // Turn off the Green LED
    digitalWrite(LED_Red, HIGH);  // Turn on the Red LED
    current_state = false;
  }
  status_change = false;  // Turn off the Power On/Off delay status flag  
}

void PowerOff()
{
  if (blinking) // Stop the LED blink if it's blinking
  {
    t.stop(Off_ledEvent);   // Stop blinking the Blue LED 
    digitalWrite(LED_Blue, LOW);  // Make sure the Blue LED is off       
  }  
  // check to make sure we're still ready to power off
  readVoltage();
  if (voltage < volt_threshold)  // We're still ok to power off
  {
    // Turn it off
    digitalWrite(RELAY, LOW); // Turn off the power
    if (debug_mode)
    {
      Serial.println("Turning off, stopping LED Blink");
    }
    current_state = false;  // Set the current power status to Off
  }
  else
  {
    // Not ok to power off - leave power on
    if (debug_mode)
    {
      Serial.println("Leaving on, stopping LED Blink");
    }
    digitalWrite(LED_Red, LOW); // Turn off the Red LED
    digitalWrite(LED_Green, HIGH);  // Turn on the Green LED    
    current_state = true;
  }
  blinking = false;  
  status_change = false;  // Turn off the Power On/Off delay status flag     
}

void Tick() // Provides a one second Timer tick
{
  if (debug_mode)
  {
    seconds = seconds +1 ;  // increment the seconds counter
    readVoltage();  // Read the input voltage
    Serial.print ("Seconds: "); // Print the Seconds and voltage
    Serial.print (seconds);
    Serial.print ("   Volts: ");
    Serial.println(voltage);
  }
}
