/*
  Antenna SWR Analyzer
  
  Glen Popiel - KW5GP

  Uses Built-In SPI library
  Uses Built-In Wire I2C Communication Library
  Uses Adafruit_ADS1015 ADS1015 16 bit A/D Library
  Uses Adafruit_GFX Core graphics library
  Uses Adafruit_ILI9340 Hardware-specific 2.2" TFT Library
  Uses Rotary Encoder Library
  Uses M0XPD DDS Library
*/

#define debug_mode 1  // Set to 1 to enable debug output

#include <SPI.h>  // Include the Build-in SPI library
#include <Wire.h>  // Include the I2C Communication Library
#include <Adafruit_ADS1015.h> // Include the ADS1015 16 bit A/D Library
#include <Adafruit_GFX.h>    // Include the Core graphics library
#include <Adafruit_ILI9340.h> // Include the Hardware-specific 2.2" TFT Library
#include <RotaryEncoder.h>  // Include the Rotary Encoder Library
#include <DDS.h>  // Include the M0XPD DDS Library

// Define the AD9850 Module pin numbers
#define W_CLK  7
#define FQ_UD  6
#define DATA  5
#define RESET  4

// Define the TFT Hardware SPI pins for the UNO/Nano
#define _sclk 13
#define _miso 12
#define _mosi 11
#define _cs 10
#define _dc 9
#define _rst 8

#define select_array_size 8  // Define the size of the Band Selection array
#define swr_array_size 100  // Define the size of the SWR Data array

//Define the Rotary Encoder pins
const int encoder_PinA = 14;  // Pin A of the Rotary Encoder (A0 on Nano)
const int encoder_PinB = 15;  // Pin B of the Rotary Encoder (A1 on Nano)
const int encoder_Switch_Pin = 2;  // Encoder Pushbutton Switch input pin (Pin 2)

int encoder_Pos = 0; // The current encoder position
int old_encoder_Pos = 0; // The previous encoder position
boolean pushbutton; // Flag to indicate encoder pushbutton was pressed
boolean encoder_change; // Flag to indicate the rotary encoder position has changed
const String band[select_array_size] = { "160m", "80m", "60m", "40m", "30m", "20m", "17m", "15m" }; // The Band Selection Menu array
const float start_freq[select_array_size] = { 1.8, 3.525, 5.3305, 7, 10.1, 14, 18.068, 21 }; // The Band Starting point data array
const float end_freq[select_array_size] = { 2, 4, 5.4035, 7.3, 10.15, 14.35, 18.168, 21.45 }; // The Band Ending point data array
byte SWR_array[swr_array_size];  // Initialize the SWR data array
float freq_step;  // Variable to hold the calculated frequency step size
int freq_select = 0;  // Variable to contain the band menu selection array index pointer
int index;  // Variable to hold the current SWR array index pointer
int last_y; // Variable to contain the previous graph Y value for line starting point
int current_y;  // variable to contain the graph Y value for line ending point
int last_x; // Variable to contain the previous graph X value for line starting point
int current_x;  // variable to contain the graph X value for line ending point
double dds_freq = start_freq[freq_select];  // Variable to contain the DDS frequency
double sweep_start, sweep_stop; // Variables to contain the frequency sweep starting and ending point
float SWR_fwd, SWR_rev;  // Variables to contain the values read from the A/D
const int delay_time = 40;  // The A/D settling delay
float SWR; // Variable to hold SWR calculations

Adafruit_ILI9340 tft = Adafruit_ILI9340(_cs, _dc, _rst);  // Instantiate the TFT display object

DDS dds(W_CLK, FQ_UD, DATA, RESET); // Instantiate the DDS object

RotaryEncoder Enc(encoder_PinA, encoder_PinB);  // Instantiate the Rotary Encoder object

Adafruit_ADS1115 ads; // Instantiate the 16-bit A/D object

void setup()
{
  if (debug_mode)
  {
    // initialize serial:
    Serial.begin(9600);
  }

  // Set up the Rotary Encoder and enable the Internal Pull-up resistor on the Encoder Inputs
  pinMode (encoder_PinA, INPUT_PULLUP);
  pinMode (encoder_PinB, INPUT_PULLUP);
  pinMode (encoder_Switch_Pin, INPUT_PULLUP);

  attachInterrupt(digitalPinToInterrupt(2), read_pushbutton, LOW);  // Set up Interrupt for the Rotary Encoder pushbutton switch

  tft.begin();  // Start the TFT display
  tft.setRotation(1); // Rotate the TFT display 90 degrees Clockwise
  tft.fillScreen(ILI9340_BLUE); // Clear the TFT display, fill with Blue background

  ads.setGain(GAIN_FOUR); // Set the A/D gain to 4 (1.024V Range  1 bit = 0.03125mV)

  if (debug_mode)
  {
    Serial.println("TFT Initialized");
  }

  dds.init();  // start the DDS
  dds.trim(125000322);   // (Optional) trim if your DDS osc freq is not at 125MHz

  ads.begin();  // Start the A/D converter

  // Display the startup screen
  tft.setCursor(120, 40);
  tft.setTextColor(ILI9340_WHITE);
  tft.setTextSize(3);
  tft.print("KW5GP");
  tft.setCursor(20, 100);
  tft.print("Antenna Analyzer");
  delay(5000);

}  // End Setup Loop

void loop()
{
  // Display the Band Selection Menu
  tft.fillScreen(ILI9340_BLUE);
  Serial.println("Sweep Ready for Band Select");
  tft.setCursor(60, 40);
  tft.setTextSize(3);
  tft.print("Select Band");
  tft.setTextSize(1);
  tft.setCursor(60, 230);
  tft.print("Press Encoder Pushbutton to Start");

  update_selection(); // Display the current Band Selection Data
  pushbutton = false; // Reset the Encoder Pushbutton Switch pressed flag
  index = 0;  // Set the SWR array index pointer to the first element of the SWR array

  // Read the Encoder and Update the Band Selection Data until the Encoder Pushbutton switch is pressed
  do
  {
    read_encoder(); // Read the rotary encoder
    if (encoder_change) // Check to see if the Encoder position has changed
    {
      update_selection(); // Update the Band Selection data and display if the Encoder position has changed
    }
  } while (!pushbutton);  // Repeat until Encoder Pushbutton Switch is pressed

  // The Band has been selected - start the frequency sweep

  freq_step = ((end_freq[freq_select] - start_freq[freq_select]) / swr_array_size) * 1000000;  // Calculate the frequency step value

  pushbutton = false; // Clear the Encoder Pushbutton Switch pressed flag

  dds_freq = start_freq[freq_select] * 1000000; // Set the DDS frequency variable to the starting frequency
  dds.setFrequency(dds_freq);  // Set the DDS to the starting freq
  sweep_start = start_freq[freq_select] * 1000000;  // Set the sweep starting point variable to the selected band start frequency
  sweep_stop = end_freq[freq_select] * 1000000;  // Set the sweep ending point variable to the selected band ending frequency

  // Set the A/D gain based on selected band
  if (dds_freq > 12000000)
  {
    ads.setGain(GAIN_SIXTEEN);  // Set the A/D gain to 16 (0.256V Range  1 bit = 0.0078125mv) for fequencies above 12 MHz
  }
  else if (dds_freq > 7000000)
  {
    ads.setGain(GAIN_EIGHT);  // Set the A/D gain to 8 (0.512V Range  1 bit = 0.015625mv) for fequencies between 7 MHz and 12 MHz
  } else
  {
    ads.setGain(GAIN_FOUR);  // Set the A/D gain to 4 (1.024V Range  1 bit = 0.03125mv) for fequencies below 7 MHz
  }

  // Clear the TFT and show that we are scanning the selected band
  tft.fillScreen(ILI9340_BLUE);
  tft.setCursor(100, 60);
  tft.setTextSize(3);
  tft.print("Scanning");
  update_common();  // Update the TFT display with the selected band data

  // Now scan the selected band

  do  // Scan from the starting frequency to the ending frequency
  {
    read_adc(); // Read the A/D data

    if (debug_mode) // Show the debug data for each scan frequency
    {
      Serial.print("PB Status: ");
      Serial.print (pushbutton);
      Serial.print("   Step: ");
      Serial.print(freq_step);
      Serial.print("   Freq: ");
      Serial.print(dds_freq ); // display the DDS Freq

      Serial.print("   Fwd: ");
      Serial.print(SWR_fwd);
      Serial.print("   Rev: ");
      Serial.print(SWR_rev);
    }

    // Calculate SWR
    if (SWR_fwd >= SWR_rev) // SWR calculations fail if the reverse SWR voltage is higher than the forward value
    {
      SWR = ((SWR_fwd + SWR_rev) / (SWR_fwd - SWR_rev)) * 10; // Multiply SWR by 10 so it will fit properly in a byte variable to save space
      if (SWR > 50)
      {
        SWR = 50; // Limit max SWR to 5 to 1
      }
    } else
    {
      SWR = 50;  // If the reverse voltage is higher than the forward voltage, force SWR to 5 to 1
    }

    if (index <= swr_array_size - 1)  // Make sure we're within the array limits before storing the SWR data
    {
      SWR_array[index] = int(SWR);  // Store the SWR value into the SWR data array
    }

    index = index + 1;  // Increment the SWR array pointer

    if (debug_mode)
    {
      Serial.print("   SWR: ");
      Serial.println(SWR / 10, 1);
    }

    dds_freq = dds_freq + freq_step;  // Increment the DDS frequency to the next data point
    dds.setFrequency(dds_freq);  // Set the DDS to the next data point

  }   while (dds_freq <= sweep_stop); // Repeat until we've scanned the entire band

  // Prepare to graph the scan data
  last_x = 200 - (SWR_array[0] * 4);  // We need to store the first datapoint so we can calculate the line starting point

  if (debug_mode)
  {
    Serial.print("SWR_array[");
    Serial.print(0);
    Serial.print("] = ");
    Serial.print(SWR_array[0]);
    Serial.println("End of Sweep Pass");
  }

  // Clear the TFT display and display the SWR graph template
  tft.fillScreen(ILI9340_BLUE);
  tft.drawFastVLine(150, 0, 200, ILI9340_BLACK);  // Draw a vertical line at the center frequency
  tft.drawFastHLine(0, 200, 300, ILI9340_BLACK);  // Draw a horizontal line at the SWR 1:1 point
  tft.drawFastHLine(0, 175, 300, ILI9340_BLACK);  // Draw a horizontal line at the SWR 1.5:1 point
  tft.drawFastHLine(0, 150, 300, ILI9340_BLACK);  // Draw a horizontal line at the SWR 2:1 point
  tft.drawFastHLine(0, 100, 300, ILI9340_BLACK);  // Draw a horizontal line at the SWR 3:1 point
  tft.drawFastHLine(0, 50, 300, ILI9340_BLACK);  // Draw a horizontal line at the SWR 4:1 point

  // Display the SWR value for the template lines
  tft.setTextSize(0);
  tft.setCursor(301, 0);
  tft.print(">5");
  tft.setCursor(301, 47);
  tft.print("4");
  tft.setCursor(301, 97);
  tft.print("3");
  tft.setCursor(301, 147);
  tft.print("2");
  tft.setCursor(301, 172);
  tft.print("1.5");
  tft.setCursor(301, 197);
  tft.print("1");
  tft.setCursor(0, 205);

  // Display the band starting and ending frequencies below the graph
  tft.print(start_freq[freq_select], 4);
  tft.setCursor(270, 205);
  tft.print(end_freq[freq_select], 4);

  // Now graph the SWR data

  for (int x = 1; x <= (swr_array_size - 1); x++)
  {
    if (debug_mode) // Print the contents of the SWR data array for debug
    {
      Serial.print("SWR_array[");
      Serial.print(x);
      Serial.print("] = ");
      Serial.print(SWR_array[x]);
      Serial.print("  Last Y= ");
      Serial.print(last_y);
      Serial.print("  Last x = ");
      Serial.print(last_x);
    }

    // Figure out the line endpoints - TFT screen is x= 0 to 300, y= 0 to 200
    current_x = 200 - (((SWR_array[x] - 10) * 5)   ); // Calculate the X value of the current line ending point
    last_x = 200 - (((SWR_array[x - 1] - 10 ) * 5));  // Calculate the X value of the current line starting point

    if (swr_array_size == 100)
    {
      last_y = (x * 3) - 3; // Calculate the Y value of the current line starting point (for 100 datapoints in the SWR array)
      current_y = x * 3;  // Calculate the X value of the current line ending point (for 100 dataponts in the SWR array)
    } else {
      last_y = (x  - 1); // Calculate the Y value of the current line starting point (for 100 datapoints in the SWR array)
      current_y = x;  // Calculate the X value of the current line ending point (for 100 dataponts in the SWR array)
    }

    // Draw the graph line for the current data points
    tft.drawLine(last_y, last_x, current_y, current_x, ILI9340_GREEN); // Note the X and Y values are transposed due to screen rotation

    if (debug_mode) // Print the Line drawing data for debug
    {
      Serial.print("  Current x= ");
      Serial.print(current_x);
      Serial.print("  Current y= ");
      Serial.print(current_y);
      Serial.print(" Draw Line from :");
      Serial.print(last_y);
      Serial.print(",");
      Serial.print(last_x);
      Serial.print(" To ");
      Serial.print(current_y);
      Serial.print(",");
      Serial.println(current_x);
    }
  }

  if (debug_mode)
  {
    Serial.println("Graph Complete");
  }

  // Display the message to continue
  tft.setCursor(50, 220);
  tft.print("Press Encoder Pushbutton to Continue");
  pushbutton = false; // Clear the Encoder Pushbutton pressed lag

  do  // Loop until the Encoder Pushbutton Switch is pressed
  {
    delay(1000);    // Don't do anything but we have to have something here for the loop to work properly
  } while (!pushbutton);  // Repeat until the Encoder Pushbutton is pressed

  if (debug_mode)
  {
    Serial.println("Restarting");
  }

  delay(1000);  // Delay to help debounce the Encoder Pushbutton Switch Interrupt
  pushbutton = false; // Clear the Encoder Pushbutton switch pressed flag


}  // End Main Loop

void read_adc() // Function to read the A/D converter
{
  float adc_1, adc_2, adc_3;
  if (debug_mode)
  {
    Serial.println("Read ADC Function  ");
  }
  // Read the Forward SWR voltage
  delay(delay_time); // adc settling delay
  adc_1 = ads.readADC_SingleEnded(0); // Read the Forward SWR voltage
  adc_2 = ads.readADC_SingleEnded(0); 
  adc_3 = ads.readADC_SingleEnded(0); 
  SWR_fwd = (adc_1 + adc_2 + adc_3) / 3;  // Average the 3 readings to get the Forward SWR value
  
  // Read the Reverse SWR voltage  
  delay(delay_time); // adc settling delay
  adc_1 = ads.readADC_SingleEnded(2);// Read the Reverse SWR voltage
  adc_2 = ads.readADC_SingleEnded(2);
  adc_3 = ads.readADC_SingleEnded(2); 
  SWR_rev = (adc_1 + adc_2 + adc_3) / 3;  // Average the 3 readings to get the Reverse SWR value
}

// Function to read the rotary encoder
void read_encoder()
{
  // Read the Encoder
  Enc.tick(); // Required for library to check the encoder position
  encoder_Pos = Enc.getPosition(); // get the current encoder position

  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update selection
  {
    encoder_change = true;

    if (encoder_Pos > old_encoder_Pos) // If we're increasing the encoder position
    {
      if (freq_select >= select_array_size - 1) // Check to see if we're trying to above the highest selection value
      {
        freq_select = select_array_size - 1; // Limit the encoder value to the highest selection value
      } else
      {
        freq_select = freq_select + 1;  // Increment the selection value
      }

    } else
    {
      if (freq_select <= 0) // If we're decreasing the encoder position
      {
        freq_select = 0; // Limit the encoder value to the lowest selection value
      } else
      {
        freq_select = freq_select - 1;  // Decrement the selection value
      }
    }

    old_encoder_Pos = encoder_Pos;  // Set the previous encoder position to the current position

    if (debug_mode)
    {
      Serial.print("Encoder change: ");
      Serial.println(freq_select);
    }
  }
}

// Interrupt Function to read the rotary encoder pushbutton switch
void read_pushbutton()
{
  if (!pushbutton) // No need to set flag again if pushbutton flag is set
  {
    if (debug_mode)
    {
      Serial.println("Pushbutton Interrupt");
    }
    pushbutton = true;  // Set the encoder pushbutton pressed flag
  }
}

// Function to update the Selection Display
void update_selection()
{
  tft.fillRect(0, 100, 319, 130, ILI9340_BLUE); // Erase the lower portion of the TFT display
  tft.setTextSize(3);
  tft.setCursor(120, 100);
  tft.print(band[freq_select]); // Update the Band Selected display
  update_common();  // Update the Frequency Selected display
}

// Function to update the common TFT display data
void update_common()
{
  tft.setTextSize(2);
  tft.setCursor(50, 150);
  tft.print(start_freq[freq_select], 3);  // Display the Starting frequency
  tft.print("MHz - ");
  tft.print(end_freq[freq_select], 3);  // Display the Ending frequency
  tft.print("MHz");
  encoder_change = false; // Turn off the Encoder position changed flag
}


