/*
   915 MHz Wireless remote data telemetry - Remote unit

   by Glen Popiel KW5GP

   uses Adafruit Feather 32U4 LoRa

   uses Adafruit DHT-sensor-library library , www.adafruit.com
   uses Radiohead library by Mike McCauley, mikem@airspayce.com

*/

#define debug_mode 0  // Set debug_mode to 1 for Serial Monitor

#include <Adafruit_Sensor.h>
#include <SPI.h>  // Build-in SPI library
#include <RH_RF95.h>  // Wirehead RF95 library
#include <DHT.h>  // Adafruit DHT-sensor-library

// DHT11 Temp/Humidity Sensor definitions
#define DHTPIN 6     // The digital pin we're connected to
#define DHTTYPE DHT11   // DHT 11

// Define the Feather 32U4 pins
#define RFM95_CS 8
#define RFM95_RST 4
#define RFM95_INT 7

// Define the I/O pins
#define IntrusionPin 2  // Define the Intrusion Detect Switch on Pin 2 - SDAS - Interrupt 1
#define BatteryPin A0 // Define the Voltage sense input pin

#define tx_delay 10000  // The delay time between transmit cycles in milliseconds

int volt_cal = 1578;  // The voltage mapping calibration value
float volt_drop = .89;  // The voltage drop across the protection diode and input circuit
float humidity; // Variable to hold the humidity value
float temperature;  // Variable to hold the temperature value
float volts;  // Variable to hold the voltage value
String hum_string = ""; // Variable to hold the humidity string extracted from the received message
String temp_string = "";  // Variable to hold the temperature string extracted from the received message
String voltage_string = "";  // Variable to hold the voltage string extracted from the received message
boolean Intrusion = false;  // Flag to indicate an intrusion has been detected
boolean Intrusion_cleared = true; // Flag to inidicate an intrusion has cleared
boolean Intrusion_sent = false; // Flag to indiate the intrusion alert has been sent
int volt_count; // Variable to hold the raw A/D count of the input voltage

char radiopacket[45];  // Variable to hold the data packet to send
uint8_t buf[RH_RF95_MAX_MESSAGE_LEN]; // Receive data buffer - set to RF95 Maximum Message Length
uint8_t len = sizeof(buf); // Variable to hold the size of the Rx buffer

// Create an instance of the DHT Temp/humidity sensor
DHT dht(DHTPIN, DHTTYPE);

// Define the radio frequency - must match RX freq!
#define RF95_FREQ 915.0

// Create an instance of the radio driver
RH_RF95 rf95(RFM95_CS, RFM95_INT);

void setup()
{
  pinMode(IntrusionPin, INPUT_PULLUP);  // Set the intrusion detection switch pin mode
  pinMode(RFM95_RST, OUTPUT); // Set the Pin Mode for the RF95's Reset pin
  digitalWrite(RFM95_RST, HIGH);  // Turn off RF95 Reset

  dht.begin();  // Start the DHT temperature/humidity sensor

  if (debug_mode)
  {
    while (!Serial);
    Serial.begin(9600);
    delay(100);
    Serial.println("Feather LoRa TX Test!");
  }

  // Manually reset the RF95 radio
  digitalWrite(RFM95_RST, LOW);
  delay(10);
  digitalWrite(RFM95_RST, HIGH);
  delay(10);

  // Initialize the RF95 radio
  // Defaults after init are 434.0MHz, modulation GFSK_Rb250Fd250, +13dBm, Bw = 125 kHz, Cr = 4/5, Sf = 128chips/symbol, CRC on
  while (!rf95.init())
  {
    if (debug_mode)
    {
      Serial.println("LoRa radio init failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.println("LoRa radio init OK!");
  }

  // Set the radio frequency
  if (!rf95.setFrequency(RF95_FREQ))
  {
    if (debug_mode)
    {
      Serial.println("setFrequency failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.print("Set Freq to: "); Serial.println(RF95_FREQ);
  }

  // Set the transmitter power output
  // The default transmitter power is 13dBm, using PA_BOOST.
  // If you are using RFM95/96/97/98 modules which uses the PA_BOOST transmitter pin, then
  // you can set transmitter powers from 5 to 23 dBm:
  rf95.setTxPower(23, false);

  attachInterrupt(digitalPinToInterrupt(IntrusionPin), read_switch, CHANGE);  // Set up Interrupt 1 for the Intrusion switch

}

void loop()
{
  read_DHT11(); // Read the Temperature and Relative Humidity

  read_battery(); // Read the input voltage

  for (int i = 0; i <= 40 ; i++)  // Fill the Outbound message string with zeroes
  {
    radiopacket[i] = '0';
  }

  // Prepare the data to send
  radiopacket[0] = 'R'; // Set the first character of the data packet to 'R'

  hum_string = String(humidity);  // Convert the humidity to a string
  temp_string = String(temperature);  // Convert the temperature to a string
  voltage_string = String(volts);  // Convert the voltage to a string

  if (debug_mode)
  {
    Serial.print("Humidity String: " + hum_string);
    Serial.print("  Length: ");
    Serial.println(hum_string.length());
    Serial.print("   Temp String: " + temp_string);
    Serial.print("  Length: ");
    Serial.println(temp_string.length());
  }

  while (hum_string.length() < 8) // Pad the humidity string with spaces to get the proper length
  {

    hum_string = " " + hum_string;
  }

  while (temp_string.length() < 8) // Pad the temperature string with spaces to get the proper length
  {

    temp_string = " " + temp_string;
  }


  while (voltage_string.length() < 8) // Pad the voltage string with spaces to get the proper length
  {

    voltage_string = " " + voltage_string;
  }

  if (debug_mode)
  {
    Serial.print(" Fixed Humidity String: " + hum_string);
    Serial.print("  Length: ");
    Serial.println(hum_string.length());
    Serial.print(" Fixed Temp String: " + temp_string);
    Serial.print("  Length: ");
    Serial.println(temp_string.length());
    Serial.print(" Fixed Voltage String: " + voltage_string);
    Serial.print("  Length: ");
    Serial.println(voltage_string.length());
  }

  build_packet(); // put the data into the data packet 


  send_data();  // Send the outbound message


  delay(tx_delay);  // Delay until next transmit cycle

}

// Function to read the DHT11 temp/humidity sensor
void read_DHT11()
{
  // Reading temperature or humidity takes about 250 milliseconds!
  // Sensor readings may also be up to 2 seconds 'old' (its a very slow sensor)
  humidity = dht.readHumidity();  // Read the humidity

  // Read temperature as Fahrenheit (isFahrenheit = true)
  temperature = dht.readTemperature(true);  // Read the temperature

  // Check if any reads failed and exit early (to try again).
  if (isnan(humidity) ||  isnan(temperature)) // if returned data is "NAN" it means we did not get valid data from the DHT11
  {
    if (debug_mode)
    {
      Serial.println("Failed to read from DHT sensor!");
    }
    return;
  }

  if (debug_mode)
  {
    Serial.print("Humidity: ");
    Serial.print(humidity);
    Serial.print(" %\t");
    Serial.print("Temperature: ");
    Serial.print(temperature);
    Serial.println(" *F\t");
  }
}

// Function to read the input voltage
void read_battery()
{
  // Read the Battery Voltage
  volt_count = analogRead(BatteryPin);  // Read the input voltage analog pin value
  delay(100);
  volt_count = analogRead(BatteryPin);  // Read the input voltage analog pin value again since the first read is usually incorrect

  if (debug_mode)
  {
    Serial.print("Battery Count: ");
    Serial.print(volt_count);
  }

  volts = map(volt_count, 0, 1023, 0, volt_cal);  // Map the A/D count to the calculated voltage * 10 (to let us get the decimal value)

  if (debug_mode)
  {
    Serial.print("   Mapped Count: ");
    Serial.print(volts);
  }

  volts = ((volts / 100) + volt_drop); // Divide the voltage by 10 to get the correct value and then subract the input voltage drop

  if (debug_mode)
  {
    Serial.print("    Voltage: ");
    Serial.println(volts);
  }
}


// Interrupt Function to read the Intrusion Detect switch
void read_switch()
{
  if (!digitalRead(IntrusionPin)) // Set the intrusion flag in the TX data if switch is closed (low)
  {
    if (debug_mode)
    {
      Serial.println("Intrusion Detected");
    }
    Intrusion = true;  // Set the Intrusion Detected flag
    Intrusion_cleared = false;  // Turn off the Intrusion Cleared flag
  } else {
    if (debug_mode)
    {
      Serial.println("Intrusion Cleared");
    }
    Intrusion_cleared = true; // Set the Intrusion Cleared flag
    
  }
}


// Function to send the data packet
void send_data()
{
  // Send a message to rf95_server

  if (debug_mode)
  {
    Serial.println("Sending to rf95_server");
    Serial.print("Sending "); Serial.println(radiopacket);
    Serial.println("Sending..."); delay(10);
  }

  rf95.send((uint8_t *)radiopacket, 45);  // Transmit the data packet

  if (debug_mode)
  {
    Serial.println("Waiting for packet to complete...");
  }

  delay(10);
  rf95.waitPacketSent();// Wait until the packet is sent

  if (Intrusion)  
  {
    Intrusion_sent = true;// Set the Intrusion alert sent flag
  }

  if (debug_mode)
  {
    Serial.println("Transmission Complete");
  }

  if (Intrusion_sent && Intrusion_cleared)  // If the
  {
    Intrusion = false;  // If the Intrusion alert has been sent and it is now cleared, turn off the Intrusion flag
  }
}

// Function to build the transmit data packet
void build_packet()
{
  for (int i = 0; i <= 7 ; i++) // Stuff the humidity string into the outbound message
  {
    radiopacket[1 + i] = hum_string[i];
  }

  for (int i = 0; i <= 7 ; i++) // Stuff the temperature string into the outbound message
  {
    radiopacket[9 + i] = temp_string[i];
  }

  for (int i = 0; i <= 7 ; i++) // Stuff the voltage string into the outbound message
  {
    radiopacket[17 + i] = voltage_string[i];
  }

  if (Intrusion)  // Stuff the intrusion detection value into the outbound message
  {
    radiopacket[25] = 'T';
  } else {
    radiopacket[25] = 'F';
  }
}

